use dmntk_feel::context::FeelContext;
use dmntk_feel::values::Value;
use dmntk_feel::Scope;
use dmntk_model::model::Definitions;

mod compliance;

/// Utility function that creates a `FEEL` context from specified input expression.
pub fn context(input: &str) -> FeelContext {
  let scope = Scope::default();
  match dmntk_feel_parser::parse_context(&scope, input, false) {
    Ok(node) => match dmntk_feel_evaluator::prepare(&node) {
      Ok(evaluator) => match evaluator(&scope) {
        Value::Context(ctx) => ctx,
        other => panic!("ERROR: expected context value, actual value is: {}", other as Value),
      },
      Err(reason) => panic!("ERROR: building evaluator failed with reason: {}", reason),
    },
    Err(reason) => panic!("ERROR: parsing context failed with reason: {}", reason),
  }
}

/// Utility function that evaluates a `Decision` specified by name and compares the result.
fn assert_decision(definitions: &Definitions, name: &str, input_data: &FeelContext, expected: &str) {
  let actual = crate::evaluate_decision_by_name(definitions, name, input_data).unwrap().to_string();
  assert_eq!(
    expected, actual,
    "Assertion error, actual value of the decision does not match the expected value:\n  expected: {}\n    actual: {}\n",
    expected, actual
  );
}

/// Utility function that evaluates a `BusinessKnowledgeModel` specified by name and compares the result.
fn assert_business_knowledge_model(definitions: &Definitions, name: &str, input_data: &FeelContext, expected: &str) {
  let actual = crate::evaluate_business_knowledge_model_by_name(definitions, name, input_data)
    .unwrap()
    .to_string();
  assert_eq!(
    expected, actual,
    "Assertion error, actual value of the business knowledge model does not match the expected value:\n  expected: {}\n    actual: {}\n",
    expected, actual
  );
}

/// Utility function that evaluates a `DecisionService` specified by name and compares the result with expected value.
fn assert_decision_service(definitions: &Definitions, name: &str, input: &str, expected: &str) {
  let actual = crate::eval_decision_service_by_name(definitions, name, &context(input)).unwrap().to_string();
  assert_eq!(
    expected, actual,
    "Assertion error, actual value of the decision service does not match the expected value:\n  expected: {}\n    actual: {}\n",
    expected, actual
  );
}
