/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model evaluator
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use dmntk_common::Stringify;
use dmntk_feel::context::FeelContext;
use dmntk_feel::values::Value;
use dmntk_feel::Scope;
use dmntk_model::model::Definitions;
use dmntk_model_evaluator::{eval_decision_service_by_name, evaluate_decision_by_name};

#[test]
fn _2_0001() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0001, "file:///2_0001.dmn").unwrap();
  let ctx = context(r#"{Full Name:"John Doe"}"#);
  assert_decision(r#""Hello John Doe""#, &definitions, "Greeting Message", &ctx);
}

#[test]
fn _2_0002() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0002, "file:///2_0002.dmn").unwrap();
  let ctx = context(r#"{Monthly Salary:8375.00}"#);
  assert_decision(r#"100500.00"#, &definitions, "Yearly Salary", &ctx);
}

#[test]
fn _2_0003() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0003, "file:///2_0003.dmn").unwrap();
  let ctx = context(r#"{Employment Status:"EMPLOYED"}"#);
  assert_decision(r#""You are EMPLOYED""#, &definitions, "Employment Status Statement", &ctx);
}

#[test]
#[ignore]
fn _2_0003_a() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0003, "file:///2_0003.dmn").unwrap();
  let ctx = context(r#"{Employment Status:"RETIRED"}"#);
  assert_decision(r#"null"#, &definitions, "Employment Status Statement", &ctx);
}

#[test]
fn _2_0004() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0004, "file:///2_0004.dmn").unwrap();
  let ctx = context(r#"{Age:18,RiskCategory:"Medium",isAffordable:true}"#);
  assert_decision(r#""Approved""#, &definitions, "Approval Status", &ctx);
  let ctx = context(r#"{Age:17,RiskCategory:"Medium",isAffordable:true}"#);
  assert_decision(r#""Declined""#, &definitions, "Approval Status", &ctx);
  let ctx = context(r#"{Age:18,RiskCategory:"High",isAffordable:true}"#);
  assert_decision(r#""Declined""#, &definitions, "Approval Status", &ctx);
}

#[test]
fn _2_0005() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0005, "file:///2_0005.dmn").unwrap();
  let ctx = context(r#"{Age:18,RiskCategory:"Medium",isAffordable:true}"#);
  assert_decision(r#""Approved""#, &definitions, "Approval Status", &ctx);
  let ctx = context(r#"{Age:17,RiskCategory:"Medium",isAffordable:true}"#);
  assert_decision(r#""Declined""#, &definitions, "Approval Status", &ctx);
  let ctx = context(r#"{Age:18,RiskCategory:"High",isAffordable:true}"#);
  assert_decision(r#""Declined""#, &definitions, "Approval Status", &ctx);
}

#[test]
fn _2_0006() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0006, "file:///2_0006.dmn").unwrap();
  let ctx = context(r#"{Age:18,RiskCategory:"Medium",isAffordable:true}"#);
  assert_decision(r#""Approved""#, &definitions, "Approval Status", &ctx);
  let ctx = context(r#"{Age:17,RiskCategory:"Medium",isAffordable:true}"#);
  assert_decision(r#""Declined""#, &definitions, "Approval Status", &ctx);
  let ctx = context(r#"{Age:18,RiskCategory:"High",isAffordable:true}"#);
  assert_decision(r#""Declined""#, &definitions, "Approval Status", &ctx);
}

#[test]
fn _2_0007() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0007, "file:///2_0007.dmn").unwrap();
  let ctx = context(r#"{Age:18,RiskCategory:"Medium",isAffordable:true}"#);
  assert_decision(r#""Approved""#, &definitions, "Approval Status", &ctx);
  let ctx = context(r#"{Age:17,RiskCategory:"Medium",isAffordable:true}"#);
  assert_decision(r#""Declined""#, &definitions, "Approval Status", &ctx);
  let ctx = context(r#"{Age:18,RiskCategory:"High",isAffordable:true}"#);
  assert_decision(r#""Declined""#, &definitions, "Approval Status", &ctx);
}

#[test]
fn _2_0008() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0008, "file:///2_0008.dmn").unwrap();
  let ctx = context(r#"{loan:{principal:600000,rate:0.0375,termMonths:360}}"#);
  assert_decision(r#"2778.6935494327667680885203832"#, &definitions, "payment", &ctx);
  let ctx = context(r#"{loan:{principal:30000,rate:0.0475,termMonths:60}}"#);
  assert_decision(r#"562.70735937326592715621433136"#, &definitions, "payment", &ctx);
  let ctx = context(r#"{loan:{principal:600000,rate:0.0399,termMonths:360}}"#);
  assert_decision(r#"2861.03377700390163671626278"#, &definitions, "payment", &ctx);
}

#[test]
fn _2_0009() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0009, "file:///2_0009.dmn").unwrap();
  let ctx = context(r#"{fee:100,Loan:{amount:600000,rate:0.0375,term:360}}"#);
  assert_decision(r#"2878.6935494327667680885203832"#, &definitions, "MonthlyPayment", &ctx);
}

#[test]
fn _3_0003() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_3_0003, "file:///3_0003.dmn").unwrap();
  let ctx = context(r#"{Loans:[{amount:200000,rate:.041,term:360},{amount:20000,rate:.049,term:60}]}"#);
  assert_decision(
    r#"[966.3967422049753602329651206,376.5090706325024247283858321]"#,
    &definitions,
    "MonthlyPayment",
    &ctx,
  );
}

#[test]
fn _3_0008() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_3_0008, "file:///3_0008.dmn").unwrap();
  let ctx = context(r#"{}"#);
  assert_decision(r#"["a","b","c"]"#, &definitions, "listGen1", &ctx);
}

#[test]
fn _3_0012() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_3_0012, "file:///3_0012.dmn").unwrap();
  let ctx = context(r#"{list1:["a","b","c"],list2:["x","y","z"]}"#);
  assert_decision(r#"false"#, &definitions, "listContainsList", &ctx);
}

#[test]
#[ignore]
fn _3_0014() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_3_0014, "file:///3_0014.dmn").unwrap();
  let ctx = context(r#"{}"#);
  assert_decision(r#"true"#, &definitions, "Bankrates", &ctx);
}

// this test is temporarily to make clippy not to complaint about unused assert_service
#[test]
#[ignore]
fn _n_000n() {
  let definitions = dmntk_model::parse(dmntk_examples::DMN_3_0003, "file:///3_0003.dmn").unwrap();
  let ctx = context(r#"{Loans:[{amount:200000,rate:.041,term:360},{amount:20000,rate:.049,term:60}]}"#);
  assert_service(r#"[966.3967422049817,376.5090706324938]"#, &definitions, "MonthlyPayment", &ctx);
}

//--------------------------------------------------------------------------------------------------
// Utility functions used in this module.
//--------------------------------------------------------------------------------------------------

/// Utility function that creates a `FEEL` context from specified input expression.
fn context(input: &str) -> FeelContext {
  let scope = Scope::default();
  match dmntk_feel_parser::parse_context(&scope, input, false) {
    Ok(node) => match dmntk_feel_evaluator::prepare(&node) {
      Ok(evaluator) => match evaluator(&scope) {
        Value::Context(ctx) => ctx,
        other => panic!("ERROR: expected context value, actual value is: {}", other.stringify()),
      },
      Err(reason) => panic!("ERROR: building evaluator failed with reason: {}", reason),
    },
    Err(reason) => panic!("ERROR: parsing context failed with reason: {}", reason),
  }
}

/// Utility function that evaluates a `Decision` specified by name and compares the result.
fn assert_decision(expected: &str, definitions: &Definitions, name: &str, ctx: &FeelContext) {
  let actual = evaluate_decision_by_name(definitions, name, ctx).unwrap().stringify();
  assert_eq!(expected, actual);
}

/// Utility function that evaluates a `DecisionService` specified by name and compares the result.
fn assert_service(expected: &str, definitions: &Definitions, name: &str, ctx: &FeelContext) {
  let actual = eval_decision_service_by_name(definitions, name, ctx).unwrap().stringify();
  assert_eq!(expected, actual);
}
