/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model evaluator
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use crate::builders::{
  information_item_type, BusinessKnowledgeModelEvaluator, DecisionEvaluator, InputDataContextEvaluator, InputDataEvaluator, ItemDefinitionContextEvaluator,
  ItemDefinitionEvaluator, ItemDefinitionTypeEvaluator,
};
use crate::errors::*;
use dmntk_common::{Result, Stringify};
use dmntk_feel::context::FeelContext;
use dmntk_feel::values::Value;
use dmntk_feel::{value_null, FeelType, Name, Scope};
use dmntk_model::model::{Decision, Definitions, DmnElement, KnowledgeRequirement, NamedElement};

///
pub struct ModelEvaluator<'evaluator> {
  /// Definitions.
  definitions: &'evaluator Definitions,
  /// Input data evaluator.
  input_data_evaluator: InputDataEvaluator,
  /// Input data context evaluator.
  //input_data_context_evaluator: InputDataContextEvaluator,
  /// Item definition evaluator.
  item_definition_evaluator: ItemDefinitionEvaluator,
  /// Item definition context evaluator.
  //item_definition_context_evaluator: ItemDefinitionContextEvaluator,
  /// Item definition type evaluator.
  item_definition_type_evaluator: ItemDefinitionTypeEvaluator,
  /// Decision evaluator.
  decision_evaluator: DecisionEvaluator,
  /// Business knowledge model evaluator.
  business_knowledge_model_evaluator: BusinessKnowledgeModelEvaluator,
}

impl<'evaluator> ModelEvaluator<'evaluator> {
  /// Creates an instance of [ModelEvaluator].
  pub fn new(definitions: &'evaluator Definitions) -> Result<Self> {
    let input_data_evaluator = InputDataEvaluator::new(definitions)?;
    let input_data_context_evaluator = InputDataContextEvaluator::new(definitions)?;
    let item_definition_evaluator = ItemDefinitionEvaluator::new(definitions)?;
    let item_definition_context_evaluator = ItemDefinitionContextEvaluator::new(definitions)?;
    let item_definition_type_evaluator = ItemDefinitionTypeEvaluator::new(definitions)?;
    let decision_evaluator = DecisionEvaluator::new(definitions, &input_data_context_evaluator, &item_definition_context_evaluator)?;
    let business_knowledge_model_evaluator = BusinessKnowledgeModelEvaluator::new(definitions, &item_definition_type_evaluator)?;
    Ok(Self {
      definitions,
      input_data_evaluator,
      //input_data_context_evaluator,
      item_definition_evaluator,
      //item_definition_context_evaluator,
      item_definition_type_evaluator,
      decision_evaluator,
      business_knowledge_model_evaluator,
    })
  }

  /// Evaluates a decision.
  pub fn eval_decision(&self, decision: &Decision, ctx: &FeelContext) -> Result<Value> {
    let decision_id = decision.id().as_ref().ok_or_else(err_empty_identifier)?;
    println!("AAA:DECISION_ID = {}", decision_id);
    // evaluate decision's local context
    let mut evaluated_context = self.eval_context(ctx, decision)?;
    // values evaluated in requirements may be overwritten by input parameters
    evaluated_context.zip(ctx);
    let scope: Scope = evaluated_context.into();
    println!("AAA:DECISION_SCOPE={}", scope.stringify());
    // evaluate the decision
    let decision_result = self
      .decision_evaluator
      .eval(decision_id, &scope)
      .unwrap_or_else(|| value_null!("no evaluator for decision logic"));
    println!("AAA:DECISION_RESULT={}", decision_result.stringify());
    let decision_result_type = decision_result.type_of();
    println!("AAA:DECISION_RESULT_TYPE={}", decision_result_type.stringify());
    let decision_output_type = if let Some(decision_output_type_ref) = decision.variable().type_ref().as_ref() {
      information_item_type(decision_output_type_ref, &self.item_definition_type_evaluator).unwrap_or(FeelType::Any)
    } else {
      FeelType::Any
    };
    println!("AAA:DECISION_OUTPUT_TYPE={:?}", decision_output_type);
    Ok(decision_result)
    // if decision_result.type_of().is_conformant(&decision_output_type) {
    //   return Ok(decision_result);
    // }
    // // coercion of singleton list into single value
    // if let Value::List(items) = &decision_result {
    //   if items.len() == 1 && items.as_vec()[0].type_of().is_conformant(&decision_output_type) {
    //     println!("AAA:COERCION_LIST_INTO_SINGLE_VALUE");
    //     return Ok(items.as_vec()[0].clone());
    //   }
    // }
    // // coercion of single value into singleton list
    // if let FeelType::List(expected_inner_type) = decision_output_type {
    //   if decision_result.type_of().is_conformant(expected_inner_type.borrow()) {
    //     println!("AAA:COERCION_SINGLE_VALUE_INTO_LIST");
    //     return Ok(Value::List(Values::new(vec![decision_result])));
    //   }
    // }
    // Ok(value_null!("coercion did not work"))
  }
  /// Evaluates local context for [Decision].
  fn eval_context(&self, ctx: &FeelContext, decision: &Decision) -> Result<FeelContext> {
    println!("AAA:EXTERNAL_CONTEXT = {}", ctx.stringify());
    let mut evaluated_ctx: FeelContext = Default::default();
    self.eval_knowledge_requirements(decision.knowledge_requirements(), &mut evaluated_ctx)?;

    for information_requirement in decision.information_requirements() {
      if let Some(href) = information_requirement.required_decision() {
        self.eval_required_decision_by_id(href.into(), ctx, &mut evaluated_ctx)?;
      }
      if let Some(href) = information_requirement.required_input() {
        println!("AAA:INFORMATION_REQUIREMENT = {:?}", href);
        let (name, value) = self.eval_input_data(href.into(), &Value::Context(ctx.clone()))?;
        evaluated_ctx.set_entry(&name, value);
        println!("AAA:EVALUATED_CONTEXT_INTERMEDIATE = {}", evaluated_ctx.stringify());
      }
    }
    println!("AAA:EVALUATED_CONTEXT = {}", evaluated_ctx.stringify());
    Ok(evaluated_ctx)
  }

  /// Evaluates input data.
  fn eval_input_data(&self, input_data_id: &str, value: &Value) -> Result<(Name, Value)> {
    self
      .input_data_evaluator
      .eval(input_data_id, value, &self.item_definition_evaluator)
      .ok_or_else(|| err_input_data_with_specified_id_not_found(input_data_id))
  }

  ///
  fn eval_knowledge_requirements(&self, knowledge_requirements: &[KnowledgeRequirement], evl: &mut FeelContext) -> Result<()> {
    for knowledge_requirement in knowledge_requirements {
      if let Some(href) = knowledge_requirement.required_knowledge() {
        //if let Some(bkm) = defs.business_knowledge_model_by_id(id) {
        self.business_knowledge_model_evaluator.eval(href.into(), evl)
        //eval_bkm_knowledge_requirement(defs, bkm, ext, evl)?;
        // } else if let Some(ds) = defs.decision_service_by_id(id) {
        //   //eval_ds_knowledge_requirement(defs, ds, ext, evl)?;
        // } else {
        //   //return Err(invocable_with_id_not_found(id));
        // }
      }
    }
    Ok(())
  }

  /// Evaluates a decision with given identifier and places the result of this evaluation
  /// in `evaluated_context` under the decision's variable name.
  fn eval_required_decision_by_id(&self, decision_id: &str, ctx: &FeelContext, evaluated_ctx: &mut FeelContext) -> Result<()> {
    if let Some(decision) = self.definitions.decision_by_id(decision_id) {
      let variable_name = decision.variable().feel_name().as_ref().ok_or_else(err_empty_feel_name)?;
      evaluated_ctx.set_entry(variable_name, self.eval_decision(decision, ctx)?);
      Ok(())
    } else {
      Err(err_decision_with_name_not_found("XXX"))
    }
  }
}
