/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model evaluator
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! ???

mod business_knowledge_model;
mod decision;
mod input_data;
mod input_data_context;
mod item_definition;
mod item_definition_context;
mod item_definition_type;

use crate::errors::err_invalid_item_definition_type;
pub use business_knowledge_model::BusinessKnowledgeModelEvaluator;
pub use decision::DecisionEvaluator;
use dmntk_common::Result;
use dmntk_feel::FeelType;
use dmntk_model::model::{Expression, ItemDefinition, ItemDefinitionType, NamedElement};
pub use input_data::InputDataEvaluator;
pub use input_data_context::InputDataContextEvaluator;
pub use item_definition::ItemDefinitionEvaluator;
pub use item_definition_context::ItemDefinitionContextEvaluator;
pub use item_definition_type::ItemDefinitionTypeEvaluator;

///
pub fn information_item_type(type_ref: &str, evaluator: &ItemDefinitionTypeEvaluator) -> Option<FeelType> {
  if let Some(feel_type) = type_ref_to_feel_type(type_ref) {
    match feel_type {
      FeelType::String => Some(FeelType::String),
      FeelType::Number => Some(FeelType::Number),
      FeelType::Boolean => Some(FeelType::Boolean),
      FeelType::Date => Some(FeelType::Date),
      FeelType::Time => Some(FeelType::Time),
      FeelType::DateTime => Some(FeelType::DateTime),
      FeelType::DaysAndTimeDuration => Some(FeelType::DaysAndTimeDuration),
      FeelType::YearsAndMonthsDuration => Some(FeelType::YearsAndMonthsDuration),
      _ => None,
    }
  } else {
    evaluator.eval(type_ref)
  }
}

///
fn item_definition_type(item_definition: &ItemDefinition) -> Result<ItemDefinitionType> {
  let feel_type = if let Some(type_ref) = item_definition.type_ref() {
    type_ref_to_feel_type(type_ref)
  } else {
    None
  };
  let condition = (
    item_definition.type_ref().is_some(),
    feel_type.is_some(),
    !item_definition.item_components().is_empty(),
    item_definition.is_collection(),
  );
  match condition {
    (_, true, false, false) => Ok(ItemDefinitionType::SimpleType(feel_type.unwrap())),
    (true, false, false, false) => Ok(ItemDefinitionType::ReferencedType(item_definition.type_ref().as_ref().unwrap().clone())),
    (false, false, true, false) => Ok(ItemDefinitionType::ComponentType),
    (_, true, false, true) => Ok(ItemDefinitionType::CollectionOfSimpleType(feel_type.unwrap())),
    (false, false, true, true) => Ok(ItemDefinitionType::CollectionOfComponentType),
    (true, false, false, true) => Ok(ItemDefinitionType::CollectionOfReferencedType(
      item_definition.type_ref().as_ref().unwrap().clone(),
    )),
    _ => Err(err_invalid_item_definition_type(item_definition.name())),
  }
}

///
fn type_ref_to_feel_type(type_ref: &str) -> Option<FeelType> {
  match type_ref.trim() {
    "string" => Some(FeelType::String),
    "number" => Some(FeelType::Number),
    "boolean" => Some(FeelType::Boolean),
    "date" => Some(FeelType::Date),
    "time" => Some(FeelType::Time),
    "dateTime" => Some(FeelType::DateTime),
    "dayTimeDuration" => Some(FeelType::DaysAndTimeDuration),
    "yearMonthDuration" => Some(FeelType::YearsAndMonthsDuration),
    _ => None,
  }
}

#[cfg(test)]
mod tests {
  use crate::builders::type_ref_to_feel_type;
  use dmntk_feel::FeelType;

  #[test]
  fn test_type_ref_to_feel_type() {
    assert_eq!(Some(FeelType::String), type_ref_to_feel_type("string"));
    assert_eq!(Some(FeelType::Number), type_ref_to_feel_type("number"));
    assert_eq!(Some(FeelType::Boolean), type_ref_to_feel_type("boolean"));
    assert_eq!(Some(FeelType::Date), type_ref_to_feel_type("date"));
    assert_eq!(Some(FeelType::Time), type_ref_to_feel_type("time"));
    assert_eq!(Some(FeelType::DateTime), type_ref_to_feel_type("dateTime"));
    assert_eq!(Some(FeelType::DaysAndTimeDuration), type_ref_to_feel_type("dayTimeDuration"));
    assert_eq!(Some(FeelType::YearsAndMonthsDuration), type_ref_to_feel_type("yearMonthDuration"));
    assert_eq!(None, type_ref_to_feel_type("text"));
  }
}
