/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model evaluator
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Builder for item definition type evaluator.

use crate::errors::*;
use dmntk_common::Result;
use dmntk_feel::{FeelType, Name};
use dmntk_model::model::{Definitions, ItemDefinition, ItemDefinitionType, NamedElement};
use std::collections::{BTreeMap, HashMap};

/// Type of function that evaluates the item definition type.
type EvaluatorFn = Box<dyn Fn(&ItemDefinitionTypeEvaluator) -> Option<FeelType>>;

/// Item definition type evaluators.
#[derive(Default)]
pub struct ItemDefinitionTypeEvaluator {
  /// Map of item definition type evaluators.
  evaluators: HashMap<String, EvaluatorFn>,
}

impl ItemDefinitionTypeEvaluator {
  /// Creates item definition type evaluators.
  pub fn new(definitions: &Definitions) -> Result<Self> {
    let mut evaluators = HashMap::new();
    for item_definition in definitions.item_definitions() {
      let evaluator = build_item_definition_type_evaluator(item_definition)?;
      let type_ref = item_definition.name().to_string();
      evaluators.insert(type_ref, evaluator);
    }
    Ok(Self { evaluators })
  }
  /// Evaluates a type of item definition with specified type reference name.
  pub fn eval(&self, type_ref: &str) -> Option<FeelType> {
    if let Some(evaluator) = self.evaluators.get(type_ref) {
      evaluator(self)
    } else {
      None
    }
  }
}

///
pub fn build_item_definition_type_evaluator(item_definition: &ItemDefinition) -> Result<EvaluatorFn> {
  match super::item_definition_type(item_definition)? {
    ItemDefinitionType::SimpleType(feel_type) => simple_type(feel_type),
    ItemDefinitionType::ReferencedType(ref_type) => referenced_type(ref_type),
    ItemDefinitionType::ComponentType => component_type(item_definition),
    ItemDefinitionType::CollectionOfSimpleType(feel_type) => collection_of_simple_type(feel_type),
    ItemDefinitionType::CollectionOfReferencedType(ref_type) => collection_of_referenced_type(ref_type),
    ItemDefinitionType::CollectionOfComponentType => collection_of_component_type(item_definition),
  }
}

///
fn simple_type(feel_type: FeelType) -> Result<EvaluatorFn> {
  match feel_type {
    FeelType::String => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::String))),
    FeelType::Number => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::Number))),
    FeelType::Boolean => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::Boolean))),
    FeelType::Date => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::Date))),
    FeelType::Time => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::Time))),
    FeelType::DateTime => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::DateTime))),
    FeelType::DaysAndTimeDuration => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::DaysAndTimeDuration))),
    FeelType::YearsAndMonthsDuration => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::YearsAndMonthsDuration))),
    _ => Err(err_unsupported_feel_type(feel_type)),
  }
}

///
fn referenced_type(ref_type: String) -> Result<EvaluatorFn> {
  Ok(Box::new(move |evaluators: &ItemDefinitionTypeEvaluator| evaluators.eval(&ref_type)))
}

///
fn component_type(item_definition: &ItemDefinition) -> Result<EvaluatorFn> {
  let mut type_evaluators: Vec<(Name, EvaluatorFn)> = vec![];
  for component_item_definition in item_definition.item_components() {
    type_evaluators.push((
      component_item_definition.name().into(),
      build_item_definition_type_evaluator(component_item_definition)?,
    ));
  }
  Ok(Box::new(move |evaluators: &ItemDefinitionTypeEvaluator| {
    let mut entries = BTreeMap::new();
    for (component_name, component_evaluator) in &type_evaluators {
      if let Some(feel_type) = component_evaluator(evaluators) {
        entries.insert(component_name.clone(), feel_type);
      }
    }
    Some(FeelType::Context(entries))
  }))
}

///
fn collection_of_simple_type(feel_type: FeelType) -> Result<EvaluatorFn> {
  match feel_type {
    FeelType::String => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::list(&FeelType::String)))),
    FeelType::Number => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::list(&FeelType::Number)))),
    FeelType::Boolean => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::list(&FeelType::Boolean)))),
    FeelType::Date => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::list(&FeelType::Date)))),
    FeelType::Time => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::list(&FeelType::Time)))),
    FeelType::DateTime => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| Some(FeelType::list(&FeelType::DateTime)))),
    FeelType::DaysAndTimeDuration => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| {
      Some(FeelType::list(&FeelType::DaysAndTimeDuration))
    })),
    FeelType::YearsAndMonthsDuration => Ok(Box::new(move |_: &ItemDefinitionTypeEvaluator| {
      Some(FeelType::list(&FeelType::YearsAndMonthsDuration))
    })),
    _ => Err(err_unsupported_feel_type(feel_type)),
  }
}

///
fn collection_of_referenced_type(type_ref: String) -> Result<EvaluatorFn> {
  Ok(Box::new(move |evaluators: &ItemDefinitionTypeEvaluator| {
    evaluators.eval(&type_ref).map(|feel_type| FeelType::List(Box::new(feel_type)))
  }))
}

///
fn collection_of_component_type(item_definition: &ItemDefinition) -> Result<EvaluatorFn> {
  let mut type_evaluators: Vec<(Name, EvaluatorFn)> = vec![];
  for component_item_definition in item_definition.item_components() {
    type_evaluators.push((
      component_item_definition.name().into(),
      build_item_definition_type_evaluator(component_item_definition)?,
    ));
  }
  Ok(Box::new(move |evaluators: &ItemDefinitionTypeEvaluator| {
    let mut entries = BTreeMap::new();
    for (component_name, component_evaluator) in &type_evaluators {
      if let Some(feel_type) = component_evaluator(evaluators) {
        entries.insert(component_name.clone(), feel_type);
      }
    }
    Some(FeelType::List(Box::new(FeelType::Context(entries))))
  }))
}

#[cfg(test)]
mod tests {
  use crate::builders::item_definition_type::ItemDefinitionTypeEvaluator;
  use dmntk_examples::item_definition::*;
  use dmntk_feel::{FeelType, Name};

  #[test]
  fn simple_type_string() {
    let definitions = &dmntk_model::parse(DMN_0101, "file:///0101.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::String), evaluators.eval("tCustomerName"));
  }

  #[test]
  fn simple_type_number() {
    let definitions = &dmntk_model::parse(DMN_0102, "file:///0102.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::Number), evaluators.eval("tMonthlySalary"));
  }

  #[test]
  fn simple_type_boolean() {
    let definitions = &dmntk_model::parse(DMN_0103, "file:///0103.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::Boolean), evaluators.eval("tIsAffordable"));
  }

  #[test]
  fn simple_type_date() {
    let definitions = &dmntk_model::parse(DMN_0104, "file:///0104.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::Date), evaluators.eval("tBirthday"));
  }

  #[test]
  fn simple_type_time() {
    let definitions = &dmntk_model::parse(DMN_0105, "file:///0105.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::Time), evaluators.eval("tDeliveryTime"));
  }

  #[test]
  fn simple_type_date_time() {
    let definitions = &dmntk_model::parse(DMN_0106, "file:///0106.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::DateTime), evaluators.eval("tAppointment"));
  }

  #[test]
  fn simple_type_days_and_time_duration() {
    let definitions = &dmntk_model::parse(DMN_0107, "file:///0107.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::DaysAndTimeDuration), evaluators.eval("tCourseDuration"));
  }

  #[test]
  fn simple_type_years_and_month_duration() {
    let definitions = &dmntk_model::parse(DMN_0108, "file:///0108.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::YearsAndMonthsDuration), evaluators.eval("tGrowthDuration"));
  }

  #[test]
  fn referenced_type_string() {
    let definitions = &dmntk_model::parse(DMN_0201, "file:///0201.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::String), evaluators.eval("tCustomerName"));
  }

  #[test]
  fn referenced_type_number() {
    let definitions = &dmntk_model::parse(DMN_0202, "file:///0202.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::Number), evaluators.eval("tMonthlySalary"));
  }

  #[test]
  fn component_type() {
    let definitions = &dmntk_model::parse(DMN_0301, "file:///0301.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    let name_principal: Name = "principal".into();
    let name_rate: Name = "rate".into();
    let name_term_months: Name = "termMonths".into();
    let type_number = FeelType::Number;
    let component_type = FeelType::context(&[(&name_principal, &type_number), (&name_rate, &type_number), (&name_term_months, &type_number)]);
    assert_eq!(Some(component_type), evaluators.eval("tLoan"));
  }

  #[test]
  fn collection_of_simple_type_string() {
    let definitions = &dmntk_model::parse(DMN_0401, "file:///0401.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::list(&FeelType::String)), evaluators.eval("tItems"));
  }

  #[test]
  fn collection_of_simple_type_number() {
    let definitions = &dmntk_model::parse(DMN_0402, "file:///0402.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::list(&FeelType::Number)), evaluators.eval("tItems"));
  }

  #[test]
  fn collection_of_simple_type_boolean() {
    let definitions = &dmntk_model::parse(DMN_0403, "file:///0403.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::list(&FeelType::Boolean)), evaluators.eval("tItems"));
  }

  #[test]
  fn collection_of_simple_type_date() {
    let definitions = &dmntk_model::parse(DMN_0404, "file:///0404.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::list(&FeelType::Date)), evaluators.eval("tItems"));
  }

  #[test]
  fn collection_of_simple_type_time() {
    let definitions = &dmntk_model::parse(DMN_0405, "file:///0405.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::list(&FeelType::Time)), evaluators.eval("tItems"));
  }

  #[test]
  fn collection_of_simple_type_date_time() {
    let definitions = &dmntk_model::parse(DMN_0406, "file:///0406.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::list(&FeelType::DateTime)), evaluators.eval("tItems"));
  }

  #[test]
  fn collection_of_simple_type_days_and_time_duration() {
    let definitions = &dmntk_model::parse(DMN_0407, "file:///0407.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::list(&FeelType::DaysAndTimeDuration)), evaluators.eval("tItems"));
  }

  #[test]
  fn collection_of_simple_type_years_and_months_duration() {
    let definitions = &dmntk_model::parse(DMN_0408, "file:///0408.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::list(&FeelType::YearsAndMonthsDuration)), evaluators.eval("tItems"));
  }

  #[test]
  fn test_evaluate_input_data_0501_1() {
    let definitions = &dmntk_model::parse(DMN_0501, "file:///0501.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::list(&FeelType::String)), evaluators.eval("tItems"));
  }

  #[test]
  fn test_evaluate_input_data_0601_1() {
    let definitions = &dmntk_model::parse(DMN_0601, "file:///0601.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    let name_number: Name = "number".into();
    let name_name: Name = "name".into();
    let name_manager: Name = "manager".into();
    let type_number = FeelType::Number;
    let type_string = FeelType::String;
    let component_type = FeelType::context(&[(&name_number, &type_number), (&name_name, &type_string), (&name_manager, &type_string)]);
    let list_type = FeelType::list(&component_type);
    assert_eq!(Some(list_type), evaluators.eval("tItems"));
  }
}
