/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model evaluator
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Item definition context evaluator.
//!
//! This evaluator generates a context that contains all values defined by an item definition.
//! Generated contexts are used to properly parse expressions.
//! `FEEL` grammar heavily depends on the context in which the expressions are parsed.
//!
//! # Example
//!
//! tbd

use crate::errors::*;
use dmntk_common::Result;
use dmntk_feel::context::FeelContext;
use dmntk_feel::values::{Value, Values};
use dmntk_feel::{FeelType, Name};
use dmntk_model::model::{Definitions, ItemDefinition, ItemDefinitionType, NamedElement};
use std::collections::{BTreeMap, HashMap};

/// Type of closure that evaluates the item definition context.
type EvaluatorFn = Box<dyn Fn(&Name, &mut FeelContext, &ItemDefinitionContextEvaluator) -> FeelType>;

/// Item definition type evaluators.
#[derive(Default)]
pub struct ItemDefinitionContextEvaluator {
  /// Map of item definition context evaluators.
  /// Keys are type reference names that uniquely identify item definitions.
  /// Values are evaluation closures.
  evaluators: HashMap<String, EvaluatorFn>,
}

impl ItemDefinitionContextEvaluator {
  /// Creates item definition type evaluators.
  pub fn new(definitions: &Definitions) -> Result<Self> {
    let mut evaluators = HashMap::new();
    for item_definition in definitions.item_definitions() {
      let evaluator = item_definition_context_evaluator(item_definition)?;
      let type_ref = item_definition.name().to_string();
      evaluators.insert(type_ref, evaluator);
    }
    Ok(Self { evaluators })
  }
  /// Evaluates a context from item definition with specified type reference name.
  pub fn eval(&self, type_ref: &str, name: &Name, ctx: &mut FeelContext) -> FeelType {
    if let Some(evaluator) = self.evaluators.get(type_ref) {
      evaluator(name, ctx, self)
    } else {
      FeelType::Any
    }
  }
}

///
fn item_definition_context_evaluator(item_definition: &ItemDefinition) -> Result<EvaluatorFn> {
  match super::item_definition_type(item_definition)? {
    ItemDefinitionType::SimpleType(feel_type) => simple_type_context_evaluator(feel_type),
    ItemDefinitionType::ReferencedType(ref_type) => referenced_type_context_evaluator(ref_type),
    ItemDefinitionType::ComponentType => component_type_context_evaluator(item_definition),
    ItemDefinitionType::CollectionOfSimpleType(feel_type) => collection_of_simple_type_context_evaluator(feel_type),
    ItemDefinitionType::CollectionOfReferencedType(ref_type) => collection_of_referenced_type_context_evaluator(ref_type),
    ItemDefinitionType::CollectionOfComponentType => collection_of_component_type_context_evaluator(item_definition),
  }
}

///
fn simple_type_context_evaluator(feel_type: FeelType) -> Result<EvaluatorFn> {
  if matches!(
    feel_type,
    FeelType::String
      | FeelType::Number
      | FeelType::Boolean
      | FeelType::Date
      | FeelType::Time
      | FeelType::DateTime
      | FeelType::DaysAndTimeDuration
      | FeelType::YearsAndMonthsDuration
  ) {
    Ok(Box::new(move |name: &Name, ctx: &mut FeelContext, _: &ItemDefinitionContextEvaluator| {
      ctx.set_entry(name, Value::FeelType(feel_type.clone()));
      feel_type.clone()
    }))
  } else {
    Err(err_unsupported_feel_type(feel_type))
  }
}

///
fn referenced_type_context_evaluator(ref_type: String) -> Result<EvaluatorFn> {
  Ok(Box::new(
    move |name: &Name, ctx: &mut FeelContext, evaluator: &ItemDefinitionContextEvaluator| evaluator.eval(&ref_type, name, ctx),
  ))
}

///
fn component_type_context_evaluator(item_definition: &ItemDefinition) -> Result<EvaluatorFn> {
  let mut context_evaluators: Vec<(Name, EvaluatorFn)> = vec![];
  for component_item_definition in item_definition.item_components() {
    context_evaluators.push((
      component_item_definition.name().into(),
      item_definition_context_evaluator(component_item_definition)?,
    ));
  }
  Ok(Box::new(
    move |name: &Name, ctx: &mut FeelContext, evaluator: &ItemDefinitionContextEvaluator| {
      let mut entries = BTreeMap::new();
      let mut evaluated_ctx = FeelContext::default();
      for (component_name, component_evaluator) in &context_evaluators {
        let feel_type = component_evaluator(component_name, &mut evaluated_ctx, evaluator);
        entries.insert(component_name.clone(), feel_type);
      }
      ctx.set_entry(name, Value::Context(evaluated_ctx));
      FeelType::Context(entries)
    },
  ))
}

///
fn collection_of_simple_type_context_evaluator(feel_type: FeelType) -> Result<EvaluatorFn> {
  if matches!(
    feel_type,
    FeelType::String
      | FeelType::Number
      | FeelType::Boolean
      | FeelType::Date
      | FeelType::Time
      | FeelType::DateTime
      | FeelType::DaysAndTimeDuration
      | FeelType::YearsAndMonthsDuration
  ) {
    Ok(Box::new(move |name: &Name, ctx: &mut FeelContext, _: &ItemDefinitionContextEvaluator| {
      let list_type = FeelType::List(Box::new(feel_type.clone()));
      let list = Value::List(Values::new(vec![Value::FeelType(feel_type.clone())]));
      ctx.set_entry(name, list);
      list_type
    }))
  } else {
    Err(err_unsupported_feel_type(feel_type))
  }
}

///
fn collection_of_referenced_type_context_evaluator(type_ref: String) -> Result<EvaluatorFn> {
  Ok(Box::new(
    move |name: &Name, ctx: &mut FeelContext, evaluator: &ItemDefinitionContextEvaluator| {
      let mut evaluated_ctx = FeelContext::default();
      let feel_type = evaluator.eval(&type_ref, name, &mut evaluated_ctx);
      let list_type = FeelType::List(Box::new(feel_type.clone()));
      let list = Value::List(Values::new(vec![Value::FeelType(feel_type)]));
      ctx.set_entry(name, list);
      list_type
    },
  ))
}

///
fn collection_of_component_type_context_evaluator(item_definition: &ItemDefinition) -> Result<EvaluatorFn> {
  let mut context_evaluators: Vec<(Name, EvaluatorFn)> = vec![];
  for component_item_definition in item_definition.item_components() {
    context_evaluators.push((
      component_item_definition.name().into(),
      item_definition_context_evaluator(component_item_definition)?,
    ));
  }
  Ok(Box::new(
    move |name: &Name, ctx: &mut FeelContext, evaluator: &ItemDefinitionContextEvaluator| {
      let mut entries = BTreeMap::new();
      let mut evaluated_ctx = FeelContext::default();
      for (component_name, component_evaluator) in &context_evaluators {
        let feel_type = component_evaluator(component_name, &mut evaluated_ctx, evaluator);
        entries.insert(component_name.clone(), feel_type);
      }
      let list_type = FeelType::List(Box::new(FeelType::Context(entries)));
      let list = Value::List(Values::new(vec![Value::Context(evaluated_ctx)]));
      ctx.set_entry(name, list);
      list_type
    },
  ))
}

#[cfg(test)]
mod tests {
  use crate::builders::item_definition_context::ItemDefinitionContextEvaluator;
  use dmntk_common::Stringify;
  use dmntk_examples::item_definition::*;
  use dmntk_feel::context::FeelContext;
  use dmntk_feel::values::{Value, Values};
  use dmntk_feel::{FeelType, Name};

  #[test]
  fn simple_type_string() {
    let definitions = &dmntk_model::parse(DMN_0101, "file:///0101.dmn").unwrap();
    let evaluator = ItemDefinitionContextEvaluator::new(definitions).unwrap();
    let mut ctx = FeelContext::default();
    let expected_type = FeelType::String;
    let mut expected_context = FeelContext::default();
    expected_context.set_entry(&"Customer Name".into(), Value::FeelType(FeelType::String));
    let actual_type = evaluator.eval("tCustomerName", &"Customer Name".into(), &mut ctx);
    assert_eq!(expected_type, actual_type);
    assert_eq!(expected_context, ctx);
    assert_eq!("{Customer Name:type(String)}", ctx.stringify());
  }

  #[test]
  fn simple_type_number() {
    let definitions = &dmntk_model::parse(DMN_0102, "file:///0102.dmn").unwrap();
    let evaluator = ItemDefinitionContextEvaluator::new(definitions).unwrap();
    let mut ctx = FeelContext::default();
    let expected_type = FeelType::Number;
    let mut expected_context = FeelContext::default();
    expected_context.set_entry(&"Monthly Salary".into(), Value::FeelType(FeelType::Number));
    let actual_type = evaluator.eval("tMonthlySalary", &"Monthly Salary".into(), &mut ctx);
    assert_eq!(expected_type, actual_type);
    assert_eq!(expected_context, ctx);
    assert_eq!("{Monthly Salary:type(Number)}", ctx.stringify());
  }
  /*
      #[test]
      fn simple_type_boolean() {
        let definitions = &dmntk_model::parse(DMN_0103, "file:///0103.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::Boolean), evaluators.eval("tIsAffordable"));
      }

      #[test]
      fn simple_type_date() {
        let definitions = &dmntk_model::parse(DMN_0104, "file:///0104.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::Date), evaluators.eval("tBirthday"));
      }

      #[test]
      fn simple_type_time() {
        let definitions = &dmntk_model::parse(DMN_0105, "file:///0105.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::Time), evaluators.eval("tDeliveryTime"));
      }

      #[test]
      fn simple_type_date_time() {
        let definitions = &dmntk_model::parse(DMN_0106, "file:///0106.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::DateTime), evaluators.eval("tAppointment"));
      }

      #[test]
      fn simple_type_days_and_time_duration() {
        let definitions = &dmntk_model::parse(DMN_0107, "file:///0107.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::DaysAndTimeDuration), evaluators.eval("tCourseDuration"));
      }

      #[test]
      fn simple_type_years_and_month_duration() {
        let definitions = &dmntk_model::parse(DMN_0108, "file:///0108.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::YearsAndMonthsDuration), evaluators.eval("tGrowthDuration"));
      }
  */

  #[test]
  fn referenced_type_string() {
    let definitions = &dmntk_model::parse(DMN_0201, "file:///0201.dmn").unwrap();
    let evaluator = ItemDefinitionContextEvaluator::new(definitions).unwrap();
    let mut ctx = FeelContext::default();
    let expected_type = FeelType::String;
    let mut expected_context = FeelContext::default();
    expected_context.set_entry(&"Customer Name".into(), Value::FeelType(FeelType::String));
    let actual_type = evaluator.eval("tCustomerName", &"Customer Name".into(), &mut ctx);
    assert_eq!(expected_type, actual_type);
    assert_eq!(expected_context, ctx);
    assert_eq!("{Customer Name:type(String)}", ctx.stringify());
  }

  /*
  #[test]
  fn referenced_type_number() {
    let definitions = &dmntk_model::parse(DMN_0202, "file:///0202.dmn").unwrap();
    let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
    assert_eq!(Some(FeelType::Number), evaluators.eval("tMonthlySalary"));
  }
  */

  #[test]
  fn component_type() {
    let definitions = &dmntk_model::parse(DMN_0301, "file:///0301.dmn").unwrap();
    let evaluator = ItemDefinitionContextEvaluator::new(definitions).unwrap();
    let mut ctx = FeelContext::default();
    let name_principal: Name = "principal".into();
    let name_rate: Name = "rate".into();
    let name_term_months: Name = "termMonths".into();
    let type_number = FeelType::Number;
    let expected_type = FeelType::context(&[(&name_principal, &type_number), (&name_rate, &type_number), (&name_term_months, &type_number)]);
    let mut inner_context = FeelContext::default();
    inner_context.set_entry(&name_principal, Value::FeelType(type_number.clone()));
    inner_context.set_entry(&name_rate, Value::FeelType(type_number.clone()));
    inner_context.set_entry(&name_term_months, Value::FeelType(type_number));
    let mut expected_context = FeelContext::default();
    expected_context.set_entry(&"Loan".into(), Value::Context(inner_context));
    let actual_type = evaluator.eval("tLoan", &"Loan".into(), &mut ctx);
    assert_eq!(expected_type, actual_type);
    assert_eq!(expected_context, ctx);
    assert_eq!("{Loan:{principal:type(Number),rate:type(Number),termMonths:type(Number)}}", ctx.stringify());
  }

  #[test]
  fn collection_of_simple_type_string() {
    let definitions = &dmntk_model::parse(DMN_0401, "file:///0401.dmn").unwrap();
    let evaluator = ItemDefinitionContextEvaluator::new(definitions).unwrap();
    let mut ctx = FeelContext::default();
    let expected_type = FeelType::List(Box::new(FeelType::String));
    let mut expected_context = FeelContext::default();
    expected_context.set_entry(&"Items".into(), Value::List(Values::new(vec![Value::FeelType(FeelType::String)])));
    let actual_type = evaluator.eval("tItems", &"Items".into(), &mut ctx);
    assert_eq!(expected_type, actual_type);
    assert_eq!(expected_context, ctx);
    assert_eq!("{Items:[type(String)]}", ctx.stringify());
  }

  /*
      #[test]
      fn collection_of_simple_type_number() {
        let definitions = &dmntk_model::parse(DMN_0402, "file:///0402.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::list(&FeelType::Number)), evaluators.eval("tItems"));
      }

      #[test]
      fn collection_of_simple_type_boolean() {
        let definitions = &dmntk_model::parse(DMN_0403, "file:///0403.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::list(&FeelType::Boolean)), evaluators.eval("tItems"));
      }

      #[test]
      fn collection_of_simple_type_date() {
        let definitions = &dmntk_model::parse(DMN_0404, "file:///0404.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::list(&FeelType::Date)), evaluators.eval("tItems"));
      }

      #[test]
      fn collection_of_simple_type_time() {
        let definitions = &dmntk_model::parse(DMN_0405, "file:///0405.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::list(&FeelType::Time)), evaluators.eval("tItems"));
      }

      #[test]
      fn collection_of_simple_type_date_time() {
        let definitions = &dmntk_model::parse(DMN_0406, "file:///0406.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::list(&FeelType::DateTime)), evaluators.eval("tItems"));
      }

      #[test]
      fn collection_of_simple_type_days_and_time_duration() {
        let definitions = &dmntk_model::parse(DMN_0407, "file:///0407.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::list(&FeelType::DaysAndTimeDuration)), evaluators.eval("tItems"));
      }

      #[test]
      fn collection_of_simple_type_years_and_months_duration() {
        let definitions = &dmntk_model::parse(DMN_0408, "file:///0408.dmn").unwrap();
        let evaluators = ItemDefinitionTypeEvaluator::new(definitions).unwrap();
        assert_eq!(Some(FeelType::list(&FeelType::YearsAndMonthsDuration)), evaluators.eval("tItems"));
      }

  */
  #[test]
  fn collection_of_referenced_type_string() {
    let definitions = &dmntk_model::parse(DMN_0501, "file:///0501.dmn").unwrap();
    let evaluator = ItemDefinitionContextEvaluator::new(definitions).unwrap();
    let mut ctx = FeelContext::default();
    let expected_type = FeelType::List(Box::new(FeelType::String));
    let mut expected_context = FeelContext::default();
    expected_context.set_entry(&"Items".into(), Value::List(Values::new(vec![Value::FeelType(FeelType::String)])));
    let actual_type = evaluator.eval("tItems", &"Items".into(), &mut ctx);
    assert_eq!(expected_type, actual_type);
    assert_eq!(expected_context, ctx);
    assert_eq!("{Items:[type(String)]}", ctx.stringify());
  }

  #[test]
  fn collection_of_component_type() {
    let definitions = &dmntk_model::parse(DMN_0601, "file:///0601.dmn").unwrap();
    let evaluator = ItemDefinitionContextEvaluator::new(definitions).unwrap();
    let mut ctx = FeelContext::default();
    let name_manager: Name = "manager".into();
    let name_name: Name = "name".into();
    let name_number: Name = "number".into();
    let type_number = FeelType::Number;
    let type_string = FeelType::String;
    let expected_type = FeelType::list(&FeelType::context(&[
      (&name_manager, &type_string),
      (&name_name, &type_string),
      (&name_number, &type_number),
    ]));
    let mut inner_context = FeelContext::default();
    inner_context.set_entry(&name_manager, Value::FeelType(type_string.clone()));
    inner_context.set_entry(&name_name, Value::FeelType(type_string));
    inner_context.set_entry(&name_number, Value::FeelType(type_number));
    let mut expected_context = FeelContext::default();
    expected_context.set_entry(&"Items".into(), Value::List(Values::new(vec![Value::Context(inner_context)])));
    let actual_type = evaluator.eval("tItems", &"Items".into(), &mut ctx);
    assert_eq!("{Items:[{manager:type(String),name:type(String),number:type(Number)}]}", ctx.stringify());
    assert_eq!(expected_type, actual_type);
    assert_eq!(expected_context, ctx);
  }
}
