/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model evaluator
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Input data evaluator.

use crate::builders::item_definition::ItemDefinitionEvaluator;
use crate::errors::{err_empty_feel_name, err_empty_identifier, err_input_data_without_type_reference};
use dmntk_common::{Result, Stringify};
use dmntk_feel::values::Value;
use dmntk_feel::{value_null, Name};
use dmntk_model::model::{Definitions, DmnElement, InputData, NamedElement};
use std::collections::HashMap;

/// Type of closure that evaluates input data definition.
type EvaluatorFn = Box<dyn Fn(&Value, &ItemDefinitionEvaluator) -> (Name, Value)>;

/// Input data evaluator.
#[derive(Default)]
pub struct InputDataEvaluator {
  /// Map with input data evaluators.
  /// Keys are identifiers of input data definitions,
  /// values are input data evaluator closures.
  evaluators: HashMap<String, EvaluatorFn>,
}

impl InputDataEvaluator {
  /// Creates a new input data evaluator.
  pub fn new(definitions: &Definitions) -> Result<Self> {
    let mut evaluators = HashMap::new();
    for input_data in definitions.input_data() {
      let evaluator = build_input_data_evaluator(input_data)?;
      let id = input_data.id().as_ref().ok_or_else(err_empty_identifier)?.clone();
      evaluators.insert(id.clone(), evaluator);
    }
    Ok(Self { evaluators })
  }
  /// Evaluates input data with specified identifier.
  pub fn eval(&self, input_data_id: &str, value: &Value, item_definition_evaluators: &ItemDefinitionEvaluator) -> Option<(Name, Value)> {
    self.evaluators.get(input_data_id).map(|evaluator| evaluator(value, item_definition_evaluators))
  }
}

///
pub fn build_input_data_evaluator(input_data: &InputData) -> Result<EvaluatorFn> {
  let type_ref = input_data
    .variable()
    .type_ref()
    .as_ref()
    .ok_or_else(|| err_input_data_without_type_reference(input_data.name()))?
    .clone();
  let name = input_data.variable().feel_name().as_ref().ok_or_else(err_empty_feel_name)?.clone();
  Ok(match type_ref.as_str() {
    "string" => Box::new(move |value: &Value, _: &ItemDefinitionEvaluator| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::String(_) = v {
            return (name.clone(), v.clone());
          }
        }
      }
      (name.clone(), value_null!())
    }),
    "number" => Box::new(move |value: &Value, _: &ItemDefinitionEvaluator| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::Number(_) = v {
            return (name.clone(), v.clone());
          }
        }
      }
      (name.clone(), value_null!())
    }),
    "boolean" => Box::new(move |value: &Value, _: &ItemDefinitionEvaluator| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::Boolean(_) = v {
            return (name.clone(), v.clone());
          }
        }
      }
      (name.clone(), value_null!())
    }),
    "date" => Box::new(move |value: &Value, _: &ItemDefinitionEvaluator| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::Date(_) = v {
            return (name.clone(), v.clone());
          }
        }
      }
      (name.clone(), value_null!())
    }),
    "time" => Box::new(move |value: &Value, _: &ItemDefinitionEvaluator| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::Time(_) = v {
            return (name.clone(), v.clone());
          }
        }
      }
      (name.clone(), value_null!())
    }),
    "dateTime" => Box::new(move |value: &Value, _: &ItemDefinitionEvaluator| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::DateTime(_) = v {
            return (name.clone(), v.clone());
          }
        }
      }
      (name.clone(), value_null!())
    }),
    "dayTimeDuration" => Box::new(move |value: &Value, _: &ItemDefinitionEvaluator| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::DaysAndTimeDuration(_) = v {
            return (name.clone(), v.clone());
          }
        }
      }
      (name.clone(), value_null!())
    }),
    "yearMonthDuration" => Box::new(move |value: &Value, _: &ItemDefinitionEvaluator| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::YearsAndMonthsDuration(_) = v {
            return (name.clone(), v.clone());
          }
        }
      }
      (name.clone(), value_null!())
    }),
    _ => Box::new(move |value: &Value, item_definition_evaluators: &ItemDefinitionEvaluator| {
      if let Value::Context(ctx) = value {
        if let Some(entry_value) = ctx.get_entry(&name) {
          let evaluated_value = item_definition_evaluators
            .eval(&type_ref, entry_value)
            .unwrap_or_else(|| value_null!("input data evaluator: item definition evaluator '{}' not found", type_ref));
          (name.clone(), evaluated_value)
        } else {
          (name.clone(), value_null!("no name {} in context {}", name.stringify(), ctx.stringify()))
        }
      } else {
        (name.clone(), value_null!("expected context, actual value is {}", value.stringify()))
      }
    }),
  })
}

#[cfg(test)]
mod tests {
  use crate::builders::item_definition::ItemDefinitionEvaluator;
  use crate::builders::InputDataEvaluator;
  use dmntk_examples::input_data::*;
  use dmntk_feel::values::Value;
  use dmntk_feel::{value_number, FeelNumber, Name};

  #[test]
  fn _0001_1() {
    let definitions = &dmntk_model::parse(DMN_0001, "file:///0001.dmn").unwrap();
    let input_data_evaluators = InputDataEvaluator::new(definitions).unwrap();
    let item_definitions_evaluators = ItemDefinitionEvaluator::new(definitions).unwrap();
    let context_str = r#"{ Full Name : "John" }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    assert_eq!(
      Some((Name::new(&["Full", "Name"]), Value::String("John".to_string()))),
      input_data_evaluators.eval("_cba86e4d-e91c-46a2-9176-e9adf88e15db", &Value::Context(context), &item_definitions_evaluators)
    );
    let context_str = r#"{ Full Name : "Phillip" }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    assert_eq!(
      Some((Name::new(&["Full", "Name"]), Value::String("Phillip".to_string()))),
      input_data_evaluators.eval("_cba86e4d-e91c-46a2-9176-e9adf88e15db", &Value::Context(context), &item_definitions_evaluators)
    );
  }

  #[test]
  fn _0001_2() {
    let definitions = &dmntk_model::parse(DMN_0001, "file:///0001.dmn").unwrap();
    let input_data_evaluators = InputDataEvaluator::new(definitions).unwrap();
    let item_definitions_evaluators = ItemDefinitionEvaluator::new(definitions).unwrap();
    let context_str = r#"{ Full Name : 50.0 }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    assert_eq!(
      Some((Name::new(&["Full", "Name"]), Value::Null(None))),
      input_data_evaluators.eval("_cba86e4d-e91c-46a2-9176-e9adf88e15db", &Value::Context(context), &item_definitions_evaluators)
    );
  }

  #[test]
  fn _0002_1() {
    let definitions = &dmntk_model::parse(DMN_0002, "file:///0002.dmn").unwrap();
    let input_data_evaluators = InputDataEvaluator::new(definitions).unwrap();
    let item_definitions_evaluators = ItemDefinitionEvaluator::new(definitions).unwrap();
    let context_str = r#"{ Monthly Salary : 12000.00 }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    assert_eq!(
      Some((Name::new(&["Monthly", "Salary"]), value_number!(12000))),
      input_data_evaluators.eval("_b7a53bad-7a5b-4033-841d-5db6b25834ad", &Value::Context(context), &item_definitions_evaluators)
    );
    let context_str = r#"{ Monthly Salary : 8135.35 }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    assert_eq!(
      Some((Name::new(&["Monthly", "Salary"]), value_number!(813535, 2))),
      input_data_evaluators.eval("_b7a53bad-7a5b-4033-841d-5db6b25834ad", &Value::Context(context), &item_definitions_evaluators)
    );
  }

  #[test]
  fn _0002_2() {
    let definitions = &dmntk_model::parse(DMN_0002, "file:///0002.dmn").unwrap();
    let input_data_evaluators = InputDataEvaluator::new(definitions).unwrap();
    let item_definitions_evaluators = ItemDefinitionEvaluator::new(definitions).unwrap();
    let context_str = r#"{ Monthly Salary : "12000.00" }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    assert_eq!(
      Some((Name::new(&["Monthly", "Salary"]), Value::Null(None))),
      input_data_evaluators.eval("_b7a53bad-7a5b-4033-841d-5db6b25834ad", &Value::Context(context), &item_definitions_evaluators)
    );
  }

  #[test]
  fn _0003_1() {
    let definitions = &dmntk_model::parse(DMN_0003, "file:///0003.dmn").unwrap();
    let input_data_evaluators = InputDataEvaluator::new(definitions).unwrap();
    let item_definitions_evaluators = ItemDefinitionEvaluator::new(definitions).unwrap();
    let context_str = r#"{ Is Affordable : true }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    assert_eq!(
      Some((Name::new(&["Is", "Affordable"]), Value::Boolean(true))),
      input_data_evaluators.eval("_b7a53bad-7a5b-4033-841d-5db6b25834ad", &Value::Context(context), &item_definitions_evaluators)
    );
    let context_str = r#"{ Is Affordable : false }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    assert_eq!(
      Some((Name::new(&["Is", "Affordable"]), Value::Boolean(false))),
      input_data_evaluators.eval("_b7a53bad-7a5b-4033-841d-5db6b25834ad", &Value::Context(context), &item_definitions_evaluators)
    );
  }

  #[test]
  fn _0003_2() {
    let definitions = &dmntk_model::parse(DMN_0003, "file:///0003.dmn").unwrap();
    let input_data_evaluators = InputDataEvaluator::new(definitions).unwrap();
    let item_definitions_evaluators = ItemDefinitionEvaluator::new(definitions).unwrap();
    let context_str = r#"{ Is Affordable : "no" }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    assert_eq!(
      Some((Name::new(&["Is", "Affordable"]), Value::Null(None))),
      input_data_evaluators.eval("_b7a53bad-7a5b-4033-841d-5db6b25834ad", &Value::Context(context), &item_definitions_evaluators)
    );
  }

  #[test]
  fn _0103_1() {
    let definitions = &dmntk_model::parse(DMN_0103, "file:///0103.dmn").unwrap();
    let input_data_evaluators = InputDataEvaluator::new(definitions).unwrap();
    let item_definitions_evaluators = ItemDefinitionEvaluator::new(definitions).unwrap();
    let context_str = r#"{ Employment Status : "EMPLOYED" }"#;
    let context = dmntk_feel_evaluator::evaluate_context(&Default::default(), context_str).unwrap();
    let name = Name::new(&["Employment", "Status"]);
    assert_eq!(
      Some((name, Value::String("EMPLOYED".to_string()))),
      input_data_evaluators.eval("_acfd4e1d-da0a-4842-aa35-ea50dd36fb01", &Value::Context(context), &item_definitions_evaluators)
    );
  }
}
