/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model evaluator
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! ???

use crate::builders::{InputDataContextEvaluator, ItemDefinitionContextEvaluator};
use crate::errors::{err_empty_decision_logic, err_empty_identifier, err_empty_literal_expression};
use crate::eval_dec_table::evaluate_decision_table;
use dmntk_common::{Result, Stringify};
use dmntk_feel::context::FeelContext;
use dmntk_feel::values::Value;
use dmntk_feel::{value_null, Scope};
use dmntk_model::model::{
  Context, Decision, DecisionTable, Definitions, DmnElement, ExpressionInstance, FunctionDefinition, Invocation, LiteralExpression, NamedElement, Relation,
};
use std::collections::HashMap;

/// Type of closure that evaluates decision logic.
type EvaluatorFn = Box<dyn Fn(&Scope) -> Value>;

///
#[derive(Default)]
pub struct DecisionEvaluator {
  evaluators: HashMap<String, EvaluatorFn>,
}

impl DecisionEvaluator {
  /// Creates a new decision evaluator.
  pub fn new(
    definitions: &Definitions,
    input_data_context_evaluator: &InputDataContextEvaluator,
    item_definition_context_evaluator: &ItemDefinitionContextEvaluator,
  ) -> Result<Self> {
    let mut evaluators = HashMap::new();
    for decision in definitions.decisions() {
      let evaluator = build_decision_evaluator(decision, input_data_context_evaluator, item_definition_context_evaluator)?;
      let decision_id = decision.id().as_ref().ok_or_else(err_empty_identifier)?;
      evaluators.insert(decision_id.to_owned(), evaluator);
    }
    Ok(Self { evaluators })
  }
  /// Evaluates a decision with specified identifier.
  pub fn eval(&self, decision_id: &str, scope: &Scope) -> Option<Value> {
    self.evaluators.get(decision_id).map(|evaluator| evaluator(scope))
  }
}

///
fn build_decision_evaluator(
  decision: &Decision,
  input_data_context_evaluator: &InputDataContextEvaluator,
  item_definition_context_evaluator: &ItemDefinitionContextEvaluator,
) -> Result<EvaluatorFn> {
  let expression_instance = decision.decision_logic().as_ref().ok_or_else(err_empty_decision_logic)?;
  let mut ctx = FeelContext::default();
  println!("BBB:PARSING_CONTEXT = {}", ctx.stringify());
  for information_requirement in decision.information_requirements() {
    // if let Some(href) = information_requirement.required_decision() {
    //   self.eval_required_decision_by_id(href.into(), ctx, &mut evaluated_ctx)?;
    // }
    if let Some(href) = information_requirement.required_input() {
      println!("BBB:INFORMATION_REQUIREMENT = {:?}", href);
      input_data_context_evaluator.eval(href.into(), &mut ctx, item_definition_context_evaluator);
      println!("BBB:EVALUATED_CONTEXT_INTERMEDIATE = {}", ctx.stringify());
    }
  }
  println!("BBB:EVALUATED_CONTEXT = {}", ctx.stringify());
  build_expression_instance_evaluator(&ctx, expression_instance)
}

///
fn build_expression_instance_evaluator(ctx: &FeelContext, expression_instance: &ExpressionInstance) -> Result<EvaluatorFn> {
  match expression_instance {
    ExpressionInstance::Context(context) => build_context_evaluator(ctx, context),
    ExpressionInstance::DecisionTable(decision_table) => build_decision_table_evaluator(ctx, decision_table),
    ExpressionInstance::FunctionDefinition(function_definition) => build_function_definition_evaluator(ctx, function_definition),
    ExpressionInstance::Invocation(invocation) => build_invocation_evaluator(ctx, invocation),
    ExpressionInstance::LiteralExpression(literal_expression) => build_literal_expression_evaluator(ctx, literal_expression),
    ExpressionInstance::UnaryTests => build_unary_tests_evaluator(ctx),
    ExpressionInstance::Relation(relation) => build_relation_evaluator(ctx, relation),
  }
}

///
fn build_context_evaluator(ctx: &FeelContext, context: &Context) -> Result<EvaluatorFn> {
  //TODO when evaluators are ready than use an evaluator! :-)
  let context = context.clone();
  let ctx = ctx.clone();
  Ok(Box::new(move |scope: &Scope| {
    let mut evaluated_context = FeelContext::default();
    scope.push(FeelContext::default());
    for context_entry in context.context_entries() {
      if let Some(variable) = &context_entry.variable {
        let name = &variable.feel_name().as_ref().unwrap().clone();
        let evaluator = build_expression_instance_evaluator(&ctx, &context_entry.value).unwrap();
        let value = evaluator(scope);
        scope.set_entry(name, value.clone());
        evaluated_context.set_entry(name, value);
      } else {
        let evaluator = build_expression_instance_evaluator(&ctx, &context_entry.value).unwrap();
        let value = evaluator(scope);
        scope.pop();
        return value;
      }
    }
    scope.pop();
    Value::Context(evaluated_context)
  }))
}

///
fn build_decision_table_evaluator(_ctx: &FeelContext, decision_table: &DecisionTable) -> Result<EvaluatorFn> {
  let dt = decision_table.clone();
  Ok(Box::new(move |scope: &Scope| {
    //TODO when evaluators for decision tables are ready than use an evaluator! :-)
    if let Ok(value) = evaluate_decision_table(scope, dt.clone()) {
      println!("AAA:DEC_TAB_VALUE = {}", value.stringify());
      value
    } else {
      value_null!()
    }
  }))
}

///
fn build_function_definition_evaluator(_ctx: &FeelContext, _function_definition: &FunctionDefinition) -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &Scope| value_null!()))
}

///
fn build_invocation_evaluator(_ctx: &FeelContext, _invocation: &Invocation) -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &Scope| value_null!()))
}

///
fn build_literal_expression_evaluator(ctx: &FeelContext, literal_expression: &LiteralExpression) -> Result<EvaluatorFn> {
  let text = literal_expression.text().as_ref().ok_or_else(err_empty_literal_expression)?;
  let scope = Scope::from(ctx.clone());
  let node = dmntk_feel_parser::parse_expression(&scope, text, false)?;
  let evaluator = dmntk_feel_evaluator::prepare(&node)?;
  Ok(Box::new(move |scope: &Scope| evaluator(scope)))
}

///
fn build_unary_tests_evaluator(_ctx: &FeelContext) -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &Scope| value_null!()))
}

///
fn build_relation_evaluator(_ctx: &FeelContext, _relation: &Relation) -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &Scope| value_null!()))
}
