/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model evaluator
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Business knowledge model evaluator.

use crate::builders::{information_item_type, ItemDefinitionTypeEvaluator};
use crate::errors::*;
use dmntk_common::Result;
use dmntk_feel::context::FeelContext;
use dmntk_feel::values::Value;
use dmntk_feel::{FeelType, FunctionBody, Name, Scope};
use dmntk_model::model::{
  BusinessKnowledgeModel, Context, DecisionTable, Definitions, DmnElement, ExpressionInstance, FunctionDefinition, Invocable, Invocation, LiteralExpression,
  NamedElement, Relation,
};
use std::collections::HashMap;
use std::rc::Rc;

/// Type of closure that evaluates business knowledge model.
type EvaluatorFn = Box<dyn Fn(&mut FeelContext)>;

/// Business knowledge model evaluator.
#[derive(Default)]
pub struct BusinessKnowledgeModelEvaluator {
  evaluators: HashMap<String, EvaluatorFn>,
}

impl BusinessKnowledgeModelEvaluator {
  /// Creates a new business knowledge model evaluator.
  pub fn new(definitions: &Definitions, item_definition_type_evaluators: &ItemDefinitionTypeEvaluator) -> Result<Self> {
    let mut evaluators = HashMap::new();
    for business_knowledge_model in definitions.business_knowledge_models() {
      let function_definition = business_knowledge_model
        .encapsulated_logic()
        .as_ref()
        .ok_or_else(err_empty_encapsulated_logic)?;
      let evaluator = build_business_knowledge_model_evaluator(business_knowledge_model, function_definition, item_definition_type_evaluators)?;
      let id = business_knowledge_model.id().as_ref().ok_or_else(err_empty_identifier)?.clone();
      evaluators.insert(id.clone(), evaluator);
    }
    Ok(Self { evaluators })
  }
  /// Evaluates business knowledge model with specified identifier.
  pub fn eval(&self, business_knowledge_model_id: &str, ctx: &mut FeelContext) {
    if let Some(evaluator) = self.evaluators.get(business_knowledge_model_id) {
      evaluator(ctx);
    }
  }
}

///
fn build_business_knowledge_model_evaluator(
  business_knowledge_model: &BusinessKnowledgeModel,
  function_definition: &FunctionDefinition,
  item_definition_type_evaluators: &ItemDefinitionTypeEvaluator,
) -> Result<EvaluatorFn> {
  let mut local_context = FeelContext::default();
  let mut formal_parameters = vec![];
  for information_item in function_definition.formal_parameters() {
    let type_ref = information_item.type_ref().as_ref().ok_or_else(err_empty_feel_type_reference)?;
    let feel_type = information_item_type(type_ref, item_definition_type_evaluators).ok_or_else(err_empty_feel_type)?;
    let feel_name = information_item.feel_name().as_ref().ok_or_else(err_empty_feel_name)?;
    formal_parameters.push((feel_name.clone(), feel_type.clone()));
    local_context.set_entry(feel_name, Value::FeelType(feel_type));
  }
  let output_variable_name = business_knowledge_model
    .variable()
    .feel_name()
    .as_ref()
    .ok_or_else(err_empty_feel_name)?
    .clone();
  if let Some(expression_instance) = function_definition.body() {
    let scope: Scope = local_context.into();
    match expression_instance {
      ExpressionInstance::Context(context) => build_context_evaluator(context),
      ExpressionInstance::DecisionTable(decision_table) => build_decision_table_evaluator(decision_table),
      ExpressionInstance::FunctionDefinition(function_definition) => build_function_definition_evaluator(function_definition),
      ExpressionInstance::Invocation(invocation) => build_invocation_evaluator(invocation),
      ExpressionInstance::LiteralExpression(literal_expression) => {
        build_literal_expression_evaluator(&scope, formal_parameters, literal_expression, &output_variable_name)
      }
      ExpressionInstance::UnaryTests => build_unary_tests_evaluator(),
      ExpressionInstance::Relation(relation) => build_relation_evaluator(relation),
    }
  } else {
    Ok(Box::new(move |_: &mut FeelContext| ()))
  }
}

///
fn build_context_evaluator(_context: &Context) -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &mut FeelContext| ()))
}

///
fn build_decision_table_evaluator(_decision_table: &DecisionTable) -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &mut FeelContext| ()))
}

///
fn build_function_definition_evaluator(_function_definition: &FunctionDefinition) -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &mut FeelContext| ()))
}

///
fn build_invocation_evaluator(_invocation: &Invocation) -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &mut FeelContext| ()))
}

///
fn build_literal_expression_evaluator(
  scope: &Scope,
  formal_parameters: Vec<(Name, FeelType)>,
  literal_expression: &LiteralExpression,
  output_variable_name: &Name,
) -> Result<EvaluatorFn> {
  let text = literal_expression.text().as_ref().ok_or_else(err_empty_literal_expression)?;
  let node = dmntk_feel_parser::parse_textual_expression(scope, text, false)?;
  let evaluator = dmntk_feel_evaluator::prepare(&node)?;
  let function = Value::FunctionDefinition(formal_parameters, FunctionBody::LiteralExpression(Rc::new(evaluator)));
  let name = output_variable_name.clone();
  Ok(Box::new(move |ctx: &mut FeelContext| ctx.set_entry(&name, function.clone())))
}

///
fn build_unary_tests_evaluator() -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &mut FeelContext| ()))
}

///
fn build_relation_evaluator(_relation: &Relation) -> Result<EvaluatorFn> {
  Ok(Box::new(move |_: &mut FeelContext| ()))
}
