/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL definitions.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Definitions of `FEEL` types.

use self::errors::*;
use crate::context::FeelContext;
use crate::names::Name;
use crate::values::{Value, Values, VALUE_NULL};
use dmntk_common::{DmntkError, Result, Stringify};
use std::collections::BTreeMap;
use std::str::FromStr;

pub const FEEL_TYPE_NAME_ANY: &str = "Any";
const FEEL_TYPE_NAME_BOOLEAN: &str = "boolean";
const FEEL_TYPE_NAME_DATE: &str = "date";
const FEEL_TYPE_NAME_DATE_AND_TIME: &str = "date and time";
const FEEL_TYPE_NAME_DAYS_AND_TIME_DURATION: &str = "days and time duration";
const FEEL_TYPE_NAME_NULL: &str = "Null";
const FEEL_TYPE_NAME_NUMBER: &str = "number";
const FEEL_TYPE_NAME_STRING: &str = "string";
const FEEL_TYPE_NAME_TIME: &str = "time";
const FEEL_TYPE_NAME_YEARS_AND_MONTHS_DURATION: &str = "years and months duration";

#[derive(Debug, Clone, PartialEq)]
pub enum FeelType {
  /// Type representing any valid `FEEL` type.
  Any,
  /// Type representing a `boolean` value.
  Boolean,
  /// Type representing `context` values.
  Context(BTreeMap<Name, FeelType>),
  /// Type representing `date` value.
  Date,
  /// Type representing `date and time` value.
  DateTime,
  /// Type representing `days and time duration` value.
  DaysAndTimeDuration,
  /// Type representing `function` values.
  Function(Vec<FeelType>, Box<FeelType>),
  /// Type representing `list` values.
  List(Box<FeelType>),
  /// Type representing `null` value.
  Null,
  /// Type representing `number` value.
  Number,
  /// Type representing `range` values.
  Range(Box<FeelType>),
  /// Type representing `string` value.
  String,
  /// Type representing `time` value.
  Time,
  /// Type representing `years and months duration` value.
  YearsAndMonthsDuration,
}

impl Stringify for FeelType {
  fn stringify(&self) -> String {
    match self {
      FeelType::Any => FEEL_TYPE_NAME_ANY,
      FeelType::Boolean => FEEL_TYPE_NAME_BOOLEAN,
      FeelType::Context(_) => "context tbd",
      FeelType::Date => FEEL_TYPE_NAME_DATE,
      FeelType::DateTime => FEEL_TYPE_NAME_DATE_AND_TIME,
      FeelType::DaysAndTimeDuration => FEEL_TYPE_NAME_DAYS_AND_TIME_DURATION,
      FeelType::Function(_, _) => "function type (tbd)",
      FeelType::List(_) => "list type (tbd)",
      FeelType::Null => FEEL_TYPE_NAME_NULL,
      FeelType::Number => FEEL_TYPE_NAME_NUMBER,
      FeelType::Range(_) => "range type (tbd)",
      FeelType::String => FEEL_TYPE_NAME_STRING,
      FeelType::Time => FEEL_TYPE_NAME_TIME,
      FeelType::YearsAndMonthsDuration => FEEL_TYPE_NAME_YEARS_AND_MONTHS_DURATION,
    }
    .to_string()
  }
}

impl FromStr for FeelType {
  type Err = DmntkError;
  /// Converts a string to built-in type.
  fn from_str(s: &str) -> Result<Self, Self::Err> {
    match s {
      FEEL_TYPE_NAME_ANY => Ok(Self::Any),
      FEEL_TYPE_NAME_BOOLEAN => Ok(Self::Boolean),
      FEEL_TYPE_NAME_DATE => Ok(Self::Date),
      FEEL_TYPE_NAME_DATE_AND_TIME => Ok(Self::DateTime),
      FEEL_TYPE_NAME_DAYS_AND_TIME_DURATION => Ok(Self::DaysAndTimeDuration),
      FEEL_TYPE_NAME_NULL => Ok(Self::Null),
      FEEL_TYPE_NAME_NUMBER => Ok(Self::Number),
      FEEL_TYPE_NAME_STRING => Ok(Self::String),
      FEEL_TYPE_NAME_TIME => Ok(Self::Time),
      FEEL_TYPE_NAME_YEARS_AND_MONTHS_DURATION => Ok(Self::YearsAndMonthsDuration),
      _ => Err(invalid_feel_type_name(s)),
    }
  }
}

/// Returns `true` when the specified name is a built-in type name.
pub fn is_built_in_type_name(name: &str) -> bool {
  matches!(
    name,
    FEEL_TYPE_NAME_ANY
      | FEEL_TYPE_NAME_BOOLEAN
      | FEEL_TYPE_NAME_DATE
      | FEEL_TYPE_NAME_DATE_AND_TIME
      | FEEL_TYPE_NAME_DAYS_AND_TIME_DURATION
      | FEEL_TYPE_NAME_NULL
      | FEEL_TYPE_NAME_NUMBER
      | FEEL_TYPE_NAME_STRING
      | FEEL_TYPE_NAME_TIME
      | FEEL_TYPE_NAME_YEARS_AND_MONTHS_DURATION
  )
}

impl FeelType {
  /// Returns a new value cloned from provided value, and retrieved with type checking.
  pub fn retrieve_value_with_type_checking(&self, value: &Value) -> Result<Value> {
    if let Value::Null(_) = value {
      // `null` value is conformant with all types
      return Ok(VALUE_NULL);
    }
    match self {
      FeelType::Any => {
        return Ok(value.clone());
      }
      FeelType::Boolean => {
        if let Value::Boolean(_) = value {
          return Ok(value.clone());
        }
      }
      FeelType::Context(entries) => {
        if let Value::Context(context) = value {
          let mut result = FeelContext::default();
          for (name, entry_type) in entries {
            if let Some(entry_value) = context.get_entry(name) {
              result.set_entry(name, entry_type.retrieve_value_with_type_checking(entry_value)?);
            }
          }
          return Ok(Value::Context(result));
        }
      }
      FeelType::Date => {
        if let Value::Date(_) = value {
          return Ok(value.clone());
        }
      }
      FeelType::DateTime => {
        if let Value::DateTime(_) = value {
          return Ok(value.clone());
        }
      }
      FeelType::DaysAndTimeDuration => {
        if let Value::DaysAndTimeDuration(_) = value {
          return Ok(value.clone());
        }
      }
      FeelType::Function(_, _) => {
        if let Value::FunctionDefinition(_, _) = value {
          return Ok(value.clone());
        }
      }
      FeelType::List(items_type) => {
        if let Value::List(items) = value {
          let mut result = vec![];
          for item in items.to_vec() {
            result.push(items_type.retrieve_value_with_type_checking(&item)?);
          }
          return Ok(Value::List(Values::new(result)));
        }
      }
      FeelType::Null => {
        if let Value::Null(_) = value {
          return Ok(VALUE_NULL);
        }
      }
      FeelType::Number => {
        if let Value::Number(_) = value {
          return Ok(value.clone());
        }
      }
      FeelType::Range(_) => {
        if let Value::Range(_, _, _, _) = value {
          return Ok(value.clone());
        }
      }
      FeelType::String => {
        if let Value::String(_) = value {
          return Ok(value.clone());
        }
      }
      FeelType::Time => {
        if let Value::Time(_) = value {
          return Ok(value.clone());
        }
      }
      FeelType::YearsAndMonthsDuration => {
        if let Value::YearsAndMonthsDuration(_) = value {
          return Ok(value.clone());
        }
      }
    }
    Err(invalid_value_for_retrieving_using_feel_type(&self.stringify(), &value.stringify()))
  }
  /// Returns `true` when this type is a simple `FEEL` type.
  pub fn is_simple_built_in_type(&self) -> bool {
    matches!(
      self,
      Self::Any
        | Self::Boolean
        | Self::Date
        | Self::DateTime
        | Self::DaysAndTimeDuration
        | Self::Null
        | Self::Number
        | Self::String
        | Self::Time
        | Self::YearsAndMonthsDuration
    )
  }
  /// Creates a `list` type with specified items' type.
  pub fn list(items_type: &FeelType) -> FeelType {
    FeelType::List(Box::new(items_type.clone()))
  }
  /// Creates a `range` type with specified elements' type.
  pub fn range(elements_type: &FeelType) -> FeelType {
    FeelType::Range(Box::new(elements_type.clone()))
  }
  /// Creates a `context` type with specified entries' types.
  pub fn context(entries_types: &[(&Name, &FeelType)]) -> FeelType {
    FeelType::Context(entries_types.iter().map(|(name, typ)| ((*name).clone(), (*typ).clone())).collect())
  }
  /// Creates a `function` type with specified parameter types and return type.
  pub fn function(parameter_types: &[FeelType], result_type: &FeelType) -> FeelType {
    FeelType::Function(parameter_types.iter().map(|typ| (*typ).clone()).collect(), Box::new((*result_type).clone()))
  }
  ///
  pub fn zip(&self, other: &FeelType) -> FeelType {
    if self == other {
      self.clone()
    } else {
      FeelType::Any
    }
  }
  ///
  pub fn is_equivalent(&self, other: &FeelType) -> bool {
    return match other {
      FeelType::Any => matches!(self, FeelType::Any),
      FeelType::Boolean => matches!(self, FeelType::Boolean),
      FeelType::Context(entries_other) => {
        if let FeelType::Context(entries_self) = self {
          if entries_self.keys().len() == entries_other.len() {
            for (name, type_self) in entries_self {
              if let Some(type_other) = entries_other.get(name) {
                if !type_self.is_equivalent(type_other) {
                  return false;
                }
              } else {
                return false;
              }
            }
            return true;
          }
        }
        false
      }
      FeelType::Date => matches!(self, FeelType::Date),
      FeelType::DateTime => matches!(self, FeelType::DateTime),
      FeelType::DaysAndTimeDuration => matches!(self, FeelType::DaysAndTimeDuration),
      FeelType::Function(params_other, result_other) => {
        if let FeelType::Function(params_self, result_self) = self {
          if params_self.len() == params_other.len() {
            for (i, param_self) in params_self.iter().enumerate() {
              if !param_self.is_equivalent(&params_other[i]) {
                return false;
              }
              if !result_self.is_equivalent(result_other) {
                return false;
              }
            }
            return true;
          }
        }
        false
      }
      FeelType::List(type_other) => {
        if let FeelType::List(type_self) = self {
          type_self.is_equivalent(type_other)
        } else {
          false
        }
      }
      FeelType::Null => matches!(self, FeelType::Null),
      FeelType::Number => matches!(self, FeelType::Number),
      FeelType::Range(type_other) => {
        if let FeelType::Range(type_self) = self {
          type_self.is_equivalent(type_other)
        } else {
          false
        }
      }
      FeelType::String => matches!(self, FeelType::String),
      FeelType::Time => matches!(self, FeelType::Time),
      FeelType::YearsAndMonthsDuration => matches!(self, FeelType::YearsAndMonthsDuration),
    };
  }
  ///
  pub fn is_conformant(&self, other: &FeelType) -> bool {
    if self.is_equivalent(other) {
      return true;
    }
    if let FeelType::Null = self {
      return true;
    }
    if let FeelType::Any = other {
      return true;
    }
    match other {
      FeelType::List(type_other) => {
        if let FeelType::List(type_self) = self {
          return type_self.is_conformant(type_other);
        }
      }
      FeelType::Context(entries_other) => {
        if let FeelType::Context(entries_self) = self {
          for (name, type_other) in entries_other {
            if let Some(type_self) = entries_self.get(name) {
              if !type_self.is_conformant(type_other) {
                return false;
              }
            } else {
              return false;
            }
          }
          return true;
        }
        return false;
      }
      FeelType::Function(parameters_other, result_other) => {
        if let FeelType::Function(parameters_self, result_self) = self {
          if parameters_self.len() == parameters_other.len() {
            for (i, parameter_other) in parameters_other.iter().enumerate() {
              if !parameter_other.is_conformant(&parameters_self[i]) {
                return false;
              }
              if !result_self.is_conformant(result_other) {
                return false;
              }
            }
            return true;
          }
        }
        return false;
      }
      FeelType::Range(type_other) => {
        if let FeelType::Range(type_self) = self {
          return type_self.is_conformant(type_other);
        }
      }
      _ => {}
    }
    false
  }
}

/// Definitions of errors raised by `types` module.
mod errors {
  use dmntk_common::DmntkError;

  /// Definition of errors raised in `types` module.
  #[derive(Debug, PartialEq)]
  pub enum TypesError {
    /// Error raised when the specified text is not a valid `FEEL` type name.
    InvalidFeelTypeName(String),
    /// Error raised when the specified value is not valid for retrieving with type checking with `FEEL` type.
    InvalidValueForRetrievingUsingFeelType(String, String),
  }

  impl From<TypesError> for DmntkError {
    /// Converts [TypesError] into [DmntkError].
    fn from(e: TypesError) -> Self {
      DmntkError::new("TypesError", &e.to_string())
    }
  }

  impl std::fmt::Display for TypesError {
    /// Implements [Display] trait for [TypesError].
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        TypesError::InvalidFeelTypeName(s) => {
          write!(f, "invalid FEEL type name: {}", s)
        }
        TypesError::InvalidValueForRetrievingUsingFeelType(s1, s2) => {
          write!(f, "invalid value for retrieving with type check, type = `{}`, value = `{}`", s1, s2)
        }
      }
    }
  }

  /// Creates an [InvalidFeelTypeName](TypesError::InvalidFeelTypeName) error.
  pub fn invalid_feel_type_name(s: &str) -> DmntkError {
    TypesError::InvalidFeelTypeName(s.to_owned()).into()
  }

  /// Creates an [InvalidValueForRetrievingWithTypeCheck](TypesError::InvalidValueForRetrievingWithTypeCheck) error.
  pub fn invalid_value_for_retrieving_using_feel_type(s1: &str, s2: &str) -> DmntkError {
    TypesError::InvalidValueForRetrievingUsingFeelType(s1.to_owned(), s2.to_owned()).into()
  }
}

#[cfg(test)]
mod tests {
  use crate::names::Name;
  use crate::types::is_built_in_type_name;
  use crate::types::FeelType::{self, Any, Number};
  use crate::values::Value;

  #[test]
  fn test_retrieve_value_with_type_checking() {
    let t_any = FeelType::Any;
    let t_boolean = FeelType::Boolean;
    let v_null = Value::Null(None);
    let v_boolean = Value::Boolean(true);
    let v_string = Value::String("hello".to_owned());
    assert_eq!(Value::Null(None), t_any.retrieve_value_with_type_checking(&v_null).unwrap());
    assert_eq!(Value::Boolean(true), t_any.retrieve_value_with_type_checking(&v_boolean).unwrap());
    assert!(t_boolean.retrieve_value_with_type_checking(&v_string).is_err());
    assert_eq!(
      r#"TypesError: invalid value for retrieving with type check, type = `boolean`, value = `"hello"`"#,
      format!("{}", t_boolean.retrieve_value_with_type_checking(&v_string).err().unwrap()).as_str()
    );
  }

  #[test]
  fn test_type_equivalence() {
    let name_a = Name::from("a");
    let name_b = Name::from("b");
    let name_c = Name::from("c");
    let t_any = FeelType::Any;
    let t_boolean = FeelType::Boolean;
    let t_date = FeelType::Date;
    let t_date_time = FeelType::DateTime;
    let t_days_and_time_duration = FeelType::DaysAndTimeDuration;
    let t_null = FeelType::Null;
    let t_number = FeelType::Number;
    let t_string = FeelType::String;
    let t_time = FeelType::Time;
    let t_years_and_months_duration = FeelType::YearsAndMonthsDuration;
    let t_list = FeelType::List(Box::new(FeelType::Boolean));
    let t_list_b = FeelType::List(Box::new(FeelType::Number));
    let t_context = FeelType::context(&[(&name_a, &t_number)]);
    let t_context_a_b = FeelType::context(&[(&name_a, &t_number), (&name_b, &t_boolean)]);
    let t_context_a_b_c = FeelType::context(&[(&name_a, &t_number), (&name_b, &t_boolean), (&name_c, &t_string)]);
    let t_function = FeelType::function(&[Number, Number], &t_number);
    let t_function_b = FeelType::function(&[Number, Number], &t_boolean);
    let t_function_c = FeelType::function(&[Number], &t_string);
    let t_range = FeelType::range(&t_number);
    let t_range_b = FeelType::range(&t_date);
    // any
    assert!(t_any.is_equivalent(&t_any));
    assert!(!t_any.is_equivalent(&t_boolean));
    assert!(!t_any.is_equivalent(&t_context));
    assert!(!t_any.is_equivalent(&t_date));
    assert!(!t_any.is_equivalent(&t_date_time));
    assert!(!t_any.is_equivalent(&t_days_and_time_duration));
    assert!(!t_boolean.is_equivalent(&t_function));
    assert!(!t_boolean.is_equivalent(&t_list));
    assert!(!t_boolean.is_equivalent(&t_null));
    assert!(!t_boolean.is_equivalent(&t_number));
    assert!(!t_boolean.is_equivalent(&t_range));
    assert!(!t_boolean.is_equivalent(&t_string));
    assert!(!t_boolean.is_equivalent(&t_time));
    assert!(!t_boolean.is_equivalent(&t_years_and_months_duration));
    // boolean
    assert!(t_boolean.is_equivalent(&t_boolean));
    assert!(!t_boolean.is_equivalent(&t_any));
    assert!(!t_boolean.is_equivalent(&t_context));
    assert!(!t_boolean.is_equivalent(&t_date));
    assert!(!t_boolean.is_equivalent(&t_date_time));
    assert!(!t_boolean.is_equivalent(&t_days_and_time_duration));
    assert!(!t_boolean.is_equivalent(&t_function));
    assert!(!t_boolean.is_equivalent(&t_list));
    assert!(!t_boolean.is_equivalent(&t_null));
    assert!(!t_boolean.is_equivalent(&t_number));
    assert!(!t_boolean.is_equivalent(&t_range));
    assert!(!t_boolean.is_equivalent(&t_string));
    assert!(!t_boolean.is_equivalent(&t_time));
    assert!(!t_boolean.is_equivalent(&t_years_and_months_duration));
    // context
    assert!(t_context.is_equivalent(&t_context));
    assert!(t_context_a_b.is_equivalent(&t_context_a_b));
    assert!(t_context_a_b_c.is_equivalent(&t_context_a_b_c));
    assert!(!t_context.is_equivalent(&t_context_a_b));
    assert!(!t_context_a_b.is_equivalent(&t_context_a_b_c));
    assert!(!t_context_a_b.is_equivalent(&t_context_a_b_c));
    assert!(!t_context.is_equivalent(&t_any));
    assert!(!t_context.is_equivalent(&t_boolean));
    assert!(!t_context.is_equivalent(&t_date));
    assert!(!t_context.is_equivalent(&t_date_time));
    assert!(!t_context.is_equivalent(&t_days_and_time_duration));
    assert!(!t_context.is_equivalent(&t_function));
    assert!(!t_context.is_equivalent(&t_list));
    assert!(!t_context.is_equivalent(&t_null));
    assert!(!t_context.is_equivalent(&t_number));
    assert!(!t_context.is_equivalent(&t_range));
    assert!(!t_context.is_equivalent(&t_string));
    assert!(!t_context.is_equivalent(&t_time));
    assert!(!t_context.is_equivalent(&t_years_and_months_duration));
    // date
    assert!(t_date.is_equivalent(&t_date));
    assert!(!t_date.is_equivalent(&t_any));
    assert!(!t_date.is_equivalent(&t_boolean));
    assert!(!t_date.is_equivalent(&t_context));
    assert!(!t_date.is_equivalent(&t_date_time));
    assert!(!t_date.is_equivalent(&t_days_and_time_duration));
    assert!(!t_date.is_equivalent(&t_function));
    assert!(!t_date.is_equivalent(&t_list));
    assert!(!t_date.is_equivalent(&t_null));
    assert!(!t_date.is_equivalent(&t_number));
    assert!(!t_date.is_equivalent(&t_range));
    assert!(!t_date.is_equivalent(&t_string));
    assert!(!t_date.is_equivalent(&t_time));
    assert!(!t_date.is_equivalent(&t_years_and_months_duration));
    // date and time
    assert!(t_date_time.is_equivalent(&t_date_time));
    assert!(!t_date_time.is_equivalent(&t_any));
    assert!(!t_date_time.is_equivalent(&t_boolean));
    assert!(!t_date_time.is_equivalent(&t_context));
    assert!(!t_date_time.is_equivalent(&t_date));
    assert!(!t_date_time.is_equivalent(&t_function));
    assert!(!t_date_time.is_equivalent(&t_list));
    assert!(!t_date_time.is_equivalent(&t_null));
    assert!(!t_date_time.is_equivalent(&t_number));
    assert!(!t_date_time.is_equivalent(&t_range));
    assert!(!t_date_time.is_equivalent(&t_string));
    assert!(!t_date_time.is_equivalent(&t_time));
    assert!(!t_date_time.is_equivalent(&t_years_and_months_duration));
    // days and time duration
    assert!(t_days_and_time_duration.is_equivalent(&t_days_and_time_duration));
    assert!(!t_days_and_time_duration.is_equivalent(&t_any));
    assert!(!t_days_and_time_duration.is_equivalent(&t_boolean));
    assert!(!t_days_and_time_duration.is_equivalent(&t_context));
    assert!(!t_days_and_time_duration.is_equivalent(&t_date));
    assert!(!t_days_and_time_duration.is_equivalent(&t_date_time));
    assert!(!t_days_and_time_duration.is_equivalent(&t_function));
    assert!(!t_days_and_time_duration.is_equivalent(&t_list));
    assert!(!t_days_and_time_duration.is_equivalent(&t_null));
    assert!(!t_days_and_time_duration.is_equivalent(&t_number));
    assert!(!t_days_and_time_duration.is_equivalent(&t_range));
    assert!(!t_days_and_time_duration.is_equivalent(&t_string));
    assert!(!t_days_and_time_duration.is_equivalent(&t_time));
    assert!(!t_days_and_time_duration.is_equivalent(&t_years_and_months_duration));
    // function
    assert!(t_function.is_equivalent(&t_function));
    assert!(t_function_b.is_equivalent(&t_function_b));
    assert!(t_function_c.is_equivalent(&t_function_c));
    assert!(!t_function.is_equivalent(&t_function_b));
    assert!(!t_function.is_equivalent(&t_function_c));
    assert!(!t_function_b.is_equivalent(&t_function));
    assert!(!t_function_b.is_equivalent(&t_function_c));
    assert!(!t_function_c.is_equivalent(&t_function));
    assert!(!t_function_c.is_equivalent(&t_function_b));
    assert!(!t_function.is_equivalent(&t_any));
    assert!(!t_function.is_equivalent(&t_boolean));
    assert!(!t_function.is_equivalent(&t_context));
    assert!(!t_function.is_equivalent(&t_date));
    assert!(!t_function.is_equivalent(&t_date_time));
    assert!(!t_function.is_equivalent(&t_days_and_time_duration));
    assert!(!t_function.is_equivalent(&t_list));
    assert!(!t_function.is_equivalent(&t_null));
    assert!(!t_function.is_equivalent(&t_range));
    assert!(!t_function.is_equivalent(&t_string));
    assert!(!t_function.is_equivalent(&t_time));
    assert!(!t_function.is_equivalent(&t_years_and_months_duration));
    // list
    assert!(t_list_b.is_equivalent(&t_list_b));
    assert!(!t_list_b.is_equivalent(&t_list));
    assert!(!t_list.is_equivalent(&t_any));
    assert!(!t_list.is_equivalent(&t_boolean));
    assert!(!t_list.is_equivalent(&t_context));
    assert!(!t_list.is_equivalent(&t_date));
    assert!(!t_list.is_equivalent(&t_date_time));
    assert!(!t_list.is_equivalent(&t_days_and_time_duration));
    assert!(!t_list.is_equivalent(&t_function));
    assert!(!t_list.is_equivalent(&t_null));
    assert!(!t_list.is_equivalent(&t_range));
    assert!(!t_list.is_equivalent(&t_string));
    assert!(!t_list.is_equivalent(&t_time));
    assert!(!t_list.is_equivalent(&t_years_and_months_duration));
    // null
    assert!(t_null.is_equivalent(&t_null));
    assert!(!t_null.is_equivalent(&t_any));
    assert!(!t_null.is_equivalent(&t_boolean));
    assert!(!t_null.is_equivalent(&t_context));
    assert!(!t_null.is_equivalent(&t_date));
    assert!(!t_null.is_equivalent(&t_date_time));
    assert!(!t_null.is_equivalent(&t_days_and_time_duration));
    assert!(!t_null.is_equivalent(&t_function));
    assert!(!t_null.is_equivalent(&t_list));
    assert!(!t_null.is_equivalent(&t_range));
    assert!(!t_null.is_equivalent(&t_string));
    assert!(!t_null.is_equivalent(&t_time));
    assert!(!t_null.is_equivalent(&t_years_and_months_duration));
    // number
    assert!(t_number.is_equivalent(&t_number));
    assert!(!t_number.is_equivalent(&t_any));
    assert!(!t_number.is_equivalent(&t_boolean));
    assert!(!t_number.is_equivalent(&t_context));
    assert!(!t_number.is_equivalent(&t_date));
    assert!(!t_number.is_equivalent(&t_date_time));
    assert!(!t_number.is_equivalent(&t_days_and_time_duration));
    assert!(!t_number.is_equivalent(&t_function));
    assert!(!t_number.is_equivalent(&t_list));
    assert!(!t_number.is_equivalent(&t_null));
    assert!(!t_number.is_equivalent(&t_range));
    assert!(!t_number.is_equivalent(&t_string));
    assert!(!t_number.is_equivalent(&t_time));
    assert!(!t_number.is_equivalent(&t_years_and_months_duration));
    // range
    assert!(t_range.is_equivalent(&t_range));
    assert!(t_range_b.is_equivalent(&t_range_b));
    assert!(!t_range.is_equivalent(&t_any));
    assert!(!t_range.is_equivalent(&t_boolean));
    assert!(!t_range.is_equivalent(&t_context));
    assert!(!t_range.is_equivalent(&t_date));
    assert!(!t_range.is_equivalent(&t_date_time));
    assert!(!t_range.is_equivalent(&t_days_and_time_duration));
    assert!(!t_range.is_equivalent(&t_function));
    assert!(!t_range.is_equivalent(&t_list));
    assert!(!t_range.is_equivalent(&t_null));
    assert!(!t_range.is_equivalent(&t_number));
    assert!(!t_range.is_equivalent(&t_string));
    assert!(!t_range.is_equivalent(&t_time));
    assert!(!t_range.is_equivalent(&t_years_and_months_duration));
    // string
    assert!(t_string.is_equivalent(&t_string));
    assert!(!t_string.is_equivalent(&t_any));
    assert!(!t_string.is_equivalent(&t_boolean));
    assert!(!t_string.is_equivalent(&t_context));
    assert!(!t_string.is_equivalent(&t_date));
    assert!(!t_string.is_equivalent(&t_date_time));
    assert!(!t_string.is_equivalent(&t_days_and_time_duration));
    assert!(!t_string.is_equivalent(&t_function));
    assert!(!t_string.is_equivalent(&t_list));
    assert!(!t_string.is_equivalent(&t_null));
    assert!(!t_string.is_equivalent(&t_number));
    assert!(!t_string.is_equivalent(&t_range));
    assert!(!t_string.is_equivalent(&t_time));
    assert!(!t_string.is_equivalent(&t_years_and_months_duration));
    // time
    assert!(t_time.is_equivalent(&t_time));
    assert!(!t_time.is_equivalent(&t_any));
    assert!(!t_time.is_equivalent(&t_boolean));
    assert!(!t_time.is_equivalent(&t_context));
    assert!(!t_time.is_equivalent(&t_date));
    assert!(!t_time.is_equivalent(&t_date_time));
    assert!(!t_time.is_equivalent(&t_days_and_time_duration));
    assert!(!t_time.is_equivalent(&t_function));
    assert!(!t_time.is_equivalent(&t_list));
    assert!(!t_time.is_equivalent(&t_null));
    assert!(!t_time.is_equivalent(&t_number));
    assert!(!t_time.is_equivalent(&t_range));
    assert!(!t_time.is_equivalent(&t_string));
    assert!(!t_time.is_equivalent(&t_years_and_months_duration));
    // years and months duration
    assert!(t_years_and_months_duration.is_equivalent(&t_years_and_months_duration));
    assert!(!t_years_and_months_duration.is_equivalent(&t_any));
    assert!(!t_years_and_months_duration.is_equivalent(&t_boolean));
    assert!(!t_years_and_months_duration.is_equivalent(&t_context));
    assert!(!t_years_and_months_duration.is_equivalent(&t_date));
    assert!(!t_years_and_months_duration.is_equivalent(&t_date_time));
    assert!(!t_years_and_months_duration.is_equivalent(&t_days_and_time_duration));
    assert!(!t_years_and_months_duration.is_equivalent(&t_function));
    assert!(!t_years_and_months_duration.is_equivalent(&t_list));
    assert!(!t_years_and_months_duration.is_equivalent(&t_null));
    assert!(!t_years_and_months_duration.is_equivalent(&t_number));
    assert!(!t_years_and_months_duration.is_equivalent(&t_range));
    assert!(!t_years_and_months_duration.is_equivalent(&t_string));
    assert!(!t_years_and_months_duration.is_equivalent(&t_time));
  }

  #[test]
  fn test_type_conformance() {
    let name_a = Name::from("a");
    let name_b = Name::from("b");
    let name_c = Name::from("c");
    let t_any = FeelType::Any;
    let t_boolean = FeelType::Boolean;
    let t_date = FeelType::Date;
    let t_date_time = FeelType::DateTime;
    let t_days_and_time_duration = FeelType::DaysAndTimeDuration;
    let t_null = FeelType::Null;
    let t_number = FeelType::Number;
    let t_string = FeelType::String;
    let t_time = FeelType::Time;
    let t_years_and_months_duration = FeelType::YearsAndMonthsDuration;
    let t_list = FeelType::List(Box::new(FeelType::Boolean));
    let t_list_b = FeelType::List(Box::new(FeelType::Number));
    let t_context = FeelType::context(&[(&name_a, &t_number)]);
    let t_context_a_b = FeelType::context(&[(&name_a, &t_number), (&name_b, &t_boolean)]);
    let t_context_a_b_c = FeelType::context(&[(&name_a, &t_number), (&name_b, &t_boolean), (&name_c, &t_string)]);
    let t_function = FeelType::function(&[Number, Number], &t_number);
    let t_function_b = FeelType::function(&[Number, Number], &t_boolean);
    let t_function_c = FeelType::function(&[Number], &t_string);
    let t_function_d = FeelType::function(&[Any], &t_string);
    let t_range = FeelType::range(&t_number);
    let t_range_b = FeelType::range(&t_date);
    // any
    assert!(t_any.is_conformant(&t_any));
    assert!(!t_any.is_conformant(&t_boolean));
    assert!(!t_any.is_conformant(&t_context));
    assert!(!t_any.is_conformant(&t_date));
    assert!(!t_any.is_conformant(&t_date_time));
    assert!(!t_any.is_conformant(&t_days_and_time_duration));
    assert!(!t_boolean.is_conformant(&t_function));
    assert!(!t_boolean.is_conformant(&t_list));
    assert!(!t_boolean.is_conformant(&t_null));
    assert!(!t_boolean.is_conformant(&t_number));
    assert!(!t_boolean.is_conformant(&t_range));
    assert!(!t_boolean.is_conformant(&t_string));
    assert!(!t_boolean.is_conformant(&t_time));
    assert!(!t_boolean.is_conformant(&t_years_and_months_duration));
    // boolean
    assert!(t_boolean.is_conformant(&t_boolean));
    assert!(t_boolean.is_conformant(&t_any));
    assert!(!t_boolean.is_conformant(&t_context));
    assert!(!t_boolean.is_conformant(&t_date));
    assert!(!t_boolean.is_conformant(&t_date_time));
    assert!(!t_boolean.is_conformant(&t_days_and_time_duration));
    assert!(!t_boolean.is_conformant(&t_function));
    assert!(!t_boolean.is_conformant(&t_list));
    assert!(!t_boolean.is_conformant(&t_null));
    assert!(!t_boolean.is_conformant(&t_number));
    assert!(!t_boolean.is_conformant(&t_range));
    assert!(!t_boolean.is_conformant(&t_string));
    assert!(!t_boolean.is_conformant(&t_time));
    assert!(!t_boolean.is_conformant(&t_years_and_months_duration));
    // context
    assert!(t_context.is_conformant(&t_context));
    assert!(t_context_a_b.is_conformant(&t_context_a_b));
    assert!(t_context_a_b.is_conformant(&t_context));
    assert!(t_context_a_b_c.is_conformant(&t_context_a_b_c));
    assert!(t_context_a_b_c.is_conformant(&t_context_a_b));
    assert!(!t_context.is_conformant(&t_context_a_b));
    assert!(!t_context_a_b.is_conformant(&t_context_a_b_c));
    assert!(!t_context_a_b.is_conformant(&t_context_a_b_c));
    assert!(t_context.is_conformant(&t_any));
    assert!(!t_context.is_conformant(&t_boolean));
    assert!(!t_context.is_conformant(&t_date));
    assert!(!t_context.is_conformant(&t_date_time));
    assert!(!t_context.is_conformant(&t_days_and_time_duration));
    assert!(!t_context.is_conformant(&t_function));
    assert!(!t_context.is_conformant(&t_list));
    assert!(!t_context.is_conformant(&t_null));
    assert!(!t_context.is_conformant(&t_number));
    assert!(!t_context.is_conformant(&t_range));
    assert!(!t_context.is_conformant(&t_string));
    assert!(!t_context.is_conformant(&t_time));
    assert!(!t_context.is_conformant(&t_years_and_months_duration));
    // date
    assert!(t_date.is_conformant(&t_date));
    assert!(t_date.is_conformant(&t_any));
    assert!(!t_date.is_conformant(&t_boolean));
    assert!(!t_date.is_conformant(&t_context));
    assert!(!t_date.is_conformant(&t_date_time));
    assert!(!t_date.is_conformant(&t_days_and_time_duration));
    assert!(!t_date.is_conformant(&t_function));
    assert!(!t_date.is_conformant(&t_list));
    assert!(!t_date.is_conformant(&t_null));
    assert!(!t_date.is_conformant(&t_number));
    assert!(!t_date.is_conformant(&t_range));
    assert!(!t_date.is_conformant(&t_string));
    assert!(!t_date.is_conformant(&t_time));
    assert!(!t_date.is_conformant(&t_years_and_months_duration));
    // date and time
    assert!(t_date_time.is_conformant(&t_date_time));
    assert!(t_date_time.is_conformant(&t_any));
    assert!(!t_date_time.is_conformant(&t_boolean));
    assert!(!t_date_time.is_conformant(&t_context));
    assert!(!t_date_time.is_conformant(&t_date));
    assert!(!t_date_time.is_conformant(&t_function));
    assert!(!t_date_time.is_conformant(&t_list));
    assert!(!t_date_time.is_conformant(&t_null));
    assert!(!t_date_time.is_conformant(&t_number));
    assert!(!t_date_time.is_conformant(&t_range));
    assert!(!t_date_time.is_conformant(&t_string));
    assert!(!t_date_time.is_conformant(&t_time));
    assert!(!t_date_time.is_conformant(&t_years_and_months_duration));
    // days and time duration
    assert!(t_days_and_time_duration.is_conformant(&t_days_and_time_duration));
    assert!(t_days_and_time_duration.is_conformant(&t_any));
    assert!(!t_days_and_time_duration.is_conformant(&t_boolean));
    assert!(!t_days_and_time_duration.is_conformant(&t_context));
    assert!(!t_days_and_time_duration.is_conformant(&t_date));
    assert!(!t_days_and_time_duration.is_conformant(&t_date_time));
    assert!(!t_days_and_time_duration.is_conformant(&t_function));
    assert!(!t_days_and_time_duration.is_conformant(&t_list));
    assert!(!t_days_and_time_duration.is_conformant(&t_null));
    assert!(!t_days_and_time_duration.is_conformant(&t_number));
    assert!(!t_days_and_time_duration.is_conformant(&t_range));
    assert!(!t_days_and_time_duration.is_conformant(&t_string));
    assert!(!t_days_and_time_duration.is_conformant(&t_time));
    assert!(!t_days_and_time_duration.is_conformant(&t_years_and_months_duration));
    // function
    assert!(t_function.is_conformant(&t_function));
    assert!(t_function_b.is_conformant(&t_function_b));
    assert!(t_function_c.is_conformant(&t_function_c));
    assert!(!t_function.is_conformant(&t_function_b));
    assert!(!t_function.is_conformant(&t_function_c));
    assert!(!t_function_b.is_conformant(&t_function));
    assert!(!t_function_b.is_conformant(&t_function_c));
    assert!(!t_function_c.is_conformant(&t_function));
    assert!(!t_function_c.is_conformant(&t_function_b));
    assert!(t_function_d.is_conformant(&t_function_c));
    assert!(t_function.is_conformant(&t_any));
    assert!(!t_function.is_conformant(&t_boolean));
    assert!(!t_function.is_conformant(&t_context));
    assert!(!t_function.is_conformant(&t_date));
    assert!(!t_function.is_conformant(&t_date_time));
    assert!(!t_function.is_conformant(&t_days_and_time_duration));
    assert!(!t_function.is_conformant(&t_list));
    assert!(!t_function.is_conformant(&t_null));
    assert!(!t_function.is_conformant(&t_range));
    assert!(!t_function.is_conformant(&t_string));
    assert!(!t_function.is_conformant(&t_time));
    assert!(!t_function.is_conformant(&t_years_and_months_duration));
    // list
    assert!(t_list_b.is_conformant(&t_list_b));
    assert!(!t_list_b.is_conformant(&t_list));
    assert!(t_list.is_conformant(&t_any));
    assert!(!t_list.is_conformant(&t_boolean));
    assert!(!t_list.is_conformant(&t_context));
    assert!(!t_list.is_conformant(&t_date));
    assert!(!t_list.is_conformant(&t_date_time));
    assert!(!t_list.is_conformant(&t_days_and_time_duration));
    assert!(!t_list.is_conformant(&t_function));
    assert!(!t_list.is_conformant(&t_null));
    assert!(!t_list.is_conformant(&t_range));
    assert!(!t_list.is_conformant(&t_string));
    assert!(!t_list.is_conformant(&t_time));
    assert!(!t_list.is_conformant(&t_years_and_months_duration));
    // null
    assert!(t_null.is_conformant(&t_null));
    assert!(t_null.is_conformant(&t_any));
    assert!(t_null.is_conformant(&t_boolean));
    assert!(t_null.is_conformant(&t_context));
    assert!(t_null.is_conformant(&t_date));
    assert!(t_null.is_conformant(&t_date_time));
    assert!(t_null.is_conformant(&t_days_and_time_duration));
    assert!(t_null.is_conformant(&t_function));
    assert!(t_null.is_conformant(&t_list));
    assert!(t_null.is_conformant(&t_range));
    assert!(t_null.is_conformant(&t_string));
    assert!(t_null.is_conformant(&t_time));
    assert!(t_null.is_conformant(&t_years_and_months_duration));
    // number
    assert!(t_number.is_conformant(&t_number));
    assert!(t_number.is_conformant(&t_any));
    assert!(!t_number.is_conformant(&t_boolean));
    assert!(!t_number.is_conformant(&t_context));
    assert!(!t_number.is_conformant(&t_date));
    assert!(!t_number.is_conformant(&t_date_time));
    assert!(!t_number.is_conformant(&t_days_and_time_duration));
    assert!(!t_number.is_conformant(&t_function));
    assert!(!t_number.is_conformant(&t_list));
    assert!(!t_number.is_conformant(&t_null));
    assert!(!t_number.is_conformant(&t_range));
    assert!(!t_number.is_conformant(&t_string));
    assert!(!t_number.is_conformant(&t_time));
    assert!(!t_number.is_conformant(&t_years_and_months_duration));
    // range
    assert!(t_range.is_conformant(&t_range));
    assert!(t_range_b.is_conformant(&t_range_b));
    assert!(t_range.is_conformant(&t_any));
    assert!(!t_range.is_conformant(&t_boolean));
    assert!(!t_range.is_conformant(&t_context));
    assert!(!t_range.is_conformant(&t_date));
    assert!(!t_range.is_conformant(&t_date_time));
    assert!(!t_range.is_conformant(&t_days_and_time_duration));
    assert!(!t_range.is_conformant(&t_function));
    assert!(!t_range.is_conformant(&t_list));
    assert!(!t_range.is_conformant(&t_null));
    assert!(!t_range.is_conformant(&t_number));
    assert!(!t_range.is_conformant(&t_string));
    assert!(!t_range.is_conformant(&t_time));
    assert!(!t_range.is_conformant(&t_years_and_months_duration));
    // string
    assert!(t_string.is_conformant(&t_string));
    assert!(t_string.is_conformant(&t_any));
    assert!(!t_string.is_conformant(&t_boolean));
    assert!(!t_string.is_conformant(&t_context));
    assert!(!t_string.is_conformant(&t_date));
    assert!(!t_string.is_conformant(&t_date_time));
    assert!(!t_string.is_conformant(&t_days_and_time_duration));
    assert!(!t_string.is_conformant(&t_function));
    assert!(!t_string.is_conformant(&t_list));
    assert!(!t_string.is_conformant(&t_null));
    assert!(!t_string.is_conformant(&t_number));
    assert!(!t_string.is_conformant(&t_range));
    assert!(!t_string.is_conformant(&t_time));
    assert!(!t_string.is_conformant(&t_years_and_months_duration));
    // time
    assert!(t_time.is_conformant(&t_time));
    assert!(t_time.is_conformant(&t_any));
    assert!(!t_time.is_conformant(&t_boolean));
    assert!(!t_time.is_conformant(&t_context));
    assert!(!t_time.is_conformant(&t_date));
    assert!(!t_time.is_conformant(&t_date_time));
    assert!(!t_time.is_conformant(&t_days_and_time_duration));
    assert!(!t_time.is_conformant(&t_function));
    assert!(!t_time.is_conformant(&t_list));
    assert!(!t_time.is_conformant(&t_null));
    assert!(!t_time.is_conformant(&t_number));
    assert!(!t_time.is_conformant(&t_range));
    assert!(!t_time.is_conformant(&t_string));
    assert!(!t_time.is_conformant(&t_years_and_months_duration));
    // years and months duration
    assert!(t_years_and_months_duration.is_conformant(&t_years_and_months_duration));
    assert!(t_years_and_months_duration.is_conformant(&t_any));
    assert!(!t_years_and_months_duration.is_conformant(&t_boolean));
    assert!(!t_years_and_months_duration.is_conformant(&t_context));
    assert!(!t_years_and_months_duration.is_conformant(&t_date));
    assert!(!t_years_and_months_duration.is_conformant(&t_date_time));
    assert!(!t_years_and_months_duration.is_conformant(&t_days_and_time_duration));
    assert!(!t_years_and_months_duration.is_conformant(&t_function));
    assert!(!t_years_and_months_duration.is_conformant(&t_list));
    assert!(!t_years_and_months_duration.is_conformant(&t_null));
    assert!(!t_years_and_months_duration.is_conformant(&t_number));
    assert!(!t_years_and_months_duration.is_conformant(&t_range));
    assert!(!t_years_and_months_duration.is_conformant(&t_string));
    assert!(!t_years_and_months_duration.is_conformant(&t_time));
  }

  #[test]
  fn test_is_built_in_type_name() {
    assert!(is_built_in_type_name("Any"));
    assert!(is_built_in_type_name("Null"));
    assert!(is_built_in_type_name("number"));
    assert!(is_built_in_type_name("string"));
    assert!(is_built_in_type_name("boolean"));
    assert!(is_built_in_type_name("days and time duration"));
    assert!(is_built_in_type_name("years and months duration"));
    assert!(is_built_in_type_name("date"));
    assert!(is_built_in_type_name("time"));
    assert!(is_built_in_type_name("date and time"));
    assert!(!is_built_in_type_name("list"));
    assert!(!is_built_in_type_name("range"));
    assert!(!is_built_in_type_name("context"));
    assert!(!is_built_in_type_name("function"));
  }
}
