/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! `FEEL` days and time durations.

use {self::errors::*, dmntk_common::DmntkError, regex::Regex, std::convert::TryFrom};

/// Regular expression pattern for parsing days and time duration.
const REGEX_DAYS_AND_TIME: &str =
  r#"^(?P<sign>-)?P((?P<days>[0-9]+)D)?(T((?P<hours>[0-9]+)H)?((?P<minutes>[0-9]+)M)?((?P<seconds>[0-9]+)(?P<fractional>\.[0-9]*)?S)?)?$"#;

lazy_static! {
  static ref RE_DAYS_AND_TIME: Regex = Regex::new(REGEX_DAYS_AND_TIME).unwrap();
}

/// Days and time duration in `FEEL`.
/// Holds the number of seconds and number of nanoseconds of the duration.
#[derive(Debug, Clone, PartialEq)]
pub struct FeelDaysAndTimeDuration(i64, u64);

/// Number of seconds in a day.
const SECONDS_IN_DAY: i64 = 86_400;

/// Number of seconds in an hour.
const SECONDS_IN_HOUR: i64 = 3_600;

/// Number of seconds in a minute.
const SECONDS_IN_MINUTE: i64 = 60;

impl FeelDaysAndTimeDuration {
  /// Creates a new days and time duration from seconds and nanoseconds.
  pub fn new(sec: i64, nano: u64) -> Self {
    Self(sec, nano)
  }
  /// Returns the number of days in this duration.
  pub fn days(&self) -> i64 {
    self.0 / SECONDS_IN_DAY
  }
  /// Returns the number of hours in this duration.
  pub fn hours(&self) -> i64 {
    (self.0 % SECONDS_IN_DAY) / SECONDS_IN_HOUR
  }
  /// Returns the number of minutes in this duration.
  pub fn minutes(&self) -> i64 {
    (self.0 % SECONDS_IN_DAY % SECONDS_IN_HOUR) / SECONDS_IN_MINUTE
  }
  /// Returns the number of seconds in this duration.
  pub fn seconds(&self) -> i64 {
    (self.0 % SECONDS_IN_DAY % SECONDS_IN_HOUR) % SECONDS_IN_MINUTE
  }
  ///
  pub fn as_seconds(&self) -> i64 {
    self.0
  }
}

impl std::ops::Add<FeelDaysAndTimeDuration> for FeelDaysAndTimeDuration {
  type Output = FeelDaysAndTimeDuration;
  /// Returns the sum of durations.
  fn add(self, rhs: FeelDaysAndTimeDuration) -> FeelDaysAndTimeDuration {
    let nanos = self.1 + rhs.1;
    FeelDaysAndTimeDuration(self.0 + rhs.0 + (nanos / super::NANOS_IN_SECOND) as i64, nanos % super::NANOS_IN_SECOND)
  }
}

impl std::ops::Sub<FeelDaysAndTimeDuration> for FeelDaysAndTimeDuration {
  type Output = FeelDaysAndTimeDuration;
  /// Returns the subtraction of durations.
  fn sub(self, rhs: FeelDaysAndTimeDuration) -> FeelDaysAndTimeDuration {
    let nanos = self.1 as i64 - rhs.1 as i64;
    if nanos < 0 {
      FeelDaysAndTimeDuration(self.0 - 1_i64 - rhs.0, (nanos + super::NANOS_IN_SECOND as i64) as u64)
    } else {
      FeelDaysAndTimeDuration(self.0 - rhs.0, nanos as u64)
    }
  }
}

impl std::cmp::PartialOrd for FeelDaysAndTimeDuration {
  fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
    match self.0.partial_cmp(&other.0) {
      Some(std::cmp::Ordering::Equal) => self.1.partial_cmp(&other.1),
      other => other,
    }
  }

  fn lt(&self, other: &Self) -> bool {
    match self.0.cmp(&other.0) {
      std::cmp::Ordering::Less => true,
      std::cmp::Ordering::Equal => self.1 < other.1,
      _ => false,
    }
  }

  fn le(&self, other: &Self) -> bool {
    match self.0.cmp(&other.0) {
      std::cmp::Ordering::Less => true,
      std::cmp::Ordering::Equal => self.1 <= other.1,
      _ => false,
    }
  }

  fn gt(&self, other: &Self) -> bool {
    match self.0.cmp(&other.0) {
      std::cmp::Ordering::Greater => true,
      std::cmp::Ordering::Equal => self.1 > other.1,
      _ => false,
    }
  }

  fn ge(&self, other: &Self) -> bool {
    match self.0.cmp(&other.0) {
      std::cmp::Ordering::Greater => true,
      std::cmp::Ordering::Equal => self.1 >= other.1,
      _ => false,
    }
  }
}

impl From<FeelDaysAndTimeDuration> for String {
  /// Converts the days and time duration into textual form.
  fn from(value: FeelDaysAndTimeDuration) -> Self {
    let sign = if value.0 < 0 { "-" } else { "" };
    let mut second = value.0.abs();
    let day = second / SECONDS_IN_DAY;
    second -= day * SECONDS_IN_DAY;
    let hour = second / SECONDS_IN_HOUR;
    second -= hour * SECONDS_IN_HOUR;
    let minute = second / SECONDS_IN_MINUTE;
    second -= minute * SECONDS_IN_MINUTE;
    let nano = super::nanoseconds_to_string(value.1);
    match (day > 0, hour > 0, minute > 0, second > 0, value.1 > 0) {
      (false, false, false, false, false) => "PT0S".to_string(),
      (false, false, false, true, false) => format!("{}PT{}S", sign, second),
      (false, false, true, false, false) => format!("{}PT{}M", sign, minute),
      (false, false, true, true, false) => format!("{}PT{}M{}S", sign, minute, second),
      (false, true, false, false, false) => format!("{}PT{}H", sign, hour),
      (false, true, false, true, false) => format!("{}PT{}H{}S", sign, hour, second),
      (false, true, true, false, false) => format!("{}PT{}H{}M", sign, hour, minute),
      (false, true, true, true, false) => format!("{}PT{}H{}M{}S", sign, hour, minute, second),
      (true, false, false, false, false) => format!("{}P{}D", sign, day),
      (true, false, false, true, false) => format!("{}P{}DT{}S", sign, day, second),
      (true, false, true, false, false) => format!("{}P{}DT{}M", sign, day, minute),
      (true, false, true, true, false) => format!("{}P{}DT{}M{}S", sign, day, minute, second),
      (true, true, false, false, false) => format!("{}P{}DT{}H", sign, day, hour),
      (true, true, false, true, false) => format!("{}P{}DT{}H{}S", sign, day, hour, second),
      (true, true, true, false, false) => format!("{}P{}DT{}H{}M", sign, day, hour, minute),
      (true, true, true, true, false) => format!("{}P{}DT{}H{}M{}S", sign, day, hour, minute, second),
      (false, false, false, false, true) => format!("{}PT0.{}S", sign, nano),
      (false, false, false, true, true) => format!("{}PT{}.{}S", sign, second, nano),
      (false, false, true, false, true) => format!("{}PT{}M0.{}S", sign, minute, nano),
      (false, false, true, true, true) => format!("{}PT{}M{}.{}S", sign, minute, second, nano),
      (false, true, false, false, true) => format!("{}PT{}H0.{}S", sign, hour, nano),
      (false, true, false, true, true) => format!("{}PT{}H{}.{}S", sign, hour, second, nano),
      (false, true, true, false, true) => format!("{}PT{}H{}M0.{}S", sign, hour, minute, nano),
      (false, true, true, true, true) => format!("{}PT{}H{}M{}.{}S", sign, hour, minute, second, nano),
      (true, false, false, false, true) => format!("{}P{}DT0.{}S", sign, day, nano),
      (true, false, false, true, true) => format!("{}P{}DT{}.{}S", sign, day, second, nano),
      (true, false, true, false, true) => format!("{}P{}DT{}M0.{}S", sign, day, minute, nano),
      (true, false, true, true, true) => format!("{}P{}DT{}M{}.{}S", sign, day, minute, second, nano),
      (true, true, false, false, true) => format!("{}P{}DT{}H0.{}S", sign, day, hour, nano),
      (true, true, false, true, true) => format!("{}P{}DT{}H{}.{}S", sign, day, hour, second, nano),
      (true, true, true, false, true) => format!("{}P{}DT{}H{}M0.{}S", sign, day, hour, minute, nano),
      (true, true, true, true, true) => format!("{}P{}DT{}H{}M{}.{}S", sign, day, hour, minute, second, nano),
    }
  }
}

impl TryFrom<&str> for FeelDaysAndTimeDuration {
  type Error = DmntkError;
  /// Converts a text form of the days and time duration into [FeelDaysAndTimeDuration] struct.
  fn try_from(value: &str) -> Result<Self, Self::Error> {
    if let Some(captures) = RE_DAYS_AND_TIME.captures(value) {
      let mut is_valid = false;
      let mut sec = 0_i64;
      if let Some(days_match) = captures.name("days") {
        if let Ok(days) = days_match.as_str().parse::<u64>() {
          sec += (days as i64) * SECONDS_IN_DAY;
          is_valid = true;
        }
      }
      if let Some(hours_match) = captures.name("hours") {
        if let Ok(hours) = hours_match.as_str().parse::<u64>() {
          sec += (hours as i64) * SECONDS_IN_HOUR;
          is_valid = true;
        }
      }
      if let Some(minutes_match) = captures.name("minutes") {
        if let Ok(minutes) = minutes_match.as_str().parse::<u64>() {
          sec += (minutes as i64) * SECONDS_IN_MINUTE;
          is_valid = true;
        }
      }
      if let Some(seconds_match) = captures.name("seconds") {
        if let Ok(seconds) = seconds_match.as_str().parse::<u64>() {
          sec += seconds as i64;
          is_valid = true;
        }
      }
      let mut nano = 0_u64;
      if let Some(fractional_match) = captures.name("fractional") {
        if let Ok(fractional) = fractional_match.as_str().parse::<f64>() {
          nano = (fractional * 1e9).trunc() as u64;
          is_valid = true;
        }
      }
      if captures.name("sign").is_some() {
        sec = -sec;
      }
      if is_valid {
        return Ok(FeelDaysAndTimeDuration(sec, nano));
      }
    }
    Err(invalid_date_and_time_duration_literal(value.to_string()))
  }
}

/// Definitions of date and time duration errors.
pub mod errors {
  use dmntk_common::DmntkError;

  /// Date and time duration errors.
  #[derive(Debug, PartialEq)]
  enum DtDurationError {
    InvalidDateAndTimeDurationLiteral(String),
  }

  //TODO https://github.com/EngosSoftware/dmntk/issues/1
  impl From<DtDurationError> for DmntkError {
    fn from(e: DtDurationError) -> Self {
      DmntkError::new("DtDurationError", &format!("{}", e))
    }
  }

  impl std::fmt::Display for DtDurationError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        DtDurationError::InvalidDateAndTimeDurationLiteral(literal) => {
          write!(f, "invalid date and time duration literal: {}", literal)
        }
      }
    }
  }

  pub fn invalid_date_and_time_duration_literal(literal: String) -> DmntkError {
    DtDurationError::InvalidDateAndTimeDurationLiteral(literal).into()
  }
}

#[cfg(test)]
mod tests {
  use {
    super::super::nanoseconds_to_string,
    super::FeelDaysAndTimeDuration,
    std::{
      cmp::Ordering,
      convert::{TryFrom, TryInto},
    },
  };

  /// Utility function for testing days and time durations equality.
  fn equals(sec: i64, nano: u64, text: &str) {
    let expected = FeelDaysAndTimeDuration(sec, nano);
    let actual = FeelDaysAndTimeDuration::try_from(text).unwrap();
    assert_eq!(expected, actual);
  }

  /// Utility function for testing invalid days and time durations.
  fn invalid(text: &str) {
    let actual = FeelDaysAndTimeDuration::try_from(text);
    assert!(actual.is_err());
  }

  /// Utility function for testing equality of textual forms of days and time durations.
  fn equals_str(expected: &str, sec: i64, nano: u64) {
    let actual: String = FeelDaysAndTimeDuration(sec, nano).try_into().unwrap();
    assert_eq!(expected, actual);
  }

  #[test]
  fn test_parsing_from_string_should_pass() {
    equals(86_400, 0, "P1D");
    equals(-86_400, 0, "-P1D");
    equals(97200, 0, "P1DT3H");
    equals(-97200, 0, "-P1DT3H");
    equals(97980, 0, "P1DT3H13M");
    equals(-97980, 0, "-P1DT3H13M");
    equals(98023, 0, "P1DT3H13M43S");
    equals(-98023, 0, "-P1DT3H13M43S");
    equals(10800, 0, "PT3H");
    equals(-10800, 0, "-PT3H");
    equals(12960, 0, "PT3H36M");
    equals(-12960, 0, "-PT3H36M");
    equals(12982, 0, "PT3H36M22S");
    equals(-12982, 0, "-PT3H36M22S");
    equals(3540, 0, "PT59M");
    equals(-3540, 0, "-PT59M");
    equals(3558, 0, "PT59M18S");
    equals(-3558, 0, "-PT59M18S");
    equals(31, 0, "PT31S");
    equals(-31, 0, "-PT31S");
    equals(87180, 0, "P1DT13M");
    equals(-87180, 0, "-P1DT13M");
    equals(97243, 0, "P1DT3H43S");
    equals(-97243, 0, "-P1DT3H43S");
    equals(0, 999_000_000, "PT0.999S");
    equals(0, 0, "PT0.S");
    equals(58, 123_123_123, "PT58.123123123S");
    equals(999, 999_999_999, "PT999.999999999999S");
  }

  #[test]
  fn test_parsing_from_string_should_fail() {
    invalid("P");
    invalid("-P");
    invalid("PT");
    invalid("-PT");
    invalid("T");
    invalid("-T");
    invalid("P11");
    invalid("-P11");
    invalid("PT1S1M");
    invalid("-PT1S1M");
    invalid("PT2M3H12S");
    invalid("-PT2M3H12S");
  }

  #[test]
  fn test_converting_to_string_should_pass() {
    equals_str("PT0.999S", 0, 999_000_000);
    equals_str("PT0S", 0, 0);
    equals_str("PT1S", 1, 0);
    equals_str("-PT1S", -1, 0);
    equals_str("PT1.123S", 1, 123_000_000);
    equals_str("-PT1.123S", -1, 123_000_000);
    equals_str("PT59S", 59, 0);
    equals_str("-PT59S", -59, 0);
    equals_str("PT1M", 60, 0);
    equals_str("-PT1M", -60, 0);
    equals_str("PT1M0.987987987S", 60, 987987987);
    equals_str("-PT1M0.987987987S", -60, 987987987);
    equals_str("PT1M1S", 61, 0);
    equals_str("-PT1M1S", -61, 0);
    equals_str("PT1M1.584S", 61, 584_000_000);
    equals_str("-PT1M1.584S", -61, 584_000_000);
    equals_str("PT59M59S", 3_599, 0);
    equals_str("-PT59M59S", -3_599, 0);
    equals_str("PT59M59.999999999S", 3_599, 999_999_999);
    equals_str("-PT59M59.999999999S", -3_599, 999_999_999);
    equals_str("PT1H", 3_600, 0);
    equals_str("-PT1H", -3_600, 0);
    equals_str("PT1H0.3459S", 3_600, 345_900_000);
    equals_str("-PT1H0.3459S", -3_600, 345_900_000);
    equals_str("PT1H59.11S", 3_659, 110_000_000);
    equals_str("-PT1H59.11S", -3_659, 110_000_000);
    equals_str("PT1H1M", 3_660, 0);
    equals_str("-PT1H1M", -3_660, 0);
    equals_str("PT1H1M0.123S", 3_660, 123_000_000);
    equals_str("-PT1H1M0.123S", -3_660, 123_000_000);
    equals_str("PT1H1M1S", 3_661, 0);
    equals_str("-PT1H1M1S", -3_661, 0);
    equals_str("PT1H1M1.123S", 3_661, 123_000_000);
    equals_str("-PT1H1M1.123S", -3_661, 123_000_000);
    equals_str("PT23H59M59S", 86_399, 0);
    equals_str("-PT23H59M59S", -86_399, 0);
    equals_str("PT23H59M59.123S", 86_399, 123_000_000);
    equals_str("-PT23H59M59.123S", -86_399, 123_000_000);
    equals_str("P1D", 86_400, 0);
    equals_str("-P1D", -86_400, 0);
    equals_str("P1DT0.123S", 86_400, 123_000_000);
    equals_str("-P1DT0.123S", -86_400, 123_000_000);
    equals_str("P1DT1S", 86_401, 0);
    equals_str("-P1DT1S", -86_401, 0);
    equals_str("P1DT1.123S", 86_401, 123_000_000);
    equals_str("-P1DT1.123S", -86_401, 123_000_000);
    equals_str("P1DT59S", 86_459, 0);
    equals_str("-P1DT59S", -86_459, 0);
    equals_str("P1DT59.123S", 86_459, 123_000_000);
    equals_str("-P1DT59.123S", -86_459, 123_000_000);
    equals_str("P1DT1M", 86_460, 0);
    equals_str("-P1DT1M", -86_460, 0);
    equals_str("P1DT1M0.123S", 86_460, 123_000_000);
    equals_str("-P1DT1M0.123S", -86_460, 123_000_000);
    equals_str("P1DT1M1S", 86_461, 0);
    equals_str("-P1DT1M1S", -86_461, 0);
    equals_str("P1DT1M1.123S", 86_461, 123_000_000);
    equals_str("-P1DT1M1.123S", -86_461, 123_000_000);
    equals_str("P1DT59M59S", 89_999, 0);
    equals_str("-P1DT59M59S", -89_999, 0);
    equals_str("P1DT59M59.123S", 89_999, 123_000_000);
    equals_str("-P1DT59M59.123S", -89_999, 123_000_000);
    equals_str("P1DT1H", 90_000, 0);
    equals_str("-P1DT1H", -90_000, 0);
    equals_str("P1DT1H0.123S", 90_000, 123_000_000);
    equals_str("-P1DT1H0.123S", -90_000, 123_000_000);
    equals_str("P1DT1H59S", 90_059, 0);
    equals_str("-P1DT1H59S", -90_059, 0);
    equals_str("P1DT1H59.123S", 90_059, 123_000_000);
    equals_str("-P1DT1H59.123S", -90_059, 123_000_000);
    equals_str("P1DT1H1M", 90_060, 0);
    equals_str("-P1DT1H1M", -90_060, 0);
    equals_str("P1DT1H1M0.123S", 90_060, 123_000_000);
    equals_str("-P1DT1H1M0.123S", -90_060, 123_000_000);
    equals_str("P1DT1H1M1S", 90_061, 0);
    equals_str("-P1DT1H1M1S", -90_061, 0);
    equals_str("P1DT1H1M1.123S", 90_061, 123_000_000);
    equals_str("-P1DT1H1M1.123S", -90_061, 123_000_000);
    equals_str("P1DT23H59M59S", 172_799, 0);
    equals_str("-P1DT23H59M59S", -172_799, 0);
    equals_str("P1DT23H59M59.123S", 172_799, 123_000_000);
    equals_str("-P1DT23H59M59.123S", -172_799, 123_000_000);
  }

  #[test]
  fn test_converting_nanos_to_string_should_pass() {
    assert_eq!("", nanoseconds_to_string(0));
    assert_eq!("000000001", nanoseconds_to_string(1));
    assert_eq!("00000001", nanoseconds_to_string(10));
    assert_eq!("0000001", nanoseconds_to_string(100));
    assert_eq!("000001", nanoseconds_to_string(1_000));
    assert_eq!("00001", nanoseconds_to_string(10_000));
    assert_eq!("0001", nanoseconds_to_string(100_000));
    assert_eq!("001", nanoseconds_to_string(1_000_000));
    assert_eq!("01", nanoseconds_to_string(10_000_000));
    assert_eq!("1", nanoseconds_to_string(100_000_000));
    assert_eq!("", nanoseconds_to_string(1_000_000_000));
  }

  #[test]
  fn test_eq_should_pass() {
    assert_eq!(
      Some(Ordering::Equal),
      FeelDaysAndTimeDuration::new(0, 0).partial_cmp(&FeelDaysAndTimeDuration::new(0, 0))
    );
    assert_eq!(
      Some(Ordering::Equal),
      FeelDaysAndTimeDuration::new(0, 10).partial_cmp(&FeelDaysAndTimeDuration::new(0, 10))
    );
    assert_eq!(
      Some(Ordering::Equal),
      FeelDaysAndTimeDuration::new(0, 999999999).partial_cmp(&FeelDaysAndTimeDuration::new(0, 999999999))
    );
    assert_eq!(
      Some(Ordering::Equal),
      FeelDaysAndTimeDuration::new(86_400, 999999999).partial_cmp(&FeelDaysAndTimeDuration::new(86_400, 999999999))
    );
    assert_eq!(FeelDaysAndTimeDuration::new(0, 0), FeelDaysAndTimeDuration::new(0, 0));
    assert_eq!(FeelDaysAndTimeDuration::new(0, 10), FeelDaysAndTimeDuration::new(0, 10));
    assert_eq!(FeelDaysAndTimeDuration::new(0, 999999999), FeelDaysAndTimeDuration::new(0, 999999999));
    assert_eq!(FeelDaysAndTimeDuration::new(86_400, 999999999), FeelDaysAndTimeDuration::new(86_400, 999999999));
  }

  #[test]
  fn test_lt_should_pass() {
    assert_eq!(
      Some(Ordering::Less),
      FeelDaysAndTimeDuration::new(10, 0).partial_cmp(&FeelDaysAndTimeDuration::new(11, 0))
    );
    assert_eq!(
      Some(Ordering::Less),
      FeelDaysAndTimeDuration::new(10, 1).partial_cmp(&FeelDaysAndTimeDuration::new(10, 2))
    );
    assert!(FeelDaysAndTimeDuration::new(10, 0) < FeelDaysAndTimeDuration::new(11, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 1) < FeelDaysAndTimeDuration::new(10, 2));
    assert!(FeelDaysAndTimeDuration::new(11, 0) >= FeelDaysAndTimeDuration::new(10, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 2) >= FeelDaysAndTimeDuration::new(10, 1));
  }

  #[test]
  fn test_le_should_pass() {
    assert!(FeelDaysAndTimeDuration::new(10, 0) <= FeelDaysAndTimeDuration::new(11, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 0) <= FeelDaysAndTimeDuration::new(10, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 0) <= FeelDaysAndTimeDuration::new(10, 1));
    assert!(FeelDaysAndTimeDuration::new(10, 1) <= FeelDaysAndTimeDuration::new(10, 1));
    assert!(FeelDaysAndTimeDuration::new(11, 0) > FeelDaysAndTimeDuration::new(10, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 2) > FeelDaysAndTimeDuration::new(10, 1));
  }

  #[test]
  fn test_gt_should_pass() {
    assert_eq!(
      Some(Ordering::Greater),
      FeelDaysAndTimeDuration::new(11, 0).partial_cmp(&FeelDaysAndTimeDuration::new(10, 0))
    );
    assert_eq!(
      Some(Ordering::Greater),
      FeelDaysAndTimeDuration::new(10, 1).partial_cmp(&FeelDaysAndTimeDuration::new(10, 0))
    );
    assert!(FeelDaysAndTimeDuration::new(11, 0) > FeelDaysAndTimeDuration::new(10, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 1) > FeelDaysAndTimeDuration::new(10, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 0) <= FeelDaysAndTimeDuration::new(11, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 1) <= FeelDaysAndTimeDuration::new(10, 2));
  }

  #[test]
  fn test_ge_should_pass() {
    assert!(FeelDaysAndTimeDuration::new(11, 0) >= FeelDaysAndTimeDuration::new(10, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 0) >= FeelDaysAndTimeDuration::new(10, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 1) >= FeelDaysAndTimeDuration::new(10, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 1) >= FeelDaysAndTimeDuration::new(10, 1));
    assert!(FeelDaysAndTimeDuration::new(10, 0) < FeelDaysAndTimeDuration::new(11, 0));
    assert!(FeelDaysAndTimeDuration::new(10, 1) < FeelDaysAndTimeDuration::new(10, 2));
  }

  #[test]
  fn test_add_should_pass() {
    let a = FeelDaysAndTimeDuration::new(11, 0);
    let b = FeelDaysAndTimeDuration::new(83, 0);
    let c = FeelDaysAndTimeDuration::new(94, 0);
    assert_eq!(c, a + b);
    let a = FeelDaysAndTimeDuration::new(11, 2_837);
    let b = FeelDaysAndTimeDuration::new(83, 23);
    let c = FeelDaysAndTimeDuration::new(94, 2_860);
    assert_eq!(c, a + b);
    let a = FeelDaysAndTimeDuration::new(1, 999_999_999);
    let b = FeelDaysAndTimeDuration::new(1, 2);
    let c = FeelDaysAndTimeDuration::new(3, 1);
    assert_eq!(c, a + b);
  }

  #[test]
  fn test_sub_should_pass() {
    let a = FeelDaysAndTimeDuration::new(12, 0);
    let b = FeelDaysAndTimeDuration::new(2, 0);
    let c = FeelDaysAndTimeDuration::new(10, 0);
    assert_eq!(c, a - b);
    let a = FeelDaysAndTimeDuration::new(99, 999_999_999);
    let b = FeelDaysAndTimeDuration::new(77, 888_888_888);
    let c = FeelDaysAndTimeDuration::new(22, 111_111_111);
    assert_eq!(c, a - b);
    let a = FeelDaysAndTimeDuration::new(1, 1);
    let b = FeelDaysAndTimeDuration::new(0, 2);
    let c = FeelDaysAndTimeDuration::new(0, 999_999_999);
    assert_eq!(c, a - b);
  }
}
