/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL definitions.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! implementation of `FEEL` name.

use {
  dmntk_common::{Feelify, Jsonify, Stringify},
  std::ops::Deref,
};

/// Common type definition for optional name.
pub type OptName = Option<Name>;

/// `FEEL` name.
#[derive(Debug, Clone, Eq, PartialEq, Ord, PartialOrd)]
pub struct Name(Vec<String>);

impl Default for Name {
  /// By default the name is an empty vector of strings.
  fn default() -> Self {
    Self(vec![])
  }
}

impl From<Vec<String>> for Name {
  /// Converts a vector of strings into [Name].
  fn from(value: Vec<String>) -> Self {
    Self(value.iter().map(|v| v.trim().to_string()).collect::<Vec<String>>())
  }
}

impl From<String> for Name {
  /// Converts a [String] into [Name].
  fn from(value: String) -> Self {
    Self(vec![value])
  }
}

impl From<&str> for Name {
  /// Converts a reference to [str] into [Name].
  fn from(value: &str) -> Self {
    Self(vec![value.to_string()])
  }
}

impl From<Name> for String {
  /// Converts [Name] to its [String] representation.
  fn from(value: Name) -> Self {
    value.to_string()
  }
}

impl From<&Name> for String {
  /// Converts a reference to [Name] to its [String] representation.
  fn from(value: &Name) -> Self {
    value.to_string()
  }
}

impl ToString for Name {
  /// Converts [Name] to [String].
  fn to_string(&self) -> String {
    self.0.join(" ").trim().to_string()
  }
}

impl Stringify for Name {
  /// Converts [Name] to its `TEXT` representation.
  fn stringify(&self) -> String {
    self.jsonify()
  }
}

impl Feelify for Name {
  /// Converts [Name] to its `FEEL` representation.
  fn feelify(&self) -> String {
    self.jsonify()
  }
}

impl Jsonify for Name {
  /// Converts [Name] to its `JSON` representation.
  fn jsonify(&self) -> String {
    format!("[{}]", self.0.iter().map(|s| format!(r#""{}""#, s)).collect::<Vec<String>>().join(","))
  }
}

impl Name {
  /// Creates a [Name] from name parts.
  pub fn new(parts: &[&str]) -> Self {
    Self(parts.iter().map(|&v| v.trim().to_string()).collect::<Vec<String>>())
  }
}

/// FEEL `QualifiedName`.
#[derive(Debug, Clone, PartialEq)]
pub struct QualifiedName(Vec<Name>);

impl QualifiedName {
  /// Creates a [QualifiedName] from [Names](Name).
  pub fn new(names: &[&Name]) -> Self {
    Self(names.iter().map(|&v| v.clone()).collect::<Vec<Name>>())
  }
}

impl ToString for QualifiedName {
  /// Converts [QualifiedName] to [String].
  fn to_string(&self) -> String {
    self.0.iter().map(|v| v.to_string()).collect::<Vec<String>>().join(".")
  }
}

impl Deref for QualifiedName {
  type Target = Vec<Name>;
  fn deref(&self) -> &Self::Target {
    &self.0
  }
}

impl QualifiedName {
  /// Appends this [QualifiedName] with a given [Name].
  pub fn push(&mut self, name: Name) {
    self.0.push(name);
  }
}

#[cfg(test)]
mod tests {
  use super::{Name, QualifiedName};

  /// Tests if the default value for [Name] is an empty vector of strings.
  #[test]
  fn test_default_name() {
    let name: Name = Default::default();
    assert_eq!("", name.to_string().as_str());
  }

  /// Tests creating a [Name] from vector of strings.
  #[test]
  fn test_from_vector() {
    let name: Name = vec!["".to_string(), "".to_string(), "".to_string()].into();
    assert_eq!("", name.to_string().as_str());
    let name: Name = vec!["x".to_string(), "y".to_string()].into();
    assert_eq!("x y", name.to_string().as_str());
    let name: Name = vec!["x".to_string(), "+".to_string(), "y".to_string()].into();
    assert_eq!("x + y", name.to_string().as_str());
    let name: Name = vec!["a".to_string(), "b".to_string(), "c".to_string()].into();
    assert_eq!("a b c", name.to_string().as_str());
  }

  /// Tests whether the constructor creates a new [QualifiedName] properly.
  #[test]
  fn test_new_qualified_name() {
    let name_a = Name::new(&["a", "+", "b"]);
    let name_b = Name::new(&["b", "-", "c"]);
    let name_c = Name::new(&["c", "/", "d"]);
    let name_d = Name::new(&["d", "*", "e"]);
    let name_e = Name::new(&["e", ".", "f"]);
    let name_f = Name::new(&["f", "'", "g"]);
    let qname = QualifiedName::new(&[]);
    assert_eq!("", qname.to_string().as_str());
    let qname = QualifiedName::new(&[&name_a]);
    assert_eq!("a + b", qname.to_string().as_str());
    let qname = QualifiedName::new(&[&name_a, &name_b]);
    assert_eq!("a + b.b - c", qname.to_string().as_str());
    let qname = QualifiedName::new(&[&name_a, &name_b, &name_c]);
    assert_eq!("a + b.b - c.c / d", qname.to_string().as_str());
    let qname = QualifiedName::new(&[&name_a, &name_b, &name_c, &name_d]);
    assert_eq!("a + b.b - c.c / d.d * e", qname.to_string().as_str());
    let qname = QualifiedName::new(&[&name_a, &name_b, &name_c, &name_d, &name_e]);
    assert_eq!("a + b.b - c.c / d.d * e.e . f", qname.to_string().as_str());
    let qname = QualifiedName::new(&[&name_a, &name_b, &name_c, &name_d, &name_e, &name_f]);
    assert_eq!("a + b.b - c.c / d.d * e.e . f.f ' g", qname.to_string().as_str());
  }
}
