/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL definitions.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use {
  self::errors::*,
  crate::{
    context::FeelContext,
    values::{Value, Values, VALUE_NULL},
    Name,
  },
  dmntk_common::{DmntkError, Stringify},
  std::convert::{TryFrom, TryInto},
};

#[derive(Debug, Deserialize)]
pub struct InputNodeDto {
  #[serde(rename = "name")]
  pub name: String,
  #[serde(rename = "value")]
  pub value: Option<ValueDto>,
}

#[derive(Debug, Serialize)]
pub struct OutputNodeDto {
  #[serde(rename = "value")]
  pub value: Option<ValueDto>,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct ValueDto {
  #[serde(rename = "simple")]
  pub simple: Option<SimpleDto>,
  #[serde(rename = "components")]
  pub components: Option<Vec<ComponentDto>>,
  #[serde(rename = "list")]
  pub list: Option<ListDto>,
}

impl Default for ValueDto {
  /// Created default and empty [ValueDto].
  fn default() -> Self {
    Self {
      simple: None,
      components: None,
      list: None,
    }
  }
}

#[derive(Debug, Serialize, Deserialize)]
pub struct SimpleDto {
  #[serde(rename = "type")]
  pub typ: Option<String>,
  #[serde(rename = "text")]
  pub text: Option<String>,
  #[serde(rename = "isNil")]
  pub nil: bool,
}

impl SimpleDto {
  /// Creates [SimpleDto] with some initial value.
  pub fn some(typ: &str, text: &str) -> Option<Self> {
    Some(Self {
      typ: Some(typ.to_string()),
      text: Some(text.to_string()),
      nil: false,
    })
  }
  /// Creates [SimpleDto] with `nil` value.
  pub fn nil() -> Option<Self> {
    Some(Self {
      typ: None,
      text: None,
      nil: true,
    })
  }
}

#[derive(Debug, Serialize, Deserialize)]
pub struct ComponentDto {
  #[serde(rename = "name")]
  pub name: Option<String>,
  #[serde(rename = "value")]
  pub value: Option<ValueDto>,
  #[serde(rename = "isNil")]
  pub nil: bool,
}

#[derive(Debug, Serialize, Deserialize)]
pub struct ListDto {
  #[serde(rename = "items")]
  pub items: Vec<ValueDto>,
  #[serde(rename = "isNil")]
  pub nil: bool,
}

impl ListDto {
  /// Creates [ListDto] with initial items.
  pub fn items(items: Vec<ValueDto>) -> Option<Self> {
    Some(Self { items, nil: false })
  }
}

impl TryFrom<Value> for OutputNodeDto {
  type Error = DmntkError;
  /// Tries to convert [Value] to [OutputNodeDto].
  fn try_from(value: Value) -> Result<Self, Self::Error> {
    match value {
      Value::String(inner) => Ok(OutputNodeDto {
        value: Some(ValueDto {
          simple: SimpleDto::some("xsd:string", &inner),
          ..Default::default()
        }),
      }),
      v @ Value::Number(_) => Ok(OutputNodeDto {
        value: Some(ValueDto {
          simple: SimpleDto::some("xsd:decimal", &v.stringify()),
          ..Default::default()
        }),
      }),
      v @ Value::Boolean(_) => Ok(OutputNodeDto {
        value: Some(ValueDto {
          simple: SimpleDto::some("xsd:boolean", &v.stringify()),
          ..Default::default()
        }),
      }),
      v @ Value::Date(_) => Ok(OutputNodeDto {
        value: Some(ValueDto {
          simple: SimpleDto::some("xsd:date", &v.stringify()),
          ..Default::default()
        }),
      }),
      v @ Value::DateTime(_) => Ok(OutputNodeDto {
        value: Some(ValueDto {
          simple: SimpleDto::some("xsd:dateTime", &v.stringify()),
          ..Default::default()
        }),
      }),
      v @ Value::Time(_) => Ok(OutputNodeDto {
        value: Some(ValueDto {
          simple: SimpleDto::some("xsd:time", &v.stringify()),
          ..Default::default()
        }),
      }),
      v @ Value::YearsAndMonthsDuration(_) => Ok(OutputNodeDto {
        value: Some(ValueDto {
          simple: SimpleDto::some("xsd:duration", &v.stringify()),
          ..Default::default()
        }),
      }),
      v @ Value::DaysAndTimeDuration(_) => Ok(OutputNodeDto {
        value: Some(ValueDto {
          simple: SimpleDto::some("xsd:duration", &v.stringify()),
          ..Default::default()
        }),
      }),
      Value::Null(_) => Ok(OutputNodeDto {
        value: Some(ValueDto {
          simple: SimpleDto::nil(),
          ..Default::default()
        }),
      }),
      Value::Context(ctx) => {
        let mut components = vec![];
        for (name, value) in ctx.iter() {
          components.push(ComponentDto {
            name: Some(name.to_string()),
            value: Some(value.try_into()?),
            nil: false,
          });
        }
        Ok(OutputNodeDto {
          value: Some(ValueDto {
            components: Some(components),
            ..Default::default()
          }),
        })
      }
      Value::List(list) => {
        let mut items = vec![];
        for value in list.to_vec() {
          items.push(value.try_into()?);
        }
        Ok(OutputNodeDto {
          value: Some(ValueDto {
            list: ListDto::items(items),
            ..Default::default()
          }),
        })
      }
      _ => Ok(OutputNodeDto { value: None }),
    }
  }
}

impl TryFrom<&Value> for ValueDto {
  type Error = DmntkError;
  /// Tries to convert [Value] to [ValueDto].
  fn try_from(value: &Value) -> Result<Self, Self::Error> {
    match value {
      Value::String(inner) => Ok(ValueDto {
        simple: SimpleDto::some("xsd:string", &inner),
        ..Default::default()
      }),
      v @ Value::Number(_) => Ok(ValueDto {
        simple: SimpleDto::some("xsd:decimal", &v.stringify()),
        ..Default::default()
      }),
      v @ Value::Boolean(_) => Ok(ValueDto {
        simple: SimpleDto::some("xsd:boolean", &v.stringify()),
        ..Default::default()
      }),
      v @ Value::Date(_) => Ok(ValueDto {
        simple: SimpleDto::some("xsd:date", &v.stringify()),
        ..Default::default()
      }),
      v @ Value::DateTime(_) => Ok(ValueDto {
        simple: SimpleDto::some("xsd:dateTime", &v.stringify()),
        ..Default::default()
      }),
      v @ Value::Time(_) => Ok(ValueDto {
        simple: SimpleDto::some("xsd:time", &v.stringify()),
        ..Default::default()
      }),
      v @ Value::YearsAndMonthsDuration(_) => Ok(ValueDto {
        simple: SimpleDto::some("xsd:duration", &v.stringify()),
        ..Default::default()
      }),
      v @ Value::DaysAndTimeDuration(_) => Ok(ValueDto {
        simple: SimpleDto::some("xsd:duration", &v.stringify()),
        ..Default::default()
      }),
      Value::Null(_) => Ok(ValueDto {
        simple: SimpleDto::nil(),
        ..Default::default()
      }),
      Value::Context(ctx) => {
        let mut components = vec![];
        for (name, value) in ctx.iter() {
          components.push(ComponentDto {
            name: Some(name.to_string()),
            value: Some(value.try_into()?),
            nil: false,
          });
        }
        Ok(ValueDto {
          components: Some(components),
          ..Default::default()
        })
      }
      Value::List(list) => {
        let mut items = vec![];
        for value in list.to_vec() {
          items.push(value.try_into()?);
        }
        Ok(ValueDto {
          list: ListDto::items(items),
          ..Default::default()
        })
      }
      _ => Ok(Default::default()),
    }
  }
}

impl TryFrom<&Vec<InputNodeDto>> for Value {
  type Error = DmntkError;
  fn try_from(items: &Vec<InputNodeDto>) -> Result<Self, Self::Error> {
    let mut ctx: FeelContext = Default::default();
    for item in items {
      // FIXME test if using the evaluator is necessary in this context
      // let name = crate::evaluator::parse_name(&item.name)?;
      let name: Name = item.name.as_str().into();
      ctx.set_entry(&name, Value::try_from(item)?);
    }
    Ok(ctx.into())
  }
}

impl TryFrom<&InputNodeDto> for Value {
  type Error = DmntkError;
  fn try_from(input_node_dto: &InputNodeDto) -> Result<Self, Self::Error> {
    if let Some(value_dto) = &input_node_dto.value {
      Value::try_from(value_dto)
    } else {
      Err(missing_parameter("InputNodeDto.value"))
    }
  }
}

impl TryFrom<&Vec<ValueDto>> for Value {
  type Error = DmntkError;
  fn try_from(items: &Vec<ValueDto>) -> Result<Self, Self::Error> {
    let mut values = vec![];
    for item in items {
      values.push(Value::try_from(item)?);
    }
    Ok(Value::List(Values::new(values)))
  }
}

impl TryFrom<&ValueDto> for Value {
  type Error = DmntkError;
  fn try_from(value: &ValueDto) -> Result<Self, Self::Error> {
    if let Some(value_dto) = &value.simple {
      return Value::try_from(value_dto);
    }
    if let Some(components) = &value.components {
      return Value::try_from(components);
    }
    if let Some(list) = &value.list {
      return Value::try_from(list);
    }
    Err(missing_parameter("no `simple`, `components` or `list` attribute in ValueTypeDto"))
  }
}

impl TryFrom<&SimpleDto> for Value {
  type Error = DmntkError;
  fn try_from(value: &SimpleDto) -> Result<Self, Self::Error> {
    if value.nil {
      return Ok(Value::Null(None));
    }
    if let Some(typ) = &value.typ {
      if let Some(text) = &value.text {
        return match typ.as_str() {
          "xsd:string" => Ok(Value::String(text.clone())),
          "xsd:integer" => Value::try_from_xsd_integer(&text),
          "xsd:decimal" => Value::try_from_xsd_decimal(&text),
          "xsd:double" => Value::try_from_xsd_double(&text),
          "xsd:boolean" => Value::try_from_xsd_boolean(&text),
          "xsd:date" => Value::try_from_xsd_date(&text),
          "xsd:time" => Value::try_from_xsd_time(&text),
          "xsd:dateTime" => Value::try_from_xsd_date_time(&text),
          "xsd:duration" => Value::try_from_xsd_duration(&text),
          _ => Err(invalid_parameter(&format!("unrecognized type: `{}` in value", typ))),
        };
      }
    }
    Err(invalid_parameter("ValueDto"))
  }
}

impl TryFrom<&Vec<ComponentDto>> for Value {
  type Error = DmntkError;
  fn try_from(items: &Vec<ComponentDto>) -> Result<Self, Self::Error> {
    let mut ctx: FeelContext = Default::default();
    for item in items {
      let item_name = item.name.as_ref().ok_or_else(|| invalid_parameter("component should have a name"))?;
      let value = Value::try_from(item)?;
      // FIXME verify if using name parser is necessary in this context
      // let key = crate::evaluator::parse_name(item_name)?;
      let key = item_name.as_str().into();
      ctx.set_entry(&key, value);
    }
    Ok(ctx.into())
  }
}

impl TryFrom<&ComponentDto> for Value {
  type Error = DmntkError;
  fn try_from(value: &ComponentDto) -> Result<Self, Self::Error> {
    if value.nil {
      return Ok(VALUE_NULL);
    }
    if let Some(v) = &value.value {
      Value::try_from(v)
    } else {
      Err(invalid_parameter("component should have a value"))
    }
  }
}

impl TryFrom<&ListDto> for Value {
  type Error = DmntkError;
  fn try_from(value: &ListDto) -> Result<Self, Self::Error> {
    if value.nil {
      return Ok(VALUE_NULL);
    }
    Value::try_from(&value.items)
  }
}

/// Definitions of errors reported by DTO conversions.
mod errors {
  use dmntk_common::DmntkError;

  /// DTO conversion errors.
  #[derive(Debug, PartialEq)]
  enum DtoError {
    MissingParameter(String),
    InvalidParameter(String),
  }

  impl From<DtoError> for DmntkError {
    fn from(e: DtoError) -> Self {
      DmntkError::new("DTOError", &format!("{}", e))
    }
  }

  impl std::fmt::Display for DtoError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        DtoError::MissingParameter(name) => {
          write!(f, "missing parameter: {}", name)
        }
        DtoError::InvalidParameter(name) => {
          write!(f, "invalid parameter: {}", name)
        }
      }
    }
  }

  pub fn missing_parameter(name: &str) -> DmntkError {
    DtoError::MissingParameter(name.to_string()).into()
  }

  pub fn invalid_parameter(description: &str) -> DmntkError {
    DtoError::InvalidParameter(description.to_string()).into()
  }
}
