/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL definitions.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! FEEL values.

use self::errors::*;
use crate::bif::Bif;
use crate::context::FeelContext;
use crate::names::Name;
use crate::temporal::date::FeelDate;
use crate::temporal::dt_duration::FeelDaysAndTimeDuration;
use crate::temporal::ym_duration::FeelYearsAndMonthsDuration;
use crate::temporal::{FeelDateTime, FeelTime};
use crate::types::FeelType;
use dmntk_common::errors::Result;
use dmntk_common::feelify::Feelify;
use dmntk_common::jsonify::Jsonify;
use dmntk_common::null_with_trace;
use std::collections::BTreeMap;
use std::convert::TryFrom;
use std::ops::Deref;
use std::str::FromStr;

/// Utility constant for `null` value of type `Null`.
pub const VALUE_NULL: Value = Value::Null(None);

/// Utility constant for `null` value of type `Null` with additional tracing message.
/// Tracing messages may be used for diagnostic purposes.
pub const VALUE_NULL_WITH_TRACE: Value = Value::Null(None);

/// Utility constant for value `true `of type `Boolean`.
pub const VALUE_TRUE: Value = Value::Boolean(true);

/// Utility constant for value `false` of type `Boolean`.
pub const VALUE_FALSE: Value = Value::Boolean(false);

/// Function body may be defined in `FEEL` or `DMN` in many ways,
/// so this enumeration is the representation of all these cases.
#[derive(Debug, Clone, PartialEq)]
pub enum FunctionBody {
  /// Function body created from model context.
  // Context(Context),
  /// Function body created from `FEEL` textual representation.
  // LiteralExpression(AstNode),
  /// Function body created from decision table.
  /// DecisionTable(DecisionTable),
  /// Function body is defined as a call to an externally defined function,
  /// it may be a Java function or PMML document. The value in this
  /// representation `SHALL` be a context.
  External(Box<Value>),
}

/// ???
#[derive(Debug, Clone, PartialEq)]
pub enum Value {
  /// Value representing `FEEL` boolean type.
  Boolean(bool),
  /// Value for storing built-in function definition.
  BuiltInFunction(Bif),
  /// Value for storing built-in type definition.
  BuiltInType(FeelType),
  /// Value for storing a collection of values representing a comma-separated list of values.
  CommaList(Values),
  /// Value representing a context.
  Context(FeelContext),
  /// Value representing a context entry.
  ContextEntry(Name, Box<Value>),
  /// Value representing a key of the context entry.
  ContextEntryKey(Name),
  /// Value representing a name of the context entry in context type definition.
  ContextEntryName(Name),
  /// Value for storing dates as [FeelDate].
  Date(FeelDate),
  /// Value for storing date and time as [FeelDateTime].
  DateTime(FeelDateTime),
  /// Value for days and time durations.
  DaysAndTimeDuration(FeelDaysAndTimeDuration),
  /// Value representing function's formal parameter with name and type.
  FormalParameter(Name, FeelType),
  /// List of formal parameters.
  FormalParameters(Vec<(Name, FeelType)>),
  /// Definition of the function body.
  FunctionBody(FunctionBody),
  /// Value representing function definition; (formal parameters, function body, result type).
  FunctionDefinition(Vec<(Name, FeelType)>, FunctionBody, FeelType),
  /// Value representing interval end.
  IntervalEnd(Box<Value>, bool),
  /// Value representing interval start.
  IntervalStart(Box<Value>, bool),
  /// **Irrelevant** value...
  Irrelevant,
  /// Value representing a list of values.
  List(Values),
  /// Named parameter.
  NamedParameter(Box<Value>, Box<Value>),
  /// Value representing a collection of name parameters.
  NamedParameters(BTreeMap<Name, Value>),
  /// **NegatedCommaList** value...
  NegatedCommaList(Vec<Value>),
  /// Null value with optional tracing message.
  Null(Option<String>),
  /// Numeric value
  Number(f64),
  /// Name of the parameter.
  ParameterName(Name),
  /// List of positional parameters.
  PositionalParameters(Vec<Value>),
  /// Value representing a `range`.
  Range(Box<Value>, bool, Box<Value>, bool),
  /// **String** value...
  String(String),
  /// Value for storing time as [FeelTime].
  Time(FeelTime),
  /// Value representing only the `FEEL` type of value.
  Type(FeelType),
  /// **UnaryGreater** value...
  UnaryGreater(Box<Value>),
  /// **UnaryGreaterOrEqual** value...
  UnaryGreaterOrEqual(Box<Value>),
  /// **UnaryLess** value...
  UnaryLess(Box<Value>),
  /// **UnaryLessOrEqual** value...
  UnaryLessOrEqual(Box<Value>),
  /// Value for storing years and months duration.
  YearsAndMonthsDuration(FeelYearsAndMonthsDuration),
}

impl Feelify for Value {
  /// Converts a [Value] to its `TEXT` representation.
  fn feelify(&self) -> String {
    match self {
      Value::Boolean(b) => format!("{}", b),
      Value::BuiltInFunction(_) => "BuiltInFunction".to_string(),
      Value::BuiltInType(_) => "BuiltInType".to_string(),
      Value::CommaList(items) => items.feelify(),
      Value::Context(ctx) => ctx.feelify(),
      Value::ContextEntry(_, _) => "ContextEntry".to_string(),
      Value::ContextEntryKey(name) => name.feelify(),
      Value::ContextEntryName(name) => name.feelify(),
      Value::Date(date) => format!("{}", date),
      Value::DateTime(date_time) => format!("{}", date_time),
      Value::DaysAndTimeDuration(duration) => duration.clone().into(),
      Value::FormalParameter(_, _) => "FormalParameter".to_string(),
      Value::FormalParameters(_) => "FormalParameters".to_string(),
      Value::FunctionBody(_) => "FunctionBody".to_string(),
      Value::FunctionDefinition(_, _, _) => "FunctionDefinition".to_string(),
      Value::IntervalEnd(_, _) => "IntervalEnd".to_string(),
      Value::IntervalStart(_, _) => "IntervalStart".to_string(),
      Value::Irrelevant => "Irrelevant".to_string(),
      Value::List(items) => items.feelify(),
      Value::NamedParameter(_, _) => "NamedParameter".to_string(),
      Value::NamedParameters(_) => "NamedParameters".to_string(),
      Value::NegatedCommaList(_) => "NegatedCommaList".to_string(),
      Value::Number(n) => format!("{}", n),
      Value::Null(trace) => format!("null{}", trace.as_ref().map_or("".to_string(), |s| format!(" ({})", s))),
      Value::ParameterName(_) => "ParameterName".to_string(),
      Value::PositionalParameters(_) => "PositionalParameters".to_string(),
      Value::Range(_, _, _, _) => "Range".to_string(),
      Value::String(s) => format!(r#""{}""#, s),
      Value::Time(time) => format!("{}", time),
      Value::Type(feel_type) => format!("type({:?})", feel_type),
      Value::UnaryGreater(_) => "UnaryGreater".to_string(),
      Value::UnaryGreaterOrEqual(_) => "UnaryGreaterOrEqual".to_string(),
      Value::UnaryLess(_) => "".to_string(),
      Value::UnaryLessOrEqual(_) => "UnaryLessOrEqual".to_string(),
      Value::YearsAndMonthsDuration(duration) => duration.clone().into(),
    }
  }
}

impl Jsonify for Value {
  /// Converts a [Value] to its `JSON` representation.
  fn jsonify(&self) -> String {
    match self {
      Value::Boolean(b) => format!("{}", b),
      Value::CommaList(items) => items.feelify(),
      Value::Context(ctx) => ctx.jsonify(),
      Value::ContextEntryKey(name) => name.feelify(),
      Value::List(items) => items.jsonify(),
      Value::Number(n) => format!("{}", n),
      Value::Null(_) => "null".to_string(),
      Value::String(s) => format!("\"{}\"", s),
      _ => format!("jsonify not implemented for: {:?}", self),
    }
  }
}

impl Value {
  /// Returns `true` when the value is of type [Value::Null].
  pub fn is_null(&self) -> bool {
    matches!(self, Value::Null(_))
  }
  /// Returns `true` when the value is of type [Value::Boolean] and is equal to `true`.
  pub fn is_true(&self) -> bool {
    matches!(self, Value::Boolean(true))
  }
  /// Returns `true` when the value is of type [Value::Number].
  pub fn is_number(&self) -> bool {
    matches!(self, Value::Number(_))
  }
  /// Returns the type of this [Value].
  pub fn type_of(&self) -> FeelType {
    match self {
      Value::Boolean(_) => FeelType::Boolean,
      Value::BuiltInFunction(_) => FeelType::Any,
      Value::BuiltInType(feel_type) => feel_type.clone(),
      Value::CommaList(_) => FeelType::Any,
      Value::Context(context) => {
        let mut entries = BTreeMap::new();
        for (name, value) in context.deref() {
          entries.insert(name.clone(), value.type_of());
        }
        FeelType::Context(entries)
      }
      Value::ContextEntry(_, _) => FeelType::Any,
      Value::ContextEntryKey(_) => FeelType::Any,
      Value::ContextEntryName(_) => FeelType::Any,
      Value::Date(_) => FeelType::Date,
      Value::DateTime(_) => FeelType::DateTime,
      Value::DaysAndTimeDuration(_) => FeelType::DaysAndTimeDuration,
      Value::FormalParameter(_, feel_type) => feel_type.clone(),
      Value::FormalParameters(_) => FeelType::Any,
      Value::FunctionBody(_) => FeelType::Any,
      Value::FunctionDefinition(parameters, _, result_type) => {
        let parameter_types = parameters.iter().map(|(_, feel_type)| feel_type.clone()).collect();
        FeelType::Function(parameter_types, Box::new(result_type.clone()))
      }
      Value::IntervalEnd(interval_end, _) => interval_end.type_of(),
      Value::IntervalStart(interval_start, _) => interval_start.type_of(),
      Value::Irrelevant => FeelType::Any,
      Value::List(values) => {
        if values.is_empty() {
          FeelType::List(Box::new(FeelType::Null))
        } else {
          let item_type = values.deref()[0].type_of();
          for item in values.deref() {
            if item.type_of() != item_type {
              return FeelType::List(Box::new(FeelType::Any));
            }
          }
          FeelType::List(Box::new(item_type))
        }
      }
      Value::NamedParameter(_, _) => FeelType::Any,
      Value::NamedParameters(_) => FeelType::Any,
      Value::NegatedCommaList(_) => FeelType::Any,
      Value::Null(_) => FeelType::Null,
      Value::Number(_) => FeelType::Number,
      Value::ParameterName(_) => FeelType::Any,
      Value::PositionalParameters(_) => FeelType::Any,
      Value::Range(range_start, _, range_end, _) => {
        let range_start_type = range_start.type_of();
        let range_end_type = range_end.type_of();
        if range_start_type == range_end_type {
          return FeelType::Range(Box::new(range_start_type));
        }
        FeelType::Range(Box::new(FeelType::Any))
      }
      Value::String(_) => FeelType::String,
      Value::Time(_) => FeelType::Time,
      Value::Type(feel_type) => feel_type.clone(),
      Value::UnaryGreater(_) => FeelType::Boolean,
      Value::UnaryGreaterOrEqual(_) => FeelType::Boolean,
      Value::UnaryLess(_) => FeelType::Boolean,
      Value::UnaryLessOrEqual(_) => FeelType::Boolean,
      Value::YearsAndMonthsDuration(_) => FeelType::YearsAndMonthsDuration,
    }
  }
  /// Applies coercion rules to this value and returns converted value or `null`.
  pub fn coerce(&self, feel_type: &FeelType) -> Value {
    // coercion of singleton list into single value when single type expected
    if let Value::List(items) = self {
      if items.len() == 1 && items[0].type_of().is_conformant(feel_type) {
        return items[0].clone();
      }
    }
    // coercion of single value into singleton list when expected a list
    if let FeelType::List(inner_feel_type) = feel_type {
      if self.type_of().is_conformant(inner_feel_type) {
        return Value::List(Values::new(vec![self.clone()]));
      }
    }
    null_with_trace!("coercion of value `{}` into type `{:?}` results with `null`", self.feelify(), feel_type)
  }
  /// Tries to convert `xsd:integer` string into valid [Value] representing a number.
  pub fn try_from_xsd_integer(text: &str) -> Result<Self> {
    let value = text.parse::<i64>().map_err(|_| invalid_xsd_integer(text))?;
    Ok(Value::Number(value as f64))
  }
  /// Tries to convert `xsd:decimal` string into valid [Value] representing a number.
  pub fn try_from_xsd_decimal(text: &str) -> Result<Self> {
    let value = text.parse::<f64>().map_err(|_| invalid_xsd_decimal(text))?;
    Ok(Value::Number(value))
  }
  /// Tries to convert `xsd:double` string into valid [Value] representing a number.
  pub fn try_from_xsd_double(text: &str) -> Result<Self> {
    let value = text.parse::<f64>().map_err(|_| invalid_xsd_double(text))?;
    Ok(Value::Number(value))
  }
  /// Tries to convert `xsd:boolean` string into valid [Value] representing a boolean.
  pub fn try_from_xsd_boolean(text: &str) -> Result<Self> {
    match text {
      "true" | "1" => Ok(Value::Boolean(true)),
      "false" | "0" => Ok(Value::Boolean(false)),
      _ => Err(invalid_xsd_boolean(text)),
    }
  }
  /// Tries to convert `xsd:date` string into valid [Value] representing a date.
  /// FEEL date format is fully conformant with `xsd:date`.
  pub fn try_from_xsd_date(text: &str) -> Result<Self> {
    if let Ok(feel_date) = FeelDate::try_from(text) {
      return Ok(Value::Date(feel_date));
    }
    Err(invalid_xsd_date(text))
  }
  /// Tries to convert `xsd:time` string into valid [Value] representing a time.
  /// FEEL time format is fully conformant with `xsd:time`.
  pub fn try_from_xsd_time(text: &str) -> Result<Self> {
    if let Ok(feel_time) = FeelTime::from_str(text) {
      return Ok(Value::Time(feel_time));
    }
    Err(invalid_xsd_time(text))
  }
  /// Tries to convert `xsd:dateTime` string into valid [Value] representing a date and time.
  /// FEEL date and time format is fully conformant with `xsd:dateTime`.
  pub fn try_from_xsd_date_time(text: &str) -> Result<Self> {
    Ok(Value::DateTime(FeelDateTime::try_from(text)?))
  }
  /// Tries to convert `xsd:duration` string into valid [Value] representing a date and time.
  /// FEEL durations are conformant with `xsd:duration` but spit into two ranges.
  pub fn try_from_xsd_duration(text: &str) -> Result<Self> {
    if let Ok(ym_duration) = FeelYearsAndMonthsDuration::try_from(text) {
      return Ok(Value::YearsAndMonthsDuration(ym_duration));
    }
    if let Ok(dt_duration) = FeelDaysAndTimeDuration::try_from(text) {
      return Ok(Value::DaysAndTimeDuration(dt_duration));
    }
    Err(invalid_xsd_duration(text))
  }
}

/// Collection of values.
#[derive(Debug, Clone, PartialEq)]
pub struct Values(Vec<Value>);

impl Values {
  pub fn new(values: Vec<Value>) -> Self {
    Self(values)
  }
}

impl Deref for Values {
  type Target = Vec<Value>;
  fn deref(&self) -> &Self::Target {
    &self.0
  }
}

impl Feelify for Values {
  ///
  fn feelify(&self) -> String {
    format!("[{}]", self.0.iter().map(|value| value.feelify()).collect::<Vec<String>>().join(","))
  }
}

impl Jsonify for Values {
  ///
  fn jsonify(&self) -> String {
    format!("[{}]", self.0.iter().map(|value| value.jsonify()).collect::<Vec<String>>().join(","))
  }
}

/// Definitions of value errors.
pub mod errors {
  use crate::values::Value;
  use dmntk_common::errors::DmntkError;
  use dmntk_common::feelify::Feelify;

  /// Value errors.
  #[derive(Debug, PartialEq)]
  enum ValueError {
    /// Used when parsed text is not acceptable xsd:integer representation.
    InvalidXsdInteger(String),
    /// Used when parsed text is not acceptable xsd:decimal representation.
    InvalidXsdDecimal(String),
    /// Used when parsed text is not acceptable xsd:double representation.
    InvalidXsdDouble(String),
    /// Used when parsed text is not acceptable xsd:boolean representation.
    InvalidXsdBoolean(String),
    /// Used when parsed text is not acceptable xsd:date representation.
    InvalidXsdDate(String),
    /// Used when parsed text is not acceptable xsd:time representation.
    InvalidXsdTime(String),
    /// Used when parsed text is not acceptable xsd:duration representation.
    InvalidXsdDuration(String),
    /// Used when converting a [Value] to [Context].
    ValueIsNotAContext(String),
  }

  impl From<ValueError> for DmntkError {
    fn from(e: ValueError) -> Self {
      DmntkError::new("ValueError", &format!("{}", e))
    }
  }

  impl std::fmt::Display for ValueError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        ValueError::InvalidXsdInteger(text) => {
          write!(f, "'{}' is not valid xsd:integer representation", text)
        }
        ValueError::InvalidXsdDecimal(text) => {
          write!(f, "'{}' is not valid xsd:decimal representation", text)
        }
        ValueError::InvalidXsdDouble(text) => {
          write!(f, "'{}' is not valid xsd:double representation", text)
        }
        ValueError::InvalidXsdBoolean(text) => {
          write!(f, "'{}' is not valid xsd:boolean representation", text)
        }
        ValueError::InvalidXsdDate(text) => {
          write!(f, "'{}' is not valid xsd:date representation", text)
        }
        ValueError::InvalidXsdTime(text) => {
          write!(f, "'{}' is not valid xsd:time representation", text)
        }
        ValueError::InvalidXsdDuration(text) => {
          write!(f, "'{}' is not valid xsd:duration representation", text)
        }
        ValueError::ValueIsNotAContext(text) => {
          write!(f, "'{}' is not a value containing context", text)
        }
      }
    }
  }

  pub fn invalid_xsd_integer(text: &str) -> DmntkError {
    ValueError::InvalidXsdInteger(text.to_string()).into()
  }

  pub fn invalid_xsd_decimal(text: &str) -> DmntkError {
    ValueError::InvalidXsdDecimal(text.to_string()).into()
  }

  pub fn invalid_xsd_double(text: &str) -> DmntkError {
    ValueError::InvalidXsdDouble(text.to_string()).into()
  }

  pub fn invalid_xsd_boolean(text: &str) -> DmntkError {
    ValueError::InvalidXsdBoolean(text.to_string()).into()
  }

  pub fn invalid_xsd_date(text: &str) -> DmntkError {
    ValueError::InvalidXsdDate(text.to_string()).into()
  }

  pub fn invalid_xsd_time(text: &str) -> DmntkError {
    ValueError::InvalidXsdTime(text.to_string()).into()
  }

  pub fn invalid_xsd_duration(text: &str) -> DmntkError {
    ValueError::InvalidXsdDuration(text.to_string()).into()
  }

  pub fn value_is_not_a_context(value: &Value) -> DmntkError {
    ValueError::ValueIsNotAContext(value.feelify()).into()
  }
}

#[cfg(test)]
mod tests {
  use crate::context::FeelContext;
  use crate::values::Value;

  #[test]
  fn test_value_is_number() {
    assert!(Value::Number(10.).is_number());
    assert!(Value::Number(10.123).is_number());
    assert!(!Value::Boolean(true).is_number());
    assert!(!Value::Boolean(false).is_number());
  }

  #[test]
  fn test_value_equality_number() {
    assert!((Value::Number(10.) == Value::Number(10.)));
    assert!((Value::Number(10.) != Value::Number(10.1)));
  }

  #[test]
  fn test_value_equality_boolean() {
    assert!((Value::Boolean(true) == Value::Boolean(true)));
    assert!((Value::Boolean(false) == Value::Boolean(false)));
    assert!((Value::Boolean(true) != Value::Boolean(false)));
    assert!((Value::Boolean(false) != Value::Boolean(true)));
  }

  #[test]
  fn test_value_equality_context() {
    assert!((Value::Context(FeelContext::default()) == Value::Context(FeelContext::default())));
  }
}
