/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL definitions.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Utilities for operating on context.

use self::errors::*;
use crate::names::{Name, QualifiedName};
use crate::types::FeelType;
use crate::values::Value;
use dmntk_common::errors::DmntkError;
use dmntk_common::feelify::Feelify;
use dmntk_common::jsonify::Jsonify;
use std::collections::{BTreeMap, HashSet};
use std::convert::TryFrom;
use std::ops::Deref;

/// Type alias for context entries.
type FeelContextEntries = BTreeMap<Name, Value>;

/// The `FEEL` context.
#[derive(Debug, Clone, PartialEq)]
pub struct FeelContext(FeelContextEntries);

impl Default for FeelContext {
  /// Creates a default empty [Context].
  fn default() -> Self {
    Self(FeelContextEntries::new())
  }
}

impl Deref for FeelContext {
  type Target = FeelContextEntries;
  fn deref(&self) -> &Self::Target {
    &self.0
  }
}

impl TryFrom<Value> for FeelContext {
  type Error = DmntkError;
  /// Tries to convert a [Value] to its [Context] representation.
  fn try_from(value: Value) -> Result<Self, Self::Error> {
    if let Value::Context(ctx) = value {
      Ok(ctx)
    } else {
      Err(value_is_not_a_context(&value))
    }
  }
}

impl From<FeelContext> for Value {
  /// Converts a [Context] to its [Value] representation.
  fn from(ctx: FeelContext) -> Self {
    Value::Context(ctx)
  }
}

impl Feelify for FeelContext {
  /// Converts this [Context] into equivalent `FEEL` representation.
  fn feelify(&self) -> String {
    format!(
      "{{{}}}",
      self
        .0
        .iter()
        .map(|(name, value)| format!(r#"{}:{}"#, name.to_string(), value.feelify()))
        .collect::<Vec<String>>()
        .join(",")
    )
  }
}

impl Jsonify for FeelContext {
  /// Converts this [Context] into equivalent `JSON` representation.
  fn jsonify(&self) -> String {
    format!(
      "{{{}}}",
      self
        .0
        .iter()
        .map(|(name, value)| format!(r#""{}":{}"#, name.to_string(), value.jsonify()))
        .collect::<Vec<String>>()
        .join(",")
    )
  }
}

impl FeelContext {
  /// Checks if this [Context] contains an entry pointed by [Name].
  pub fn contains_entry(&self, name: &Name) -> bool {
    self.0.contains_key(&name)
  }
  /// Checks if this [Context] contains an entry pointed by [QualifiedName].
  pub fn contains_entries(&self, qname: &QualifiedName) -> bool {
    self.contains_deep(qname.as_slice())
  }
  /// Sets a single value for specified entry name in this [Context].
  pub fn set_entry(&mut self, name: &Name, value: Value) {
    self.0.insert(name.clone(), value);
  }
  /// Returns a value of an entry specified by name.
  pub fn get_entry(&self, name: &Name) -> Option<&Value> {
    self.0.get(name)
  }
  /// Returns a first value contained by context.
  pub fn get_first(&self) -> Option<&Value> {
    self.0.values().take(1).next()
  }
  /// Returns the number of entries in this context.
  pub fn len(&self) -> usize {
    self.0.len()
  }
  /// Returns `true` when this context is empty.
  pub fn is_empty(&self) -> bool {
    self.0.is_empty()
  }
  ///
  pub fn zip(&mut self, other: &FeelContext) {
    for (name, value) in &other.0 {
      self.0.insert(name.clone(), value.clone());
    }
  }
  /// Creates an entry with a value for specified [QualifiedName].
  /// All non existing intermediary contexts will be created.
  pub fn create_entry(&mut self, qname: &QualifiedName, value: Value) {
    self.create_deep(qname.as_slice(), value);
  }
  /// Returns a list of flattened keys for this [Context].
  pub fn flatten_keys(&self) -> Vec<String> {
    let mut keys: HashSet<String> = HashSet::new();
    for (key, value) in self.0.iter() {
      keys.insert(key.into());
      if let Value::Context(sub_ctx) = value {
        let sub_keys = sub_ctx.flatten_keys();
        if !sub_keys.is_empty() {
          for sub_key in sub_keys {
            keys.insert(sub_key.clone());
            keys.insert(format!("{}.{}", key.to_string(), sub_key));
          }
        }
      }
      if let Value::List(items) = value {
        for item in items.deref() {
          if let Value::Context(item_ctx) = item {
            let sub_keys = item_ctx.flatten_keys();
            if !sub_keys.is_empty() {
              for sub_key in sub_keys {
                keys.insert(sub_key.clone());
                keys.insert(format!("{}.{}", key.to_string(), sub_key));
              }
            }
          }
        }
      }
      if let Value::Type(FeelType::Context(a)) = value {
        for name in a.keys() {
          let sub_key = name.to_string();
          keys.insert(sub_key.clone());
          keys.insert(format!("{}.{}", key.to_string(), sub_key));
        }
      }
    }
    keys.iter().cloned().collect()
  }
  /// Searches for a value of an entry pointed by specified qualified name.
  pub fn search_entry<'search>(&'search self, qname: &'search QualifiedName) -> Option<&'search Value> {
    self.search_deep(qname.as_slice())
  }
  /// Deep check for a value pointed by slice of names.
  pub fn contains_deep(&self, names: &[Name]) -> bool {
    if names.is_empty() {
      return false;
    }
    let tail = &names[1..];
    if let Some(value) = self.0.get(&names[0]) {
      if tail.is_empty() {
        return true;
      }
      if let Value::Context(context) = value {
        return context.contains_deep(tail);
      }
    }
    false
  }
  /// Creates intermediary contexts when needed.
  pub fn create_deep(&mut self, names: &[Name], value: Value) {
    // if there are no names, then return
    if names.is_empty() {
      return;
    }
    let key = names[0].clone();
    let tail = &names[1..];
    // if tail is empty, then insert the value under
    // specified key in current context and return
    if tail.is_empty() {
      self.0.insert(key, value);
      return;
    }
    // if there is a context under the specified key,
    // then insert value to this context and return
    if let Some(Value::Context(sub_ctx)) = self.0.get_mut(&key) {
      sub_ctx.create_deep(tail, value);
      return;
    }
    // finally, when got to this point, insert a value
    // to newly created context
    let mut sub_ctx = FeelContext::default();
    sub_ctx.create_deep(tail, value);
    self.0.insert(key, sub_ctx.into());
  }
  /// Deep search for a value pointed by names.
  pub fn search_deep(&self, names: &[Name]) -> Option<&Value> {
    if !names.is_empty() {
      let tail = &names[1..];
      if let Some(value) = self.0.get(&names[0]) {
        if let Value::Context(ctx) = value {
          return if tail.is_empty() { Some(value) } else { ctx.search_deep(tail) };
        } else if tail.is_empty() {
          return Some(value);
        }
      }
    }
    None
  }
}

/// Definitions of context errors.
pub mod errors {
  use crate::values::Value;
  use dmntk_common::errors::DmntkError;
  use dmntk_common::feelify::Feelify;

  /// Context errors.
  #[derive(Debug, PartialEq)]
  enum ContextError {
    /// Used when converting a [Value] to [Context].
    ValueIsNotAContext(String),
  }

  impl From<ContextError> for DmntkError {
    fn from(e: ContextError) -> Self {
      DmntkError::new("ContextError", &format!("{}", e))
    }
  }

  impl std::fmt::Display for ContextError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        ContextError::ValueIsNotAContext(text) => {
          write!(f, "'{}' is not a value containing context", text)
        }
      }
    }
  }

  pub fn value_is_not_a_context(value: &Value) -> DmntkError {
    ContextError::ValueIsNotAContext(value.feelify()).into()
  }
}

#[cfg(test)]
mod tests {
  use crate::context::FeelContext;
  use crate::names::{Name, QualifiedName};
  use crate::values::Value;
  use dmntk_common::feelify::Feelify;
  use dmntk_common::jsonify::Jsonify;

  #[test]
  fn test_context_default() {
    let ctx: FeelContext = Default::default();
    assert_eq!("{}", ctx.feelify());
    let empty: Vec<String> = vec![];
    assert_eq!(empty, ctx.flatten_keys());
  }

  #[test]
  fn test_context_stringify() {
    let name_a = Name::from("a");
    let name_x_y = Name::new(&["x", "y"]);
    let name_k_plus_l_minus_m = Name::new(&["k", "+", "l", "-", "m"]);
    let mut ctx: FeelContext = Default::default();
    ctx.set_entry(&name_a, Value::Number(10.0));
    assert_eq!(r#"{a:10}"#, ctx.feelify());
    ctx.set_entry(&name_x_y, Value::Boolean(true));
    assert_eq!(r#"{a:10,x y:true}"#, ctx.feelify());
    ctx.set_entry(&name_k_plus_l_minus_m, Value::String("KLM".to_string()));
    assert_eq!(r#"{a:10,k + l - m:"KLM",x y:true}"#, ctx.feelify());
  }

  #[test]
  fn test_context_jsonify() {
    let name_a = Name::from("a");
    let name_x_y = Name::new(&["x", "y"]);
    let name_k_plus_l_minus_m = Name::new(&["k", "+", "l", "-", "m"]);
    let mut ctx: FeelContext = Default::default();
    ctx.set_entry(&name_a, Value::Number(10.0));
    assert_eq!(r#"{"a":10}"#, ctx.jsonify());
    ctx.set_entry(&name_x_y, Value::Boolean(true));
    assert_eq!(r#"{"a":10,"x y":true}"#, ctx.jsonify());
    ctx.set_entry(&name_k_plus_l_minus_m, Value::String("KLM".to_string()));
    assert_eq!(r#"{"a":10,"k + l - m":"KLM","x y":true}"#, ctx.jsonify());
  }

  #[test]
  fn test_context_one_level() {
    let name_a = Name::from("a");
    let name_a_b = Name::new(&["a", "b"]);
    let name_a_b_c = Name::new(&["a", "b", "c"]);
    let qname_a = QualifiedName::new(&[&name_a]);
    let qname_a_b = QualifiedName::new(&[&name_a_b]);
    let qname_a_b_c = QualifiedName::new(&[&name_a_b_c]);
    let mut ctx_a: FeelContext = Default::default();
    ctx_a.set_entry(&name_a, Value::Number(10.0));
    assert_eq!("{a:10}", ctx_a.feelify());
    assert_eq!(true, ctx_a.contains_entry(&name_a));
    assert_eq!(true, ctx_a.contains_entries(&qname_a));
    assert_eq!(false, ctx_a.contains_entry(&name_a_b));
    assert_eq!(false, ctx_a.contains_entries(&qname_a_b));
    assert_eq!(false, ctx_a.contains_entry(&name_a_b_c));
    assert_eq!(false, ctx_a.contains_entries(&qname_a_b_c));
    assert_eq!(true, ctx_a.flatten_keys().contains(&"a".to_string()));
    assert_eq!("10", ctx_a.get_entry(&name_a).unwrap().feelify().as_str());
  }

  #[test]
  fn test_context_two_levels() {
    let name_married = Name::from("married");
    let name_age = Name::from("age");
    let name_b = Name::from("b");
    let name_x_y = Name::new(&["x", "y"]);
    let qname_married = QualifiedName::new(&[&name_married]);
    let qname_age = QualifiedName::new(&[&name_age]);
    let qname_b = QualifiedName::new(&[&name_b]);
    let qname_b_married = QualifiedName::new(&[&name_b, &name_married]);
    let qname_b_married_age = QualifiedName::new(&[&name_b, &name_married, &name_age]);
    let mut ctx_b: FeelContext = Default::default();
    ctx_b.set_entry(&name_married, Value::Boolean(true));
    assert_eq!("{married:true}", ctx_b.feelify());
    assert_eq!(true, ctx_b.contains_entry(&name_married));
    assert_eq!(true, ctx_b.contains_entries(&qname_married));
    let mut ctx_a: FeelContext = Default::default();
    ctx_a.set_entry(&name_age, Value::Number(49.0));
    ctx_a.set_entry(&name_x_y, Value::Boolean(true));
    ctx_a.set_entry(&name_b, ctx_b.into());
    assert_eq!("{age:49,b:{married:true},x y:true}", ctx_a.feelify());
    assert_eq!(true, ctx_a.contains_entry(&name_age));
    assert_eq!(true, ctx_a.contains_entry(&name_b));
    assert_eq!(true, ctx_a.contains_entry(&name_x_y));
    assert_eq!(true, ctx_a.contains_entries(&qname_age));
    assert_eq!(true, ctx_a.contains_entries(&qname_b));
    assert_eq!(true, ctx_a.contains_entries(&qname_b_married));
    assert_eq!(false, ctx_a.contains_entries(&qname_b_married_age));
    assert_eq!(true, ctx_a.flatten_keys().contains(&"age".to_string()));
    assert_eq!(true, ctx_a.flatten_keys().contains(&"b".to_string()));
    assert_eq!(true, ctx_a.flatten_keys().contains(&"b.married".to_string()));
    assert_eq!("49", ctx_a.get_entry(&name_age).unwrap().feelify().as_str());
    assert_eq!("{married:true}", ctx_a.get_entry(&name_b).unwrap().feelify().as_str());
  }

  #[test]
  fn test_context_three_levels() {
    let name_car = Name::from("car");
    let name_married = Name::from("married");
    let name_age = Name::from("age");
    let name_b = Name::from("b");
    let name_c = Name::from("c");
    let mut ctx_c: FeelContext = Default::default();
    ctx_c.set_entry(&name_car, Value::String("opel".to_string()));
    assert_eq!(r#"{car:"opel"}"#, ctx_c.feelify());
    assert_eq!(true, ctx_c.contains_entry(&name_car));
    let mut ctx_b: FeelContext = Default::default();
    ctx_b.set_entry(&name_married, Value::Boolean(true));
    ctx_b.set_entry(&name_c, ctx_c.into());
    assert_eq!(r#"{c:{car:"opel"},married:true}"#, ctx_b.feelify());
    assert_eq!(true, ctx_b.contains_entry(&name_married));
    assert_eq!(true, ctx_b.contains_entry(&name_c));
    let mut ctx_a: FeelContext = Default::default();
    ctx_a.set_entry(&name_age, Value::Number(49.0));
    ctx_a.set_entry(&name_b, ctx_b.into());
    assert_eq!(r#"{age:49,b:{c:{car:"opel"},married:true}}"#, ctx_a.feelify());
    assert_eq!(true, ctx_a.contains_entry(&name_age));
    assert_eq!(true, ctx_a.contains_entry(&name_b));
    assert_eq!(true, ctx_a.flatten_keys().contains(&"age".to_string()));
    assert_eq!(true, ctx_a.flatten_keys().contains(&"b".to_string()));
    assert_eq!(true, ctx_a.flatten_keys().contains(&"b.c".to_string()));
    assert_eq!(true, ctx_a.flatten_keys().contains(&"b.married".to_string()));
    assert_eq!(true, ctx_a.flatten_keys().contains(&"b.c.car".to_string()));
  }

  #[test]
  fn test_context_search_entry() {
    let name_married = Name::from("married");
    let name_age = Name::from("age");
    let name_b = Name::from("b");
    let mut ctx_b: FeelContext = Default::default();
    ctx_b.set_entry(&name_married, Value::Boolean(true));
    let mut ctx_a: FeelContext = Default::default();
    ctx_a.set_entry(&name_age, Value::Number(49.0));
    ctx_a.set_entry(&name_b, ctx_b.into());
    let qn_empty = QualifiedName::new(&[]);
    assert_eq!(None, ctx_a.search_entry(&qn_empty));
    let qn_b = QualifiedName::new(&[&name_b]);
    assert_eq!("{married:true}", ctx_a.search_entry(&qn_b).unwrap().feelify().as_str());
    let qn_b_married = QualifiedName::new(&[&name_b, &name_married]);
    assert_eq!("true", ctx_a.search_entry(&qn_b_married).unwrap().feelify().as_str());
  }

  #[test]
  fn test_context_search_entries() {
    // prepare names
    let name_a = Name::from("a");
    let name_b = Name::from("b");
    let name_c = Name::from("c");
    let name_d = Name::from("d");
    let name_e = Name::from("e");
    // prepare qualified names
    let qn_a = QualifiedName::new(&[&name_a]);
    let qn_b = QualifiedName::new(&[&name_b]);
    let qn_c = QualifiedName::new(&[&name_c]);
    let qn_a_b = QualifiedName::new(&[&name_a, &name_b]);
    let qn_a_c = QualifiedName::new(&[&name_a, &name_c]);
    let qn_b_c = QualifiedName::new(&[&name_b, &name_c]);
    let qn_b_e = QualifiedName::new(&[&name_b, &name_e]);
    let qn_b_d = QualifiedName::new(&[&name_b, &name_d]);
    let qn_b_d_a = QualifiedName::new(&[&name_b, &name_d, &name_a]);
    let qn_b_d_e = QualifiedName::new(&[&name_b, &name_d, &name_e]);
    // prepare contexts
    let mut ctx_c: FeelContext = Default::default();
    ctx_c.set_entry(&name_e, Value::String("e".to_string()));
    assert_eq!(r#"{e:"e"}"#, ctx_c.feelify());
    let mut ctx_b: FeelContext = Default::default();
    ctx_b.set_entry(&name_c, Value::String("c".to_string()));
    ctx_b.set_entry(&name_d, ctx_c.into());
    assert_eq!(r#"{c:"c",d:{e:"e"}}"#, ctx_b.feelify());
    let mut ctx_a: FeelContext = Default::default();
    ctx_a.set_entry(&name_a, Value::String("a".to_string()));
    ctx_a.set_entry(&name_b, ctx_b.into());
    assert_eq!(r#"{a:"a",b:{c:"c",d:{e:"e"}}}"#, ctx_a.feelify());
    // test searching entries
    assert_eq!(true, ctx_a.contains_entries(&qn_a));
    assert_eq!(true, ctx_a.contains_entries(&qn_b));
    assert_eq!(false, ctx_a.contains_entries(&qn_c));
    assert_eq!(false, ctx_a.contains_entries(&qn_a_b));
    assert_eq!(false, ctx_a.contains_entries(&qn_a_c));
    assert_eq!(true, ctx_a.contains_entries(&qn_b_c));
    assert_eq!(false, ctx_a.contains_entries(&qn_b_e));
    assert_eq!(true, ctx_a.contains_entries(&qn_b_d));
    assert_eq!(false, ctx_a.contains_entries(&qn_b_d_a));
    assert_eq!(true, ctx_a.contains_entries(&qn_b_d_e));
  }

  #[test]
  fn test_context_create_entry() {
    let name_a = Name::from("a");
    let name_b = Name::from("b");
    let name_c = Name::from("c");
    let name_d = Name::from("d");
    let qn_a = QualifiedName::new(&[&name_a]);
    let qn_b = QualifiedName::new(&[&name_b]);
    let qn_a_b = QualifiedName::new(&[&name_a, &name_b]);
    let qn_a_c = QualifiedName::new(&[&name_a, &name_c]);
    let qn_a_d = QualifiedName::new(&[&name_a, &name_d]);
    let qn_c_d = QualifiedName::new(&[&name_c, &name_d]);
    let qn_a_b_c = QualifiedName::new(&[&name_a, &name_b, &name_c]);
    let qn_a_b_c_d = QualifiedName::new(&[&name_a, &name_b, &name_c, &name_d]);
    let mut ctx: FeelContext = Default::default();
    ctx.create_entry(&qn_a_b_c_d, Value::Boolean(true));
    assert_eq!("{a:{b:{c:{d:true}}}}", ctx.feelify().as_str());
    assert_eq!("{b:{c:{d:true}}}", ctx.search_entry(&qn_a).unwrap().feelify().as_str());
    assert_eq!("{c:{d:true}}", ctx.search_entry(&qn_a_b).unwrap().feelify().as_str());
    assert_eq!("{d:true}", ctx.search_entry(&qn_a_b_c).unwrap().feelify().as_str());
    assert_eq!("true", ctx.search_entry(&qn_a_b_c_d).unwrap().feelify().as_str());
    let mut ctx: FeelContext = Default::default();
    ctx.create_entry(&qn_a, Value::Boolean(true));
    ctx.create_entry(&qn_b, Value::Boolean(false));
    ctx.create_entry(&qn_c_d, Value::String("deep".to_string()));
    assert_eq!(r#"{a:true,b:false,c:{d:"deep"}}"#, ctx.feelify().as_str());
    let mut ctx: FeelContext = Default::default();
    ctx.create_entry(&qn_a_b, Value::String("b".to_string()));
    ctx.create_entry(&qn_a_c, Value::String("c".to_string()));
    ctx.create_entry(&qn_a_d, Value::String("d".to_string()));
    assert_eq!(r#"{a:{b:"b",c:"c",d:"d"}}"#, ctx.feelify().as_str());
  }

  #[test]
  fn test_context_flatten_keys_one_level() {
    let name_a = Name::from("a");
    let name_b = Name::from("b");
    let mut ctx: FeelContext = Default::default();
    ctx.set_entry(&name_a, Value::Number(1.0));
    ctx.set_entry(&name_b, Value::Number(2.0));
    assert_eq!(r#"{a:1,b:2}"#, ctx.feelify());
    let keys = ctx.flatten_keys();
    assert_eq!(2, keys.len());
    assert!(keys.contains(&"a".to_string()));
    assert!(keys.contains(&"b".to_string()));
  }

  #[test]
  fn flatten_two_levels() {
    let name_a = Name::from("a");
    let name_b = Name::from("b");
    let name_c = Name::from("c");
    let mut ctx_b: FeelContext = Default::default();
    ctx_b.set_entry(&name_a, Value::Number(10.0));
    ctx_b.set_entry(&name_b, Value::Number(20.0));
    ctx_b.set_entry(&name_c, Value::Number(30.0));
    let mut ctx_a: FeelContext = Default::default();
    ctx_a.set_entry(&name_a, Value::Number(1.0));
    ctx_a.set_entry(&name_b, Value::Number(2.0));
    ctx_a.set_entry(&name_c, ctx_b.into());
    let keys = ctx_a.flatten_keys();
    assert_eq!(6, keys.len());
    assert!(keys.contains(&"a".to_string()));
    assert!(keys.contains(&"b".to_string()));
    assert!(keys.contains(&"c".to_string()));
    assert!(keys.contains(&"c.a".to_string()));
    assert!(keys.contains(&"c.b".to_string()));
    assert!(keys.contains(&"c.c".to_string()));
  }

  #[test]
  fn flatten_three_levels() {
    let name_a = Name::from("a");
    let name_b = Name::from("b");
    let name_c = Name::from("c");
    let name_d = Name::from("d");
    let mut ctx_c: FeelContext = Default::default();
    ctx_c.set_entry(&name_a, Value::Number(100.0));
    ctx_c.set_entry(&name_b, Value::Number(200.0));
    ctx_c.set_entry(&name_c, Value::Number(300.0));
    ctx_c.set_entry(&name_d, Value::Number(400.0));
    let mut ctx_b: FeelContext = Default::default();
    ctx_b.set_entry(&name_a, Value::Number(10.0));
    ctx_b.set_entry(&name_b, Value::Number(20.0));
    ctx_b.set_entry(&name_c, Value::Number(30.0));
    ctx_b.set_entry(&name_d, ctx_c.into());
    let mut ctx_a: FeelContext = Default::default();
    ctx_a.set_entry(&name_a, Value::Number(1.0));
    ctx_a.set_entry(&name_b, Value::Number(2.0));
    ctx_a.set_entry(&name_c, ctx_b.into());
    let keys = ctx_a.flatten_keys();
    assert_eq!(16, keys.len());
    assert!(keys.contains(&"a".to_string()));
    assert!(keys.contains(&"b".to_string()));
    assert!(keys.contains(&"c".to_string()));
    assert!(keys.contains(&"d".to_string()));
    assert!(keys.contains(&"c.a".to_string()));
    assert!(keys.contains(&"c.b".to_string()));
    assert!(keys.contains(&"c.c".to_string()));
    assert!(keys.contains(&"c.d".to_string()));
    assert!(keys.contains(&"c.d.a".to_string()));
    assert!(keys.contains(&"c.d.b".to_string()));
    assert!(keys.contains(&"c.d.c".to_string()));
    assert!(keys.contains(&"c.d.d".to_string()));
    assert!(keys.contains(&"d.a".to_string()));
    assert!(keys.contains(&"d.b".to_string()));
    assert!(keys.contains(&"d.c".to_string()));
    assert!(keys.contains(&"d.d".to_string()));
  }

  #[test]
  fn flatten_names_with_additional_characters() {
    let name_a = Name::new(&["lorem", "ipsum", "dolor", "sit", "amet"]);
    let name_b = Name::new(&["b"]);
    let name_c = Name::new(&["now", "let", "'", "s", "go", "to", "the", "next", "paragraph"]);
    let name_d = Name::new(&["Curabitur", "rhoncus", "+", "sodales", "odio", "in", "fringilla"]);
    let mut ctx_b: FeelContext = Default::default();
    ctx_b.set_entry(&name_d, Value::Boolean(false));
    let mut ctx_a: FeelContext = Default::default();
    ctx_a.set_entry(&name_a, Value::Number(1.0));
    ctx_a.set_entry(&name_b, Value::Number(2.0));
    ctx_a.set_entry(&name_c, ctx_b.into());
    let keys = ctx_a.flatten_keys();
    assert_eq!(5, keys.len());
    assert!(keys.contains(&"b".to_string()));
    assert!(keys.contains(&"lorem ipsum dolor sit amet".to_string()));
    assert!(keys.contains(&"now let ' s go to the next paragraph".to_string()));
    assert!(keys.contains(&"now let ' s go to the next paragraph.Curabitur rhoncus + sodales odio in fringilla".to_string()));
    assert!(keys.contains(&"Curabitur rhoncus + sodales odio in fringilla".to_string()));
  }
}
