/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use regex::Captures;
use std::ops::{Div, Rem};

/// FEEL time zones.
#[derive(Debug, Clone, PartialEq)]
pub enum FeelZone {
  /// UTC time zone.
  Utc,
  /// Local time zone.
  Local,
  /// Time zone defined as an offset from UTC.
  Offset(i32),
  /// Time zone defined as a value from IANA database.
  Zone(String),
}

impl std::fmt::Display for FeelZone {
  fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
    match self {
      FeelZone::Utc => write!(f, "Z"),
      FeelZone::Local => write!(f, ""),
      FeelZone::Offset(offset) => {
        let hours = offset / 3_600;
        let minutes = offset.abs().rem(3_600).div(60);
        let seconds = offset.abs().rem(3_600).rem(60);
        if seconds > 0 {
          write!(f, "{:+03}:{:02}:{02}", hours, minutes, seconds)
        } else {
          write!(f, "{:+03}:{:02}", hours, minutes)
        }
      }
      FeelZone::Zone(zone) => write!(f, "@{}", zone),
    }
  }
}

impl FeelZone {
  pub fn new(offset: i64) -> Self {
    if offset != 0 {
      Self::Offset(offset as i32)
    } else {
      Self::Utc
    }
  }
  ///
  pub fn from_captures(captures: &Captures) -> Option<Self> {
    if captures.name("zulu").is_some() {
      return Some(FeelZone::Utc);
    }
    if let Some(sign_match) = captures.name("offSign") {
      if let Some(hour_match) = captures.name("offHour") {
        if let Ok(hour) = hour_match.as_str().parse::<i32>() {
          if let Some(minute_match) = captures.name("offMinute") {
            if let Ok(minute) = minute_match.as_str().parse::<i32>() {
              let mut seconds = (hour * 3600 + minute * 60).rem(86400);
              if sign_match.as_str() == "-" {
                seconds = -seconds;
              }
              if hour > 14 {
                // the hour magnitude is limited to at most 14
                return None;
              }
              return Some(FeelZone::new(seconds as i64));
            }
          }
        }
      }
    }
    if let Some(zone_match) = captures.name("zone") {
      return if zone_match.as_str().parse::<chrono_tz::Tz>().is_ok() {
        Some(FeelZone::Zone(zone_match.as_str().to_string()))
      } else {
        None
      };
    }
    Some(FeelZone::Local)
  }
}
