/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! FEEL scope.

use crate::context::FeelContext;
use crate::values::Value;
use crate::Name;
use dmntk_common::{Jsonify, Stringify};
use std::cell::RefCell;
use std::collections::HashSet;

/// The `FEEL` scope.
pub struct Scope {
  /// The stack of contexts.
  contexts: RefCell<Vec<FeelContext>>,
}

impl Default for Scope {
  /// Creates a default [Scope] containing one default [FeelContext].
  fn default() -> Self {
    Self {
      contexts: RefCell::new(vec![FeelContext::default()]),
    }
  }
}

impl From<FeelContext> for Scope {
  /// Creates a [Scope] from [FeelContext].
  fn from(context: FeelContext) -> Self {
    Self {
      contexts: RefCell::new(vec![context]),
    }
  }
}

impl Stringify for Scope {
  /// Converts this [Scope] to its `TEXT` representation.
  fn stringify(&self) -> String {
    format!(
      "[{}]",
      self.contexts.borrow_mut().iter().map(|ctx| ctx.stringify()).collect::<Vec<String>>().join(",")
    )
  }
}

impl Jsonify for Scope {
  /// Converts this [Scope] to its `JSON` representation.
  fn jsonify(&self) -> String {
    format!(
      "[{}]",
      self.contexts.borrow_mut().iter().map(|ctx| ctx.jsonify()).collect::<Vec<String>>().join(",")
    )
  }
}

impl Scope {
  /// Creates a new and empty [Scope].
  pub fn new() -> Self {
    Self {
      contexts: RefCell::new(vec![]),
    }
  }
  /// Pushes a context on the top of the scope stack.
  pub fn push(&self, ctx: FeelContext) {
    self.contexts.borrow_mut().push(ctx)
  }
  /// Takes and returns a context from the top of the scope stack.
  pub fn pop(&self) -> Option<FeelContext> {
    self.contexts.borrow_mut().pop()
  }
  /// Returns a vector of flattened keys in all contexts in scope.
  pub fn flatten_keys(&self) -> HashSet<String> {
    self
      .contexts
      .borrow_mut()
      .iter()
      .flat_map(|ctx| ctx.flatten_keys())
      .collect::<HashSet<String>>()
  }
  /// Returns a value for an entry specified by name.
  /// Entries are searched from the last to the first context
  /// (from top to bottom of scope stack).
  pub fn get_entry(&self, name: &Name) -> Option<Value> {
    for context in self.contexts.borrow_mut().iter().rev() {
      if let Some(value) = context.get_entry(name) {
        return Some(value.clone());
      }
    }
    None
  }
  ///
  pub fn search_deep(&self, names: &[Name]) -> Option<Value> {
    for context in self.contexts.borrow_mut().iter().rev() {
      if let Some(value) = context.search_deep(names) {
        return Some(value.clone());
      }
    }
    None
  }
  /// Sets a value for entry name in [FeelContext] placed on the top of the scope stack (last context).
  pub fn set_entry(&self, name: &Name, value: Value) {
    if let Some(context) = self.contexts.borrow_mut().last_mut() {
      context.set_entry(name, value);
    }
  }
}

#[cfg(test)]
mod tests {
  use crate::context::FeelContext;
  use crate::values::Value;
  use crate::{Name, Scope};
  use dmntk_common::Stringify;

  #[test]
  fn test_scope_default() {
    assert_eq!("[{}]", Scope::default().stringify());
  }

  #[test]
  fn test_scope_new() {
    assert_eq!("[]", Scope::new().stringify());
  }

  #[test]
  fn test_scope_single_empty_context() {
    let scope = Scope::new();
    let ctx = FeelContext::default();
    scope.push(ctx);
    assert_eq!("[{}]", scope.stringify());
    let scope: Scope = FeelContext::default().into();
    assert_eq!("[{}]", scope.stringify());
  }

  #[test]
  fn test_scope_multiple_empty_contexts() {
    let scope = Scope::new();
    scope.push(FeelContext::default());
    scope.push(FeelContext::default());
    scope.push(FeelContext::default());
    assert_eq!("[{},{},{}]", scope.stringify());
    let scope = Scope::default();
    scope.push(FeelContext::default());
    scope.push(FeelContext::default());
    scope.push(FeelContext::default());
    assert_eq!("[{},{},{},{}]", scope.stringify());
  }

  #[test]
  fn test_scope_single_context() {
    let scope = Scope::default();
    assert_eq!("[{}]", scope.stringify());
    let name_a = Name::from("a");
    let name_b = Name::from("b");
    scope.set_entry(&name_a, Value::Number(49.5));
    assert_eq!("[{a:49.5}]", scope.stringify());
    scope.set_entry(&name_b, Value::Number(50.0));
    assert_eq!("[{a:49.5,b:50}]", scope.stringify());
    scope.pop();
    assert_eq!("[]", scope.stringify());
  }

  #[test]
  fn test_scope_no_context() {
    let scope = Scope::new();
    assert_eq!("[]", scope.stringify());
    let name_a = Name::from("a");
    let name_b = Name::from("b");
    scope.set_entry(&name_a, Value::Number(1.25));
    assert_eq!("[]", scope.stringify());
    scope.set_entry(&name_b, Value::Number(1.75));
    assert_eq!("[]", scope.stringify());
    scope.pop();
    assert_eq!("[]", scope.stringify());
  }
}
