/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL parser.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use super::accept;
use crate::lalr::TokenType::*;
use dmntk_feel::values::VALUE_NULL;
use dmntk_feel::Scope;

#[test]
fn test_numeric() {
  let scope = &mut Scope::default();
  accept(
    scope,
    StartTextualExpression,
    "1",
    r#"
        Numeric 1.
      "#,
    false,
  );
  accept(
    scope,
    StartTextualExpression,
    "(1.32983740938573405329458372450983275)",
    r#"
        Numeric 1.32983740938573405329458372450983275
      "#,
    false,
  );
  accept(
    scope,
    StartTextualExpression,
    "-14",
    r#"
        Neg
          Numeric 14.
      "#,
    false,
  );
}

#[test]
fn test_string() {
  let scope = &mut Scope::default();
  accept(
    scope,
    StartTextualExpression,
    r#""earth""#,
    r#"
        String `earth`
      "#,
    false,
  );
}

#[test]
fn test_list() {
  let scope = &mut Scope::default();
  accept(
    scope,
    StartBoxedExpression,
    "[]",
    r#"
        List
          []
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    "[12.45]",
    r#"
        List
          Numeric 12.45
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    "[1.1,2.2]",
    r#"
        List
          Numeric 1.1
          Numeric 2.2
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    r#"["a","b","c"]"#,
    r#"
        List
          String `a`
          String `b`
          String `c`
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    "[[1]]",
    r#"
        List
          List
            Numeric 1.
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    "[[1],[2]]",
    r#"
        List
          List
            Numeric 1.
          List
            Numeric 2.
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    "[[1],[2],[3]]",
    r#"
        List
          List
            Numeric 1.
          List
            Numeric 2.
          List
            Numeric 3.
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    "[[[1]]]",
    r#"
        List
          List
            List
              Numeric 1.
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    "[[[1],[2],[3]],[[4],[5]],[7,8,9]]",
    r#"
        List
          List
            List
              Numeric 1.
            List
              Numeric 2.
            List
              Numeric 3.
          List
            List
              Numeric 4.
            List
              Numeric 5.
          List
            Numeric 7.
            Numeric 8.
            Numeric 9.
      "#,
    false,
  );
}

#[test]
fn test_context() {
  let scope = &mut Scope::default();
  accept(
    scope,
    StartContext,
    "{}",
    r#"
        Context
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    "{}",
    r#"
        Context
      "#,
    false,
  );
  accept(
    scope,
    StartContext,
    "{age:49}",
    r#"
        Context
          ContextEntry
            ContextEntryKey `age`
            Numeric 49.
      "#,
    false,
  );
  accept(
    scope,
    StartContext,
    r#"{"age":49}"#,
    r#"
        Context
          ContextEntry
            ContextEntryKey `age`
            Numeric 49.
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    "{age:49}",
    r#"
        Context
          ContextEntry
            ContextEntryKey `age`
            Numeric 49.
      "#,
    false,
  );
  accept(
    scope,
    StartContext,
    r#"{name:"Dariusz",age:49}"#,
    r#"
        Context
          ContextEntry
            ContextEntryKey `name`
            String `Dariusz`
          ContextEntry
            ContextEntryKey `age`
            Numeric 49.
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    r#"{name:"Dariusz",age:49}"#,
    r#"
        Context
          ContextEntry
            ContextEntryKey `name`
            String `Dariusz`
          ContextEntry
            ContextEntryKey `age`
            Numeric 49.
      "#,
    false,
  );
  accept(
    scope,
    StartContext,
    r#"{name:"Dariusz",age:49,car:{model:"Skoda",production year:2016}}"#,
    r#"
        Context
          ContextEntry
            ContextEntryKey `name`
            String `Dariusz`
          ContextEntry
            ContextEntryKey `age`
            Numeric 49.
          ContextEntry
            ContextEntryKey `car`
            Context
              ContextEntry
                ContextEntryKey `model`
                String `Skoda`
              ContextEntry
                ContextEntryKey `production year`
                Numeric 2016.
      "#,
    false,
  );
  accept(
    scope,
    StartBoxedExpression,
    r#"{name:"Dariusz",age:49,car:{model:"Skoda",production year:2016}}"#,
    r#"
        Context
          ContextEntry
            ContextEntryKey `name`
            String `Dariusz`
          ContextEntry
            ContextEntryKey `age`
            Numeric 49.
          ContextEntry
            ContextEntryKey `car`
            Context
              ContextEntry
                ContextEntryKey `model`
                String `Skoda`
              ContextEntry
                ContextEntryKey `production year`
                Numeric 2016.
      "#,
    false,
  );
  accept(
    scope,
    StartContext,
    "{ a: 1, b: 2, c: 1 + 2}",
    r#"
        Context
          ContextEntry
            ContextEntryKey `a`
            Numeric 1.
          ContextEntry
            ContextEntryKey `b`
            Numeric 2.
          ContextEntry
            ContextEntryKey `c`
            Add
              Numeric 1.
              Numeric 2.
      "#,
    false,
  );
  let scope = &mut Scope::default();
  scope.set_entry(&"d".into(), VALUE_NULL);
  scope.set_entry(&"e".into(), VALUE_NULL);
  accept(
    scope,
    StartContext,
    "{ a: 1, b: 2, c: d + e}",
    r#"
        Context
          ContextEntry
            ContextEntryKey `a`
            Numeric 1.
          ContextEntry
            ContextEntryKey `b`
            Numeric 2.
          ContextEntry
            ContextEntryKey `c`
            Add
              Name
                `d`
              Name
                `e`
      "#,
    false,
  );
  accept(
    scope,
    StartContext,
    "{ a: 1, b: 2, c: a + b}",
    r#"
        Context
          ContextEntry
            ContextEntryKey `a`
            Numeric 1.
          ContextEntry
            ContextEntryKey `b`
            Numeric 2.
          ContextEntry
            ContextEntryKey `c`
            Add
              Name
                `a`
              Name
                `b`
      "#,
    false,
  );
}

#[test]
fn test_filter() {
  let scope = &mut Scope::default();
  accept(
    scope,
    StartTextualExpression,
    "[1,2,3][2]",
    r#"
        Filter
          List
            Numeric 1.
            Numeric 2.
            Numeric 3.
          Numeric 2.
      "#,
    false,
  );
  accept(
    scope,
    StartTextualExpression,
    "EmployeeTable[name=LastName]",
    r#"
        Filter
          Name
            `EmployeeTable`
          Eq
            Name
              `name`
            Name
              `LastName`
      "#,
    false,
  );
  accept(
    scope,
    StartTextualExpression,
    "EmployeeTable[1].deptNum",
    r#"
        Path
          Filter
            Name
              `EmployeeTable`
            Numeric 1.
          Name
            `deptNum`
      "#,
    false,
  );
  accept(
    scope,
    StartTextualExpression,
    "DeptTable[number=EmployeeTable[name=LastName].deptNum[1]].manager[1]",
    r#"
        Filter
          Path
            Filter
              Name
                `DeptTable`
              Eq
                Name
                  `number`
                Filter
                  Path
                    Filter
                      Name
                        `EmployeeTable`
                      Eq
                        Name
                          `name`
                        Name
                          `LastName`
                    Name
                      `deptNum`
                  Numeric 1.
            Name
              `manager`
          Numeric 1.
      "#,
    false,
  );
}

#[test]
fn test_subtraction() {
  let scope = &mut Scope::default();
  accept(
    scope,
    StartTextualExpression,
    "3 - 5",
    r#"
        Sub
          Numeric 3.
          Numeric 5.
      "#,
    false,
  );
}

#[test]
fn test_unary_tests() {
  let scope = &mut Scope::default();
  accept(
    scope,
    StartUnaryTests,
    "1",
    r#"
        ExpressionList
          Numeric 1.
      "#,
    false,
  );
  accept(
    scope,
    StartUnaryTests,
    "1,2",
    r#"
        ExpressionList
          Numeric 1.
          Numeric 2.
      "#,
    false,
  );
  accept(
    scope,
    StartUnaryTests,
    "not(1)",
    r#"
        NegatedList
          ExpressionList
            Numeric 1.
      "#,
    false,
  );
  accept(
    scope,
    StartUnaryTests,
    "not(1,2)",
    r#"
        NegatedList
          ExpressionList
            Numeric 1.
            Numeric 2.
      "#,
    false,
  );
  accept(
    scope,
    StartUnaryTests,
    "-",
    r#"
        Irrelevant
      "#,
    false,
  );
}

#[test]
fn test_boxed_expression() {
  // accept(StartBoxedExpression, "[]");
  // accept(StartBoxedExpression, "[1]");
  // accept(StartBoxedExpression, "[1,2,3]");
  // accept(StartBoxedExpression, "{}");

  // accept(&[StartContext, LeftBrace, RightBrace, YyEof]);
  // accept(&[StartContext, LeftBrace, Name, Colon, Numeric, RightBrace, YyEof]);
  // accept(&[StartContext, LeftBrace, Name, Colon, Numeric, Comma, Name, Colon, String, RightBrace, YyEof]);
  // accept(&[StartTextualExpression, Numeric, YyEof]);
  // accept(&[StartTextualExpression, LeftParen, Numeric, RightParen, YyEof]);
  // accept(&[StartTextualExpression, Numeric, Plus, Numeric, YyEof]);
  // accept(&[StartTextualExpression, Numeric, Plus, Minus, Numeric, YyEof]);
  // accept(&[StartTextualExpression, LeftParen, LeftBrace, RightBrace, RightParen, YyEof]);
  // accept(&[StartTextualExpressions, Numeric, YyEof]);
  // accept(&[StartTextualExpressions, Numeric, Comma, Numeric, YyEof]);
  // accept(&[StartUnaryTests, Minus, YyEof]);
  // accept(&[StartUnaryTests, Numeric, YyEof]);
  // accept(&[StartUnaryTests, Numeric, Comma, Boolean, YyEof]);
  // accept(&[StartUnaryTests, Numeric, Comma, Boolean, Comma, String, YyEof]);
  // accept(&[StartUnaryTests, Not, LeftParen, Numeric, Comma, Boolean, Comma, String, RightParen, YyEof]);

  // accept(&[StartTextualExpression, Numeric, YyEof]);
  // accept(&[StartTextualExpression, LeftParen, Numeric, RightParen, YyEof]);
  // accept(&[StartTextualExpression, Numeric, Plus, Numeric, YyEof]);
  // accept(&[StartTextualExpression, Numeric, Plus, Minus, Numeric, YyEof]);
  // accept(&[StartTextualExpression, LeftParen, LeftBrace, RightBrace, RightParen, YyEof]);
  // accept(&[StartTextualExpressions, Numeric, YyEof]);
  // accept(&[StartTextualExpressions, Numeric, Comma, Numeric, YyEof]);
  // accept(&[StartUnaryTests, Minus, YyEof]);
  // accept(&[StartUnaryTests, Numeric, YyEof]);
  // accept(&[StartUnaryTests, Numeric, Comma, Boolean, YyEof]);
  // accept(&[StartUnaryTests, Numeric, Comma, Boolean, Comma, String, YyEof]);
  // accept(&[StartUnaryTests, Not, LeftParen, Numeric, Comma, Boolean, Comma, String, RightParen, YyEof]);
}
