/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL parser.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Implementation of the `LALR` parser for `FEEL` grammar.

use dmntk_feel::context::FeelContext;
use dmntk_feel::values::VALUE_NULL;
use dmntk_feel::FeelType;
use {
  self::errors::*,
  crate::{
    lalr::{TokenType, *},
    lexer::{Lexer, TokenValue},
  },
  dmntk_common::Result,
  dmntk_feel::{AstNode, Name, Scope},
};

#[derive(Debug, Clone)]
enum Action {
  Accept,
  NewState,
  Default,
  Shift,
  Reduce,
  Error,
  Error1,
}

macro_rules! trace {
  ($s:tt, $fmt:expr, $($arg:tt)*) => {
    if $s.yy_trace { println!($fmt, $($arg)*); }
  };
  ($s:tt, $fmt:expr) => {
    if $s.yy_trace { println!($fmt); }
  };
}

macro_rules! trace_action {
  ($s:tt, $msg:literal) => {
    if $s.yy_trace {
      println!("  action: [\u{001b}[32m{}\u{001b}[0m]", $msg);
      println!("  state_stack={:?}", $s.yy_state_stack);
      println!("  value_stack={:?}", $s.yy_value_stack);
      println!("  node_stack={:?}", $s.yy_node_stack);
    }
  };
}

/// Parser.
pub struct Parser<'parser> {
  /// Parsing scope.
  scope: &'parser Scope,
  /// Flag indicating whether the tracing messages should be printed to standard output.
  yy_trace: bool,
  /// The FEEL [Lexer] used by this FEEL [Parser] as an input token stream.
  yy_lexer: Lexer<'parser>,
  /// The lookahead token type returned by lexer.
  yy_char: i16,
  /// The lookahead semantic value associated with token type returned by lexer.
  yy_value: TokenValue,
  ///
  yy_token: i16,
  /// Current state.
  yy_state: usize,
  /// This is an all purpose variable, it may represent a state number or a rule number.
  yy_n: i16,
  /// The number of symbols on the RHS of the reduced rule, keep to zero when no symbol should be popped.
  yy_len: i16,
  /// State stack.
  yy_state_stack: Vec<usize>,
  /// Semantic value stack.
  yy_value_stack: Vec<TokenValue>,
  /// AST node stack.
  yy_node_stack: Vec<AstNode>,
}

impl<'parser> Parser<'parser> {
  /// Creates a new parser.
  pub fn new(scope: &'parser Scope, start_token_type: TokenType, input: &str, trace: bool) -> Self {
    let lexer = Lexer::new(scope, start_token_type, input);
    Self {
      scope,
      yy_trace: trace,
      yy_lexer: lexer,
      yy_char: TokenType::YyEmpty as i16,
      yy_value: TokenValue::YyEmpty,
      yy_token: TokenType::YyEmpty as i16,
      yy_state: 0,
      yy_n: 0,
      yy_len: 0,
      yy_state_stack: vec![0],
      yy_value_stack: vec![TokenValue::YyEmpty],
      yy_node_stack: vec![],
    }
  }
  /// Parses the input.
  pub fn parse(&mut self) -> Result<AstNode> {
    let mut action = Action::NewState;
    loop {
      match action {
        Action::NewState => {
          trace!(self, "\nNEW-STATE: {}", self.yy_state);
          // if the final state then accept
          if self.yy_state == YY_FINAL {
            action = Action::Accept;
            continue;
          }
          // first try to decide what to do without reference to lookahead token
          self.yy_n = YY_PACT[self.yy_state];
          if self.yy_n == YY_PACT_N_INF {
            // process the default action
            action = Action::Default;
            continue;
          }
          // not known, so get a lookahead token if don't already have one
          if self.yy_char == TokenType::YyEmpty as i16 {
            let (token_type, opt_token_value) = self.yy_lexer.get_next_token()?;
            self.yy_char = token_type as i16;
            self.yy_token = SymbolKind::YyEmpty as i16;
            self.yy_value = opt_token_value;
            trace!(self, "  lexer: yy_char={}", self.yy_char);
            trace!(self, "  lexer: yy_value={:?}", self.yy_value);
          }
          trace!(self, "  yy_char={}", self.yy_char);
          if self.yy_char <= TokenType::YyEof as i16 {
            self.yy_char = TokenType::YyEof as i16;
            self.yy_token = SymbolKind::YyEof as i16;
            trace!(self, "  Now at end of input.");
          } else if self.yy_char == TokenType::YyError as i16 {
            self.yy_char = TokenType::YyUndef as i16;
            self.yy_token = SymbolKind::YyUndef as i16;
            action = Action::Error1;
            continue;
          } else {
            self.yy_token = YY_TRANSLATE[self.yy_char as usize] as i16;
          }
          trace!(self, "  yy_token={}", self.yy_token);
          trace!(self, "  state_stack={:?}", self.yy_state_stack);
          trace!(self, "  value_stack={:?}", self.yy_value_stack);
          trace!(self, "  node_stack={:?}", self.yy_node_stack);
          //
          let yy_token_code = self.yy_token as i16;
          self.yy_n += yy_token_code;
          if self.yy_n < 0 || YY_LAST < self.yy_n || YY_CHECK[self.yy_n as usize] != yy_token_code {
            action = Action::Default;
            continue;
          }
          self.yy_n = YY_TABLE[self.yy_n as usize];
          if self.yy_n <= 0 {
            if self.yy_n == YY_TABLE_N_INF {
              action = Action::Error;
            } else {
              self.yy_n = -self.yy_n;
              action = Action::Reduce;
            }
          } else {
            action = Action::Shift;
          }
        }
        Action::Default => {
          trace!(self, "\nDEFAULT");
          self.yy_n = YY_DEF_ACT[self.yy_state] as i16;
          if self.yy_n == 0 {
            action = Action::Error;
          } else {
            trace!(self, "  reduce_using_rule = {}", self.yy_n);
            action = Action::Reduce;
          }
        }
        Action::Shift => {
          trace!(self, "\nSHIFT");
          self.yy_state = self.yy_n as usize;
          self.yy_state_stack.push(self.yy_state);
          self.yy_value_stack.push(self.yy_value.clone());
          trace!(self, "  state_stack={:?}", self.yy_state_stack);
          trace!(self, "  value_stack={:?}", self.yy_value_stack);
          trace!(self, "  node_stack={:?}", self.yy_node_stack);
          self.yy_char = TokenType::YyEmpty as i16;
          self.yy_value = TokenValue::YyEmpty;
          action = Action::NewState;
        }
        Action::Reduce => {
          trace!(self, "\nREDUCE");
          // get the length of the right-hand side
          self.yy_len = YY_R2[self.yy_n as usize] as i16;
          trace!(self, "  reduce count = {}", self.yy_len);
          // yy_n is the number of a rule to reduce with
          trace!(self, "  --------------------------------------------");
          trace!(self, "  reducing_using_rule = {}", self.yy_n);
          crate::lalr::reduce(self, self.yy_n)?;
          trace!(self, "  --------------------------------------------");
          // pop the state stack and semantic value stack
          for _ in 0..self.yy_len {
            self.yy_state_stack.pop();
            self.yy_value_stack.pop();
          }
          // keep yy_len = 0
          self.yy_len = 0;
          let yy_lhs = (YY_R1[self.yy_n as usize] as usize) - (YY_N_TOKENS as usize);
          let top_state = self.yy_state_stack[self.yy_state_stack.len() - 1] as i16;
          let yy_i = YY_P_GOTO[yy_lhs] + top_state;
          // calculate the new state number
          self.yy_state = if (0..=YY_LAST).contains(&yy_i) && YY_CHECK[yy_i as usize] == top_state {
            YY_TABLE[yy_i as usize] as usize
          } else {
            YY_DEF_GOTO[yy_lhs] as usize
          };
          trace!(self, "  new_state = {}", self.yy_state);
          // push the new state on the stack
          self.yy_state_stack.push(self.yy_state);
          self.yy_value_stack.push(TokenValue::YyState(self.yy_state));
          trace!(self, "  state_stack={:?}", self.yy_state_stack);
          trace!(self, "  value_stack={:?}", self.yy_value_stack);
          trace!(self, "  node_stack={:?}", self.yy_node_stack);
          action = Action::NewState;
        }
        Action::Error => {
          trace!(self, "\nERROR");
          self.yy_token = SymbolKind::YyError as i16;
          return Err(syntax_error());
        }
        Action::Error1 => {
          trace!(self, "\nERROR 1");
          return Err(syntax_error());
        }
        Action::Accept => {
          trace!(self, "\n**********");
          trace!(self, "* ACCEPT *");
          trace!(self, "**********\n");
          self.yy_token = SymbolKind::YyAccept as i16;
          if let Some(node) = self.yy_node_stack.pop() {
            if self.yy_node_stack.is_empty() {
              if self.yy_trace {
                node.trace();
              }
              return Ok(node);
            }
          }
          return Err(invalid_parse_result());
        }
      }
    }
  }
  //************************************************************************************************
  // ACTIONS
  //************************************************************************************************
  #[inline(always)]
  pub fn action_addition(&mut self) -> Result<()> {
    trace_action!(self, "addition");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Add(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_between(&mut self) -> Result<()> {
    trace_action!(self, "between");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(mhs) = self.yy_node_stack.pop() {
        if let Some(lhs) = self.yy_node_stack.pop() {
          self.yy_node_stack.push(AstNode::Between(Box::new(lhs), Box::new(mhs), Box::new(rhs)));
        }
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_between_begin(&mut self) -> Result<()> {
    trace_action!(self, "between_begin");
    self.yy_lexer.set_between_begin();
    Ok(())
  }

  #[inline(always)]
  pub fn action_expression_list_tail(&mut self) -> Result<()> {
    trace_action!(self, "expression_list_tail");
    if let Some(node) = self.yy_node_stack.pop() {
      if let AstNode::ExpressionList(mut items) = node {
        if let Some(item) = self.yy_node_stack.pop() {
          items.insert(0, item);
        }
        self.yy_node_stack.push(AstNode::ExpressionList(items));
      } else {
        self.yy_node_stack.push(AstNode::ExpressionList(vec![node]));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_eq(&mut self) -> Result<()> {
    trace_action!(self, "comparison equal");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Eq(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_gt(&mut self) -> Result<()> {
    trace_action!(self, "comparison greater than");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Gt(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_ge(&mut self) -> Result<()> {
    trace_action!(self, "comparison greater or equal");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Ge(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_in(&mut self) -> Result<()> {
    trace_action!(self, "comparison in");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::In(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_lt(&mut self) -> Result<()> {
    trace_action!(self, "comparison less than");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Lt(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_le(&mut self) -> Result<()> {
    trace_action!(self, "comparison less or equal");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Le(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_nq(&mut self) -> Result<()> {
    trace_action!(self, "comparison not equal");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Nq(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_unary_ge(&mut self) -> Result<()> {
    trace_action!(self, "comparison_unary_ge");
    if let Some(lhs) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::UnaryGe(Box::new(lhs)));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_unary_gt(&mut self) -> Result<()> {
    trace_action!(self, "comparison_unary_gt");
    if let Some(lhs) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::UnaryGt(Box::new(lhs)));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_unary_lt(&mut self) -> Result<()> {
    trace_action!(self, "comparison_unary_lt");
    if let Some(lhs) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::UnaryLt(Box::new(lhs)));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_comparison_unary_le(&mut self) -> Result<()> {
    trace_action!(self, "comparison_unary_le");
    if let Some(lhs) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::UnaryLe(Box::new(lhs)));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_conjunction(&mut self) -> Result<()> {
    trace_action!(self, "conjunction");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::And(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_context_begin(&mut self) -> Result<()> {
    trace_action!(self, "context_begin");
    self.yy_lexer.push_to_scope();
    Ok(())
  }

  #[inline(always)]
  pub fn action_context_end(&mut self) -> Result<()> {
    trace_action!(self, "context_end");
    self.yy_lexer.pop_from_scope();
    Ok(())
  }

  #[inline(always)]
  pub fn action_context_entry(&mut self) -> Result<()> {
    trace_action!(self, "context_entry");
    if let Some(value_node) = self.yy_node_stack.pop() {
      if let Some(key_node) = self.yy_node_stack.pop() {
        if let AstNode::ContextEntryKey(name) = &key_node {
          self.yy_lexer.add_name_to_scope(&name);
        }
        self.yy_node_stack.push(AstNode::ContextEntry(Box::new(key_node), Box::new(value_node)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_context_entry_tail(&mut self) -> Result<()> {
    trace_action!(self, "context_entry_tail");
    if let Some(node) = self.yy_node_stack.pop() {
      if let AstNode::Context(mut items) = node {
        if let Some(item) = self.yy_node_stack.pop() {
          items.insert(0, item);
          self.yy_node_stack.push(AstNode::Context(items));
        }
      } else {
        self.yy_node_stack.push(AstNode::Context(vec![node]));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_context_type_entry(&mut self) -> Result<()> {
    trace_action!(self, "context_type_entry");
    if let Some(type_node) = self.yy_node_stack.pop() {
      let entry_type = type_node.type_of(self.yy_lexer.scope());
      if let TokenValue::Name(name) = &self.yy_value_stack[self.yy_value_stack.len() - self.yy_len as usize] {
        let entry_name = AstNode::Name(name.clone());
        self.yy_node_stack.push(AstNode::ContextTypeEntry(Box::new(entry_name), entry_type));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_context_type_entry_tail(&mut self) -> Result<()> {
    trace_action!(self, "context_type_entry_tail");
    if let Some(node) = self.yy_node_stack.pop() {
      if let AstNode::ContextType(mut items) = node {
        if let Some(item) = self.yy_node_stack.pop() {
          items.insert(0, item);
          self.yy_node_stack.push(AstNode::ContextType(items));
        }
      } else {
        self.yy_node_stack.push(AstNode::ContextType(vec![node]));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_unary_tests_begin(&mut self) -> Result<()> {
    trace_action!(self, "unary_tests_begin");
    self.yy_lexer.set_unary_tests();
    Ok(())
  }

  #[inline(always)]
  pub fn action_disjunction(&mut self) -> Result<()> {
    trace_action!(self, "disjunction");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Or(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_division(&mut self) -> Result<()> {
    trace_action!(self, "division");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Div(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_every(&mut self) -> Result<()> {
    trace_action!(self, "every");
    Ok(())
  }

  #[inline(always)]
  pub fn action_exponentiation(&mut self) -> Result<()> {
    trace_action!(self, "exponentiation");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Exp(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_filter(&mut self) -> Result<()> {
    trace_action!(self, "filter");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::FilterExpression(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  /// Reduces the function definition.
  /// The top element on the `node stack` is the AstNode defining the function body.
  /// The AstNode just below the function's body is the list of function's formal parameters.
  #[inline(always)]
  pub fn action_function_definition(&mut self) -> Result<()> {
    trace_action!(self, "function_definition");
    if let Some(body) = self.yy_node_stack.pop() {
      if let Some(parameters) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::FunctionDefinition(Box::new(parameters), Box::new(body)));
      }
    }
    Ok(())
  }

  /// Reduces the definition of the function body. This function body is **not** external.
  /// The content of the function body is the AstNode on the top of the `node stack`.
  /// After reducing the function body, the top context from scope is popped,
  /// because this context is temporary and contains the function parameter names
  /// to be properly interpreted while parsing the function's body.
  #[inline(always)]
  pub fn action_function_body(&mut self) -> Result<()> {
    trace_action!(self, "function_body");
    if let Some(function_body_node) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::FunctionBody(Box::new(function_body_node), false));
    }
    // pop the temporary context from the top of scope stack
    self.scope.pop();
    Ok(())
  }

  #[inline(always)]
  pub fn action_function_body_external(&mut self) -> Result<()> {
    trace_action!(self, "function_body_external");
    self.scope.pop();
    Ok(())
  }

  #[inline(always)]
  pub fn action_for(&mut self) -> Result<()> {
    trace_action!(self, "for");
    Ok(())
  }

  #[inline(always)]
  pub fn action_formal_parameters_begin(&mut self) -> Result<()> {
    trace_action!(self, "function_formal_parameters_begin");
    // when the list of formal parameters begins, push a local context onto scope stack
    self.scope.push(FeelContext::default());
    Ok(())
  }

  #[inline(always)]
  pub fn action_formal_parameter_with_type(&mut self) -> Result<()> {
    trace_action!(self, "function_formal_parameter_with_type");
    // the type of the parameter is on top of node stack
    if let Some(rhs) = self.yy_node_stack.pop() {
      // the name of the parameter is in value stack
      if let TokenValue::Name(name) = &self.yy_value_stack[self.yy_value_stack.len() - self.yy_len as usize] {
        // push the new formal parameter on top of node stack
        self.yy_node_stack.push(AstNode::FormalParameter(name.clone(), rhs.type_of(self.scope)));
        // set the name of the parameter to local context on the top of the scope stack
        // this name will be properly interpreted as a name while parsing the function body
        self.scope.set_entry(name, VALUE_NULL);
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_formal_parameter_without_type(&mut self) -> Result<()> {
    trace_action!(self, "function_formal_parameter_without_type");
    // the name of the parameter is in value stack
    if let TokenValue::Name(name) = &self.yy_value_stack[self.yy_value_stack.len() - self.yy_len as usize] {
      // push the new formal parameter on top of node stack
      self.yy_node_stack.push(AstNode::FormalParameter(name.clone(), FeelType::Any));
      // set the name of the parameter to local context on the top of the scope stack
      // this name will be properly interpreted as a name while parsing the function body
      self.scope.set_entry(name, VALUE_NULL);
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_formal_parameters_empty(&mut self) -> Result<()> {
    trace_action!(self, "function_formal_parameters_empty");
    // push the empty list of formal parameters onto the node stack
    self.yy_node_stack.push(AstNode::FormalParameters(vec![]));
    Ok(())
  }

  #[inline(always)]
  pub fn action_formal_parameters_first(&mut self) -> Result<()> {
    trace_action!(self, "function_formal_parameters_first");
    // the first parameter is on the top of the node stack
    if let Some(formal_parameter_node) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::FormalParameters(vec![formal_parameter_node]));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_formal_parameters_tail(&mut self) -> Result<()> {
    trace_action!(self, "function_formal_parameters_tail");
    // the next parameter is on the top of the node stack
    if let Some(formal_parameter_node) = self.yy_node_stack.pop() {
      // the collection of formal parameters is now on top of the node stack
      if let Some(AstNode::FormalParameters(mut items)) = self.yy_node_stack.pop() {
        items.push(formal_parameter_node);
        self.yy_node_stack.push(AstNode::FormalParameters(items));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_function_invocation(&mut self) -> Result<()> {
    trace_action!(self, "function_invocation");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::FunctionInvocation(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_function_invocation_no_parameters(&mut self) -> Result<()> {
    trace_action!(self, "function_invocation_no_parameters");
    if let Some(lhs) = self.yy_node_stack.pop() {
      let rhs = AstNode::PositionalParameters(vec![]);
      self.yy_node_stack.push(AstNode::FunctionInvocation(Box::new(lhs), Box::new(rhs)));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_function_type(&mut self) -> Result<()> {
    trace_action!(self, "function_type");
    if let Some(result_type) = self.yy_node_stack.pop() {
      if let Some(parameter_types) = self.yy_node_stack.pop() {
        let feel_type = result_type.type_of(self.yy_lexer.scope());
        self.yy_node_stack.push(AstNode::FunctionType(Box::new(parameter_types), feel_type));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_function_type_parameters_empty(&mut self) -> Result<()> {
    trace_action!(self, "function_type_parameters_empty");
    self.yy_node_stack.push(AstNode::ParameterTypes(vec![]));
    Ok(())
  }

  #[inline(always)]
  pub fn action_function_type_parameters_tail(&mut self) -> Result<()> {
    trace_action!(self, "function_type_parameters_tail");
    if let Some(node) = self.yy_node_stack.pop() {
      if let AstNode::ParameterTypes(mut items) = node {
        if let Some(item) = self.yy_node_stack.pop() {
          items.insert(0, item.type_of(self.yy_lexer.scope()));
          self.yy_node_stack.push(AstNode::ParameterTypes(items));
        }
      } else {
        self.yy_node_stack.push(AstNode::ParameterTypes(vec![node.type_of(self.yy_lexer.scope())]));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_if(&mut self) -> Result<()> {
    trace_action!(self, "if");
    Ok(())
  }

  #[inline(always)]
  pub fn action_instance_of(&mut self) -> Result<()> {
    trace_action!(self, "instance_of");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::InstanceOf(Box::new(lhs), rhs.type_of(self.yy_lexer.scope())));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_interval(&mut self) -> Result<()> {
    trace_action!(self, "interval");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Range(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_interval_start(&mut self) -> Result<()> {
    trace_action!(self, "interval_start");
    let closed = matches!(&self.yy_value_stack[self.yy_value_stack.len() - self.yy_len as usize], TokenValue::LeftBracket);
    if let Some(lhs) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::IntervalStart(Box::new(lhs), closed));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_interval_end(&mut self) -> Result<()> {
    trace_action!(self, "interval_end");
    let closed = matches!(&self.yy_value_stack[self.yy_value_stack.len() - 1], TokenValue::RightBracket);
    if let Some(lhs) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::IntervalEnd(Box::new(lhs), closed));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_list(&mut self) -> Result<()> {
    trace_action!(self, "list");
    if let Some(AstNode::CommaList(items)) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::List(items));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_list_empty(&mut self) -> Result<()> {
    trace_action!(self, "list_empty");
    self.yy_node_stack.push(AstNode::CommaList(vec![]));
    Ok(())
  }

  #[inline(always)]
  pub fn action_list_tail(&mut self) -> Result<()> {
    trace_action!(self, "list tail");
    if let Some(node) = self.yy_node_stack.pop() {
      if let AstNode::CommaList(mut items) = node {
        if let Some(item) = self.yy_node_stack.pop() {
          items.insert(0, item);
          self.yy_node_stack.push(AstNode::CommaList(items));
        }
      } else {
        self.yy_node_stack.push(AstNode::CommaList(vec![node]));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_list_type(&mut self) -> Result<()> {
    trace_action!(self, "list_type");
    if let Some(rhs) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::ListType(rhs.type_of(self.yy_lexer.scope())));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_empty_context(&mut self) -> Result<()> {
    trace_action!(self, "empty context");
    self.yy_node_stack.push(AstNode::Context(vec![]));
    Ok(())
  }

  #[inline(always)]
  pub fn action_key_name(&mut self) -> Result<()> {
    trace_action!(self, "key_name");
    if let Some(TokenValue::Name(name)) = self.yy_value_stack.last() {
      self.yy_node_stack.push(AstNode::ContextEntryKey(name.clone()));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_key_string(&mut self) -> Result<()> {
    trace_action!(self, "key_string");
    if let Some(TokenValue::String(value)) = self.yy_value_stack.last() {
      self.yy_node_stack.push(AstNode::ContextEntryKey(Name::from(value.clone())));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_literal_at(&mut self) -> Result<()> {
    trace_action!(self, "literal_at");
    if let Some(TokenValue::String(value)) = self.yy_value_stack.last() {
      self.yy_node_stack.push(AstNode::At(value.clone()));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_literal_boolean(&mut self) -> Result<()> {
    trace_action!(self, "literal_boolean");
    if let Some(TokenValue::Boolean(value)) = self.yy_value_stack.last() {
      self.yy_node_stack.push(AstNode::Boolean(*value));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_literal_date_time(&mut self) -> Result<()> {
    trace_action!(self, "literal_date_time");
    if let TokenValue::NameDateTime(name) = &self.yy_value_stack[self.yy_value_stack.len() - 2] {
      self.yy_node_stack.push(AstNode::Name(name.clone()));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_literal_null(&mut self) -> Result<()> {
    trace_action!(self, "literal_null");
    if let Some(TokenValue::Null) = self.yy_value_stack.last() {
      self.yy_node_stack.push(AstNode::Null);
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_literal_numeric(&mut self) -> Result<()> {
    trace_action!(self, "numeric_literal");
    if let Some(TokenValue::Numeric(before, after)) = self.yy_value_stack.last() {
      self.yy_node_stack.push(AstNode::Numeric(before.clone(), after.clone()));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_literal_string(&mut self) -> Result<()> {
    trace_action!(self, "string_literal");
    if let Some(TokenValue::String(value)) = self.yy_value_stack.last() {
      self.yy_node_stack.push(AstNode::String(value.clone()));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_multiplication(&mut self) -> Result<()> {
    trace_action!(self, "multiplication");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Mul(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_name(&mut self) -> Result<()> {
    trace_action!(self, "name");
    if let Some(TokenValue::Name(value)) = self.yy_value_stack.last() {
      self.yy_node_stack.push(AstNode::Name(value.clone()));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_name_date_time(&mut self) -> Result<()> {
    trace_action!(self, "name_date_time");
    if let Some(TokenValue::NameDateTime(value)) = self.yy_value_stack.last() {
      self.yy_node_stack.push(AstNode::Name(value.clone()));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_named_parameter(&mut self) -> Result<()> {
    trace_action!(self, "named_parameter");
    trace!(self, "{:?}", self.yy_value_stack);
    if let TokenValue::Name(name) = &self.yy_value_stack[self.yy_value_stack.len() - 3] {
      if let Some(rhs) = self.yy_node_stack.pop() {
        let lhs = AstNode::ParameterName(name.clone());
        self.yy_node_stack.push(AstNode::NamedParameter(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_named_parameters_tail(&mut self) -> Result<()> {
    trace_action!(self, "named_parameters_tail");
    if let Some(node) = self.yy_node_stack.pop() {
      if let AstNode::NamedParameters(mut items) = node {
        if let Some(item) = self.yy_node_stack.pop() {
          items.insert(0, item);
          self.yy_node_stack.push(AstNode::NamedParameters(items));
        }
      } else {
        self.yy_node_stack.push(AstNode::NamedParameters(vec![node]));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_negation(&mut self) -> Result<()> {
    trace_action!(self, "negation");
    if let Some(node) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::Neg(Box::new(node)));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_path(&mut self) -> Result<()> {
    trace_action!(self, "path");
    if let Some(lhs) = self.yy_node_stack.pop() {
      if let Some(TokenValue::Name(name)) = &self.yy_value_stack.last() {
        let rhs = AstNode::Name(name.clone());
        self.yy_node_stack.push(AstNode::Path(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_positional_parameters_tail(&mut self) -> Result<()> {
    trace_action!(self, "positional_parameters_tail");
    if let Some(node) = self.yy_node_stack.pop() {
      if let AstNode::PositionalParameters(mut items) = node {
        if let Some(item) = self.yy_node_stack.pop() {
          items.insert(0, item);
          self.yy_node_stack.push(AstNode::PositionalParameters(items));
        }
      } else {
        self.yy_node_stack.push(AstNode::PositionalParameters(vec![node]));
      }
    }
    Ok(())
  }

  // #[inline(always)]
  // pub fn action_positive_unary_tests_tail(&mut self) -> Result<()> {
  //   trace_action!(self, "positive_unary_tests_tail");
  //   if let Some(node) = self.yy_node_stack.pop() {
  //     if let AstNode::List(mut items) = node {
  //       if let Some(item) = self.yy_node_stack.pop() {
  //         items.insert(0, item);
  //       }
  //       self.yy_node_stack.push(AstNode::List(items));
  //     } else {
  //       self.yy_node_stack.push(AstNode::List(vec![node]));
  //     }
  //   }
  //   Ok(())
  // }

  #[inline(always)]
  pub fn action_range_type(&mut self) -> Result<()> {
    trace_action!(self, "range_type");
    if let Some(rhs) = self.yy_node_stack.pop() {
      self.yy_node_stack.push(AstNode::RangeType(rhs.type_of(self.yy_lexer.scope())));
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_some(&mut self) -> Result<()> {
    trace_action!(self, "some");
    Ok(())
  }

  #[inline(always)]
  pub fn action_subtraction(&mut self) -> Result<()> {
    trace_action!(self, "subtraction");
    if let Some(rhs) = self.yy_node_stack.pop() {
      if let Some(lhs) = self.yy_node_stack.pop() {
        self.yy_node_stack.push(AstNode::Sub(Box::new(lhs), Box::new(rhs)));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_type_name(&mut self) -> Result<()> {
    trace_action!(self, "type_name");
    self.yy_lexer.set_type_name();
    Ok(())
  }

  // #[inline(always)]
  // pub fn action_unary_tests(&mut self) -> Result<()> {
  //   trace_action!(self, "unary_tests");
  //   if let Some(node) = self.yy_node_stack.pop() {
  //     if let AstNode::List(mut items) = node {
  //       if let Some(item) = self.yy_node_stack.pop() {
  //         items.insert(0, item);
  //       }
  //       self.yy_node_stack.push(AstNode::List(items));
  //     } else {
  //       self.yy_node_stack.push(AstNode::List(vec![node]));
  //     }
  //   }
  //   Ok(())
  // }

  #[inline(always)]
  pub fn action_unary_tests_negated(&mut self) -> Result<()> {
    trace_action!(self, "unary_tests_negated");
    if let TokenValue::Name(name) = &self.yy_value_stack[self.yy_value_stack.len() - self.yy_len as usize] {
      let keyword = name.to_string();
      if keyword != "not" {
        return Err(negated_unary_tests_require_not_keyword(keyword));
      }
    }
    if let Some(node) = self.yy_node_stack.pop() {
      if let AstNode::List(mut items) = node {
        if let Some(item) = self.yy_node_stack.pop() {
          items.insert(0, item);
        }
        self.yy_node_stack.push(AstNode::NegatedList(items));
      } else {
        self.yy_node_stack.push(AstNode::NegatedList(vec![node]));
      }
    }
    Ok(())
  }

  #[inline(always)]
  pub fn action_unary_tests_irrelevant(&mut self) -> Result<()> {
    trace_action!(self, "unary_tests_irrelevant");
    self.yy_node_stack.push(AstNode::Irrelevant);
    Ok(())
  }
}

/// Definitions of errors raised by the parser.
pub mod errors {
  use dmntk_common::DmntkError;

  /// Parser errors.
  #[derive(Debug, PartialEq)]
  enum ParserError {
    InvalidParseResult,
    SyntaxError,
    NegatedUnaryTestsRequireNotKeyword(String),
  }

  impl From<ParserError> for DmntkError {
    fn from(e: ParserError) -> Self {
      DmntkError::new("ParserError", &format!("{}", e))
    }
  }

  impl std::fmt::Display for ParserError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        ParserError::InvalidParseResult => {
          write!(f, "invalid parse result, expected non empty AST node as a result when parser accepts input")
        }
        ParserError::SyntaxError => {
          write!(f, "syntax error")
        }
        ParserError::NegatedUnaryTestsRequireNotKeyword(s) => {
          write!(f, "negated unary tests require `not` keyword before opening left paren, found `{}`", s)
        }
      }
    }
  }

  pub fn invalid_parse_result() -> DmntkError {
    ParserError::InvalidParseResult.into()
  }

  pub fn syntax_error() -> DmntkError {
    ParserError::SyntaxError.into()
  }

  pub fn negated_unary_tests_require_not_keyword(s: String) -> DmntkError {
    ParserError::NegatedUnaryTestsRequireNotKeyword(s).into()
  }
}
