/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL parser.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use {
  super::{accept, empty_scope},
  crate::{lalr::TokenType::StartTextualExpression, scope::ParsingScope},
};

/// Utility function that compares the given name with expected name after parsing.
fn accept_name(scope: &ParsingScope, name: &str, expected: &str) {
  accept(
    scope,
    StartTextualExpression,
    name,
    &format!(
      r#"
        Name
          `{}`
      "#,
      expected
    ),
    false,
  );
}

#[test]
fn test_name_1() {
  let scope = &mut empty_scope();
  scope.add_name("hello world");
  accept(
    scope,
    StartTextualExpression,
    "hello world",
    r#"
        Name
          `hello world`
      "#,
    false,
  );
}

#[test]
fn test_name_2() {
  let scope = &mut empty_scope();
  scope.add_name("  \n  \n  \t  thing \n \t \t ");
  accept(
    scope,
    StartTextualExpression,
    "thing",
    r#"
        Name
          `thing`
      "#,
    false,
  );
}

#[test]
fn test_name_3() {
  let scope = &mut empty_scope();
  scope.add_name("income/loss");
  accept(
    scope,
    StartTextualExpression,
    "income/loss",
    r#"
        Name
          `income / loss`
      "#,
    false,
  );
}

#[test]
fn test_name_4() {
  let scope = &mut empty_scope();
  scope.add_name("fr**n*s");
  accept(
    scope,
    StartTextualExpression,
    "fr**n*s",
    r#"
        Name
          `fr * * n * s`
      "#,
    false,
  );
}

#[test]
fn test_name_5() {
  let scope = &mut empty_scope();
  accept(
    scope,
    StartTextualExpression,
    "book",
    r#"
        Name
          `book`
      "#,
    false,
  );
}

#[test]
fn test_name_6() {
  let scope = &mut empty_scope();
  accept(
    scope,
    StartTextualExpression,
    "profit/loss",
    r#"
        Name
          `profit / loss`
      "#,
    false,
  );
}

#[test]
fn test_name_7() {
  let scope = &mut empty_scope();
  scope.add_name("before.after");
  accept(
    scope,
    StartTextualExpression,
    "before.after",
    r#"
        Name
          `before . after`
      "#,
    false,
  );
}

#[test]
fn test_name_multiple() {
  let scope = &mut empty_scope();
  scope.add_name("thing");
  scope.add_name("one two three four");
  scope.add_name("one and two");
  scope.add_name("one or two");
  scope.add_name("before.after");
  scope.add_name("before.or.after");
  scope.add_name("before.and.after");
  scope.add_name("before.between.after");
  scope.add_name("before.next to between.worm");
  scope.add_name("income/loss");
  scope.add_name("per/month/income/loss");
  scope.add_name("a-b");
  scope.add_name("to-be-or-not-to-be");
  scope.add_name("that's");
  scope.add_name("ok that's");
  scope.add_name("bed+breakfast");
  scope.add_name("night+and+day");
  scope.add_name("fr**n*s");
  scope.add_name("bo++e--m*");
  scope.add_name("wh*t*v*r");
  accept_name(scope, "thing", "thing");
  accept_name(scope, "one two three four", "one two three four");
  accept_name(scope, "one and two", "one and two");
  accept_name(scope, "one or two", "one or two");
  accept_name(scope, "before.after", "before . after");
  accept_name(scope, "before.or.after", "before . or . after");
  accept_name(scope, "before.and.after", "before . and . after");
  accept_name(scope, "before.between.after", "before . between . after");
  accept_name(scope, "before.next to between.worm", "before . next to between . worm");
  accept_name(scope, "income/loss", "income / loss");
  accept_name(scope, "per/month/income/loss", "per / month / income / loss");
  accept_name(scope, "a-b", "a - b");
  accept_name(scope, "to-be-or-not-to-be", "to - be - or - not - to - be");
  accept_name(scope, "that's", "that ' s");
  accept_name(scope, "ok that's", "ok that ' s");
  accept_name(scope, "bed+breakfast", "bed + breakfast");
  accept_name(scope, "night+and+day", "night + and + day");
  accept_name(scope, "fr**n*s", "fr * * n * s");
  accept_name(scope, "bo++e--m*", "bo + + e - - m *");
  accept_name(scope, "wh*t*v*r", "wh * t * v * r");
}
