/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL and DMN model evaluator.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use crate::model::errors::*;
use dmntk_common::Result;
use dmntk_feel::context::FeelContext;
use dmntk_feel::values::{Value, Values};
use dmntk_feel::{Name, Scope};
use dmntk_model::model::{Definitions, ItemDefinition, ItemDefinitionType, NamedElement};
use std::collections::HashMap;

pub(crate) mod enricher;
mod errors;
pub(crate) mod eval;
pub(crate) mod eval_bkm;
pub(crate) mod eval_dec;
pub(crate) mod eval_dec_service;
pub(crate) mod eval_dec_table;
pub(crate) mod eval_input;

#[cfg(test)]
mod tests;

/// Evaluates the decision table.
pub fn evaluate_decision_table_from_text(scope: &Scope, input: &str) -> Result<Value> {
  crate::model::eval_dec_table::evaluate_decision_table(scope, dmntk_recognizer::build(&input)?)
}

/// Evaluates a decision table against specified context.
pub fn evaluate_decision_table_and_context(decision_table_input: &str, context_input: &str) -> Result<Value> {
  let scope: Scope = crate::evaluate_context(&Scope::default(), context_input)?.into();
  evaluate_decision_table_from_text(&scope, decision_table_input)
}

/// Evaluates all tests associated with decision table.
pub fn evaluate_decision_table_and_test(_input: &str, _sep: &str) -> Result<(bool, Value, Value)> {
  //FIXME implement this feature
  unimplemented!()
}

#[derive(Default, Debug, Clone)]
pub struct ItemDefinitions {
  item_definitions: HashMap<String, ItemDefinition>,
}

impl ItemDefinitions {
  /// Creates a new item definition collection from [Definition]s.
  pub fn new(definitions: &Definitions) -> Self {
    let mut item_definitions = HashMap::new();
    for item_definition in definitions.item_definitions() {
      item_definitions.insert(item_definition.name().to_owned(), item_definition.clone());
    }
    Self { item_definitions }
  }
  /// Retrieves a new value from provided value using item definition with specified name.
  pub fn retrieve_value_by_name(&self, name: &str, value: &Value) -> Result<Value> {
    if let Some(item_definition) = self.item_definitions.get(name) {
      self.retrieve_value(item_definition, value)
    } else {
      Err(item_definition_with_name_not_found(name))
    }
  }
  /// Retrieves a new value from provided value using specified item definition.
  fn retrieve_value(&self, item_definition: &ItemDefinition, value: &Value) -> Result<Value> {
    match item_definition.item_definition_type().as_ref().ok_or_else(invalid_item_definition_type)? {
      ItemDefinitionType::SimpleType(feel_type) => {
        return feel_type.retrieve_value_with_type_checking(value);
      }
      ItemDefinitionType::ReferencedType(ref_type_name) => {
        return self.retrieve_value_by_name(&ref_type_name, value);
      }
      ItemDefinitionType::ComponentType => {
        if let Value::Context(external_context) = value {
          let mut evaluated_context = FeelContext::default();
          for item_component in item_definition.item_components() {
            let name = Name::from(item_component.name());
            if let Some(external_value) = external_context.get_entry(&name) {
              let evaluated_value = self.retrieve_value(item_component, external_value)?;
              evaluated_context.set_entry(&name, evaluated_value);
            }
          }
          return Ok(Value::Context(evaluated_context));
        }
      }
      ItemDefinitionType::CollectionOfSimpleType(feel_type) => {
        if let Value::List(external_items) = value {
          let mut evaluated_items = vec![];
          for external_item in external_items.to_vec() {
            let evaluated_value = feel_type.retrieve_value_with_type_checking(external_item)?;
            evaluated_items.push(evaluated_value);
          }
          return Ok(Value::List(Values::new(evaluated_items)));
        }
      }
      ItemDefinitionType::CollectionOfReferencedType(ref_type_name) => {
        if let Value::List(external_items) = value {
          let mut evaluated_items = vec![];
          for external_value in external_items.to_vec() {
            let evaluated_value = self.retrieve_value_by_name(&ref_type_name, external_value)?;
            evaluated_items.push(evaluated_value);
          }
          return Ok(Value::List(Values::new(evaluated_items)));
        }
      }
      ItemDefinitionType::CollectionOfComponentType => {
        if let Value::List(external_items) = value {
          let mut evaluated_items = vec![];
          for external_item in external_items.to_vec() {
            if let Value::Context(external_context) = external_item {
              let mut evaluated_context = FeelContext::default();
              for item_component in item_definition.item_components() {
                let name: Name = item_component.name().into();
                if let Some(external_value) = external_context.get_entry(&name) {
                  let evaluated_value = self.retrieve_value(item_component, external_value)?;
                  evaluated_context.set_entry(&name, evaluated_value);
                }
              }
              evaluated_items.push(Value::Context(evaluated_context));
            }
          }
          return Ok(Value::List(Values::new(evaluated_items)));
        }
      }
    }
    Err(invalid_value_for_retrieving_using_item_definition())
  }
}
