/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL and DMN model evaluator.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use crate::model::errors::*;
use dmntk_common::Result;
use dmntk_feel::values::{Value, VALUE_NULL};
use dmntk_feel::Name;
use dmntk_model::model::{Definitions, DmnElement, InputData, NamedElement};
use std::collections::HashMap;

///
type NameValueExtractor = Box<dyn Fn(&Value) -> Result<(Name, Value)>>;

///
type ValueExtractor = Box<dyn Fn(&Value) -> Result<Value>>;

///
#[derive(Default)]
pub struct Evaluator {
  input_data_extractors: HashMap<String, NameValueExtractor>,
}

impl Evaluator {
  ///
  pub fn new(definitions: &Definitions) -> Result<Self> {
    let mut input_data_extractors = HashMap::new();
    for input_data in definitions.input_data() {
      let input_data_id = input_data.id().as_ref().ok_or_else(empty_identifier)?.clone();
      input_data_extractors.insert(input_data_id, build_extractor_from_input_data(input_data)?);
    }
    Ok(Self { input_data_extractors })
  }
  ///
  pub fn evaluate_input_data(&self, input_data_id: &str, value: &Value) -> Result<(Name, Value)> {
    // take an extractor function with specified identifier
    let extractor = self
      .input_data_extractors
      .get(input_data_id)
      .ok_or_else(|| input_data_with_id_not_found(input_data_id))?;
    // call an extractor and retrieve input data
    extractor(value)
  }
}

///
pub fn build_extractor_from_something(_input_data: &InputData) -> Result<ValueExtractor> {
  Ok(Box::new(move |_: &Value| Ok(VALUE_NULL)))
}

///
pub fn build_extractor_from_input_data(input_data: &InputData) -> Result<NameValueExtractor> {
  let type_ref = input_data.variable().type_ref().as_ref().ok_or_else(empty_type_ref)?.clone();
  let name = input_data.variable().feel_name().as_ref().ok_or_else(empty_feel_name)?.clone();
  Ok(match type_ref.as_str() {
    "string" => Box::new(move |value: &Value| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::String(_) = v {
            return Ok((name.clone(), v.clone()));
          }
        }
      }
      Ok((name.clone(), VALUE_NULL))
    }),
    "number" => Box::new(move |value: &Value| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::Number(_) = v {
            return Ok((name.clone(), v.clone()));
          }
        }
      }
      Ok((name.clone(), VALUE_NULL))
    }),
    "boolean" => Box::new(move |value: &Value| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::Boolean(_) = v {
            return Ok((name.clone(), v.clone()));
          }
        }
      }
      //Ok((name.clone(), VALUE_NULL))
      Err(empty_feel_name())
    }),
    //TODO add more basic types
    _ => {
      let mut a = vec![];
      a.push(build_extractor_from_something(input_data)?);
      a.push(build_extractor_from_something(input_data)?);
      a.push(build_extractor_from_something(input_data)?);
      Box::new(move |value: &Value| {
        for ve in &a {
          ve(&value)?;
        }
        Ok((name.clone(), VALUE_NULL))
      })
    }
  })
}

#[cfg(test)]
mod tests {
  use crate::model::eval::Evaluator;
  use dmntk_feel::context::FeelContext;
  use dmntk_feel::values::{Value, VALUE_NULL};
  use dmntk_feel::Name;

  #[test]
  pub fn test_build_extractor_from_input_data_string() {
    let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0001, "file:///2_0001.dmn").unwrap();
    let evaluator = Evaluator::new(&definitions).unwrap();
    let mut ctx: FeelContext = Default::default();
    let name_a = Name::new(&["Full", "Name"]);
    let value_a = Value::String("John".to_owned());
    ctx.set_entry(&name_a, value_a.clone());
    let value_ctx = Value::Context(ctx);
    let input_data_id = "_cba86e4d-e91c-46a2-9176-e9adf88e15db";
    assert_eq!((name_a, value_a), evaluator.evaluate_input_data(input_data_id, &value_ctx).unwrap());
  }

  #[test]
  pub fn test_build_extractor_from_input_data_number() {
    let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0002, "file:///2_0002.dmn").unwrap();
    let evaluator = Evaluator::new(&definitions).unwrap();
    let mut ctx: FeelContext = Default::default();
    let name_a = Name::new(&["Monthly", "Salary"]);
    let value_a = Value::Number(12000.0);
    ctx.set_entry(&name_a, value_a.clone());
    let value_ctx = Value::Context(ctx);
    let input_data_id = "i_MonthlySalary";
    assert_eq!((name_a, value_a), evaluator.evaluate_input_data(input_data_id, &value_ctx).unwrap());
  }

  #[test]
  pub fn test_build_extractor_from_input_data_error() {
    let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0001, "file:///2_0001.dmn").unwrap();
    let evaluator = Evaluator::new(&definitions).unwrap();
    let mut ctx: FeelContext = Default::default();
    let name_a = Name::new(&["Full", "Name"]);
    let value_a = Value::Number(50.0);
    ctx.set_entry(&name_a, value_a);
    let value_ctx = Value::Context(ctx);
    let input_data_id = "_cba86e4d-e91c-46a2-9176-e9adf88e15db";
    assert_eq!((name_a, VALUE_NULL), evaluator.evaluate_input_data(input_data_id, &value_ctx).unwrap());
  }
}
