use super::*;
use dmntk_feel::FeelTime;
use std::f64::consts::LN_10;
use std::ops::{Div, Mul};

#[test]
fn test_abs() {
  let scope = &te_scope("{ Order size: -4.5 }");
  te_number(false, scope, "abs(1)", 1.0);
  te_number(false, scope, "abs(-1)", 1.0);
  te_number(false, scope, "abs(n:-34)", 34.0);
  te_null(false, scope, "abs(number:-34)", "");
  te_number(false, scope, "abs(Order size)", 4.5);
}

#[test]
fn test_append() {
  let scope = &te_scope("{}");
  te_be_value(false, scope, r#"append([1],2,3)"#, r#"[1,2,3]"#);
}

#[test]
fn test_ceiling() {
  let scope = &te_scope("{ Order size: 23.27 }");
  te_number(false, scope, "ceiling(1.5)", 2.0);
  te_number(false, scope, "ceiling(-1.5)", -1.0);
  te_number(false, scope, "ceiling(--1)", 1.0);
  te_number(false, scope, "ceiling(-5/2.3*5)", -10.0);
  te_number(false, scope, "ceiling(n:5.777)", 6.0);
  te_number(false, scope, "ceiling(n:-.33333)", 0.0);
  te_number(false, scope, "ceiling(n:Order size)", 24.0);
  te_null(false, scope, "ceiling(number:5.777)", "");
}

#[test]
fn test_date() {
  let scope = &te_scope("{}");
  te_date(false, scope, r#"date("2012-12-25")"#, 2012, 12, 25);
  te_date(false, scope, r#"date(2012,12,25)"#, 2012, 12, 25);
  te_date(false, scope, r#"date("262143-12-31")"#, 262143, 12, 31);
  te_date(false, scope, r#"date("999999999-12-31")"#, 999_999_999, 12, 31);
  te_date(false, scope, r#"date(999999999,12,31)"#, 999_999_999, 12, 31);
  te_date(false, scope, r#"date("-262144-01-01")"#, -262144, 1, 1);
  te_date(false, scope, r#"date("-999999999-01-01")"#, -999_999_999, 1, 1);
  te_date(false, scope, r#"date(-999999999,01,01)"#, -999_999_999, 1, 1);
  te_date(false, scope, r#"date(date and time("2012-12-25T12:23:18"))"#, 2012, 12, 25);
  te_date(false, scope, r#"date(date and time("2012-12-25T12:23:18Z"))"#, 2012, 12, 25);
  te_date(false, scope, r#"date(date and time("2012-12-25T12:23:18z"))"#, 2012, 12, 25);
  te_bool(false, scope, r#"date("2012-12-25") in [date("2012-12-24")..date("2012-12-26")]"#, true);
  te_bool(false, scope, r#"date("2000-12-25") in [date("2012-12-24")..date("2012-12-26")]"#, false);
  te_bool(false, scope, r#"date("2020-12-25") in [date("2012-12-24")..date("2012-12-26")]"#, false);
  te_bool(false, scope, r#"date("2012-12-31") in (date("2012-12-25")..date("2013-02-14"))"#, true);
  te_null(false, scope, r#"date("2017-13-10")"#, "");
  te_null(false, scope, r#"date("2017-13-10")"#, "");
  te_null(false, scope, r#"date("2017,13,31")"#, "");
}

#[test]
fn test_decimal() {
  let scope = &te_scope("{}");
  te_number(false, scope, "decimal(1,2)", 1.00);
  te_number(false, scope, "decimal(1/3,2)", 0.330);
  te_number(false, scope, "decimal(1/3, 2.5)", 0.330);
  te_number(false, scope, "decimal(0.505,2)", 0.50);
  te_number(false, scope, "decimal(0.515,2)", 0.52);
}

#[test]
fn test_duration() {
  let scope = &te_scope("{}");
  te_years_and_months_duration(false, scope, r#"duration("P1Y")"#, 1, 0);
  te_years_and_months_duration(false, scope, r#"duration("P4Y")"#, 4, 0);
  te_years_and_months_duration(false, scope, r#"duration("-P999999999Y")"#, -999_999_999, 0);
  te_years_and_months_duration(false, scope, r#"duration("P999999999Y")"#, 999_999_999, 0);
  te_years_and_months_duration(false, scope, r#"duration(from:"P26M")"#, 2, 2);
  te_days_and_time_duration(false, scope, r#"duration("P1D")"#, SECONDS_IN_DAY, 0);
  te_days_and_time_duration(false, scope, r#"duration("P4D")"#, 4 * SECONDS_IN_DAY, 0);
  te_days_and_time_duration(false, scope, r#"duration("PT2H")"#, 2 * SECONDS_IN_HOUR, 0);
  te_days_and_time_duration(false, scope, r#"duration("PT3M")"#, 3 * SECONDS_IN_MINUTE, 0);
  te_days_and_time_duration(false, scope, r#"duration("PT4S")"#, 4, 0);
  te_days_and_time_duration(false, scope, r#"duration(from:"PT24H")"#, SECONDS_IN_DAY, 0);
  te_null(false, scope, "duration(null)", "");
  te_null(false, scope, "duration()", "");
  te_null(false, scope, "duration([])", "");
  te_null(false, scope, r#"duration("")"#, "");
  te_null(false, scope, "duration(2017)", "");
  te_null(false, scope, r#"duration("2012T-12-2511:00:00Z")"#, "");
  te_null(false, scope, r#"duration("P")"#, "");
  te_null(false, scope, r#"duration("P0")"#, "");
  te_null(false, scope, r#"duration("1Y")"#, "");
  te_null(false, scope, r#"duration("1D")"#, "");
  te_null(false, scope, r#"duration("P1H")"#, "");
  te_null(false, scope, r#"duration("P1S")"#, "");
}

#[test]
fn test_even() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "even(2)", true);
  te_bool(false, scope, "even(1)", false);
  te_bool(false, scope, "even(-2)", true);
  te_bool(false, scope, "even(-1)", false);
  te_bool(false, scope, "even(0)", true);
  te_bool(false, scope, "even(0.0)", true);
  te_null(false, scope, "even()", "");
  te_null(false, scope, "even(4,4)", "");
  te_bool(false, scope, "even(number:4)", true);
  te_null(false, scope, "even(n:4)", "");
  te_null(false, scope, "even(null)", "");
  te_null(false, scope, r#"even("4")"#, "");
  te_null(false, scope, "even(true)", "");
  te_null(false, scope, "even(false)", "");
  te_null(false, scope, r#"even(duration("P4D"))"#, "");
  te_null(false, scope, r#"even(duration("P4Y"))"#, "");
  te_null(false, scope, r#"even(date("2018-12-06"))"#, "");
  te_null(false, scope, r#"even(time("00:00:00"))"#, "");
  te_null(false, scope, r#"even(date and time("2018-12-06T00:00:00"))"#, "");
  te_bool(false, scope, "even(2.35)", true);
  te_bool(false, scope, "even(-2.35)", true);
  te_bool(false, scope, "even(1.78)", false);
  te_bool(false, scope, "even(-1.78)", false);
}

#[test]
fn test_exp() {
  let scope = &te_scope("{}");
  te_number(false, scope, "exp(4)", 4.0_f64.exp());
  te_number(false, scope, "exp(4)", 54.598150033144236);
  te_number(false, scope, "exp(-1)", (-1.0_f64).exp());
  te_number(false, scope, "exp(0)", 1.0);
  te_number(false, scope, "exp(number:4)", 4.0_f64.exp());
  te_null(false, scope, "exp(n:4)", "");
  te_null(false, scope, "exp()", "");
  te_null(false, scope, "exp(4,4)", "");
  te_null(false, scope, "exp(null)", "");
  te_null(false, scope, r#"exp("4")"#, "");
  te_null(false, scope, "exp(true)", "");
  te_null(false, scope, r#"exp(duration("P4D"))"#, "");
  te_null(false, scope, r#"exp(duration("P4Y"))"#, "");
  te_null(false, scope, r#"exp(date("2018-12-06"))"#, "");
  te_null(false, scope, r#"exp(time("00:00:00"))"#, "");
  te_null(false, scope, r#"exp(date and time("2018-12-06T00:00:00"))"#, "");
}

#[test]
fn test_flatten() {
  let scope = &te_scope("{}");
  te_be_value(false, scope, r#"flatten([["w","x"],["y"],["z"]])"#, r#"["w","x","y","z"]"#);
  te_be_value(
    false,
    scope,
    r#"flatten([["w","x"],["y",["a","b","c"]],["z"]])"#,
    r#"["w","x","y","a","b","c","z"]"#,
  );
}

#[test]
fn test_floor() {
  let scope = &te_scope("{}");
  te_number(false, scope, "floor(1.5)", 1.0);
  te_number(false, scope, "floor(-1.5)", -2.0);
  te_number(false, scope, "floor(--1)", 1.0);
  te_number(false, scope, "floor(-5/2.3*5)", -11.0);
}

#[test]
fn test_insert_before() {
  let scope = &te_scope(r#"{}"#);
  te_be_value(false, scope, r#"insert before([2,3,4,5],1,1)"#, r#"[1,2,3,4,5]"#);
  te_be_value(false, scope, r#"insert before([1,2,3,5],-1,4)"#, r#"[1,2,3,4,5]"#);
}

#[test]
fn test_join() {
  let scope = &te_scope(
    r#"{DeptTable:[{manager:"Smith",name:"Sales",number:10},{manager:"Jones",name:"Finance",number:20},{manager:"King",name:"Engineering",number:30}],EmployeeTable:[{deptNum:10,id:"7792",name:"Clark"},{deptNum:10,id:"7934",name:"Miller"},{deptNum:20,id:"7976",name:"Adams"},{deptNum:20,id:"7902",name:"Ford"},{deptNum:30,id:"7900",name:"James"}],LastName:"Clark"}"#,
  );
  te_be_value(false, scope, r#"EmployeeTable[name=LastName]"#, r#"{deptNum:10,id:"7792",name:"Clark"}"#);
  te_value(false, scope, r#"EmployeeTable[name=LastName].deptNum"#, r#"10"#);
  te_value(false, scope, r#"EmployeeTable[name=LastName].deptNum[1]"#, r#"10"#);
  te_be_value(false, scope, r#"DeptTable[number=10]"#, r#"{manager:"Smith",name:"Sales",number:10}"#);
  te_value(false, scope, r#"DeptTable[number=10].manager[1]"#, r#""Smith""#);
  //te_value(false,scope, r#"DeptTable[number=EmployeeTable[name=LastName].deptNum[1]].manager[1]"#, r#""Smith""#);
}

#[test]
fn test_log() {
  let scope = &te_scope("{}");
  te_number(false, scope, "log(4)", 4.0_f64.ln());
  te_number(false, scope, "log(4)", 1.3862943611198906);
  te_null(false, scope, "log(-1)", "");
  te_null(false, scope, "log(0)", "");
  te_null(false, scope, "log(0.0)", "");
  te_null(false, scope, "log()", "");
  te_null(false, scope, "log(4,4)", "");
  te_number(false, scope, "log(number:4)", 4.0_f64.ln());
  te_null(false, scope, "log(n:4)", "");
  te_null(false, scope, "log(null)", "");
  te_null(false, scope, r#"log("4")"#, "");
  te_null(false, scope, "log(true)", "");
  te_null(false, scope, "log(false)", "");
  te_null(false, scope, r#"log(duration("P4D"))"#, "");
  te_null(false, scope, r#"log(duration("P4Y"))"#, "");
  te_null(false, scope, r#"log(date("2018-12-06"))"#, "");
  te_null(false, scope, r#"log(time("00:00:00"))"#, "");
  te_null(false, scope, r#"log(date and time("2018-12-06T00:00:00"))"#, "");
  te_number(false, scope, "log(10)", 10.0_f64.ln());
  te_number(false, scope, "decimal(log(10),4)", f64::ceil(LN_10.mul(10_000.0)).div(10_000.0));
}

#[test]
fn test_matches() {
  let scope = &te_scope(r#"{}"#);
  te_bool(false, scope, r#"matches("foobar","^fo*b")"#, true);
  te_bool(false, scope, r#"matches("abracadabra","bra")"#, true);
  te_bool(false, scope, r#"matches("abracadabra","^a.*a$")"#, true);
  te_bool(false, scope, r#"matches("abracadabra","^bra")"#, false);
  let scope = &te_scope(
    r#"{poem:"<poem author=\"Wilhelm Busch\">\nKaum hat dies der Hahn gesehen,\nFängt er auch schon an zu krähen:\nKikeriki! Kikikerikih!!\nTak, tak, tak! - da kommen sie.\n</poem>"}"#,
  );
  te_bool(false, scope, r#"matches(poem, "Kaum.*krähen")"#, false);
  te_bool(false, scope, r#"matches("hello\nworld","hello.*world")"#, false);
  te_bool(false, scope, r#"matches("hello\nworld","hello.*world","s")"#, true);
}

#[test]
fn test_max() {
  let scope = &te_scope("{}");
  te_null(false, scope, "max()", "");
  te_null(false, scope, "max([])", "");
  te_null(false, scope, "max(l:[])", "");
  te_null(false, scope, "max(l:[1,2,3])", "");
  te_number(false, scope, "max([2021])", 2021.0);
  te_number(false, scope, "max(2021)", 2021.0);
  te_number(false, scope, "max([1,2,3])", 3.0);
  te_number(false, scope, "max(1,2,3)", 3.0);
  te_number(false, scope, "max([8,4,2.89,3,8.0001,2.89,5])", 8.0001);
  te_number(false, scope, "max(8,4,2.89,3,8.0001,2.89,5)", 8.0001);
  te_number(false, scope, "max([2837465.9584,-39408573.456749])", 2837465.9584);
  te_number(false, scope, "max(2837465.9584,-39408573.456749)", 2837465.9584);
  te_string(false, scope, r#"max(["a","b","c"])"#, "c");
  te_string(false, scope, r#"max("a","b","c")"#, "c");
  te_string(false, scope, r#"max("John","Johnny")"#, "Johnny");
}

#[test]
fn test_median() {
  let scope = &te_scope("{}");
  te_null(false, scope, "median()", "");
  te_null(false, scope, "median([])", "");
  te_null(false, scope, "median(l:[])", "");
  te_null(false, scope, "median(l:[1,2,3])", "");
  te_null(false, scope, "median([true,false])", "");
  te_number(false, scope, "median([8, 2, 5, 7])", 6.0);
  te_number(false, scope, "median(list:[8, 2, 5, 7])", 6.0);
  te_number(false, scope, "median([8,2,5,3,4])", 4.0);
  te_number(false, scope, "median(list:[8,2,5,3,4])", 4.0);
  te_number(false, scope, "median(8,2,5,3,4)", 4.0);
  te_number(false, scope, "median([8,2,5,3,4.25])", 4.25);
  te_number(false, scope, "median(list:[8,2,5,3,4.25])", 4.25);
  te_number(false, scope, "median(8,2,5,3,4.25)", 4.25);
  te_number(false, scope, "median([6,1,2,3])", 2.5);
  te_number(false, scope, "median(list:[6,1,2,3])", 2.5);
  te_number(false, scope, "median(6,1,2,3)", 2.5);
  te_number(false, scope, "median([2021])", 2021.);
  te_number(false, scope, "median(list:[2021])", 2021.);
  te_number(false, scope, "median(2021)", 2021.);
  te_number(false, scope, "median([1999,2999])", 2499.);
  te_number(false, scope, "median(list:[1999,2999])", 2499.);
  te_number(false, scope, "median(1999,2999)", 2499.);
}

#[test]
fn test_min() {
  let scope = &te_scope("{}");
  te_null(false, scope, "min()", "");
  te_null(false, scope, "min([])", "");
  te_null(false, scope, "min(l:[])", "");
  te_null(false, scope, "min(list:[])", "");
  te_null(false, scope, "min(l:[1,2,3])", "");
  te_number(false, scope, "min(list:[1,2,3])", 1.0);
  te_number(false, scope, "min([2021])", 2021.0);
  te_number(false, scope, "min(2021)", 2021.0);
  te_number(false, scope, "min([1,2,3])", 1.0);
  te_number(false, scope, "min(1,2,3)", 1.0);
  te_number(false, scope, "min([8,4,2.89,3,2.89,5])", 2.89);
  te_number(false, scope, "min(8,4,2.89,3,2.89,5)", 2.89);
  te_number(false, scope, "min([2837465.9584,-39408573.456749])", -39408573.456749);
  te_number(false, scope, "min(2837465.9584,-39408573.456749)", -39408573.456749);
  te_string(false, scope, r#"min(["a","b","c"])"#, "a");
  te_string(false, scope, r#"min("a","b","c")"#, "a");
  te_string(false, scope, r#"min("John","Johnny")"#, "John");
}

#[test]
fn test_mode() {
  let scope = &te_scope("{}");
  te_null(false, scope, "mode()", "");
  te_null(false, scope, "mode(l:[])", "");
  te_null(false, scope, "mode(l:[1,2,3])", "");
  te_null(false, scope, "mode([true,false])", "");
  te_be_value(false, scope, "mode(list:[])", "[]");
  te_be_value(false, scope, "mode([])", "[]");
  te_be_value(false, scope, "mode(list:[23])", "[23]");
  te_be_value(false, scope, "mode([23])", "[23]");
  te_be_value(false, scope, "mode(23)", "[23]");
  te_be_value(false, scope, "mode(list:[6,3,9,6,6])", "[6]");
  te_be_value(false, scope, "mode([6,3,9,6,6])", "[6]");
  te_be_value(false, scope, "mode(6,3,9,6,6)", "[6]");
  te_be_value(false, scope, "mode(list:[6,1,9,6,1])", "[1,6]");
  te_be_value(false, scope, "mode([6,1,9,6,1])", "[1,6]");
  te_be_value(false, scope, "mode(6,1,9,6,1)", "[1,6]");
}

#[test]
fn test_modulo() {
  let scope = &te_scope("{}");
  te_null(false, scope, "modulo()", "");
  te_null(false, scope, "modulo(1)", "");
  te_null(false, scope, "modulo(1, true)", "");
  te_null(false, scope, "modulo(a:1,b:2)", "");
  te_number(false, scope, "modulo(1,1)", 0.0);
  te_number(false, scope, "modulo(1,2)", 1.0);
  te_number(false, scope, "modulo(dividend:1,divisor:2)", 1.0);
  te_number(false, scope, "modulo(modulo(10,6),3)", 1.0);
  te_number(false, scope, "modulo(-12,5)", 3.0);
  te_number(false, scope, "modulo(12,5)", 2.0);
  te_number(false, scope, "modulo(12,-5)", -3.0);
  te_number(false, scope, "modulo(-12,-5)", -2.0);
  te_number(false, scope, "decimal(modulo(10.1,4.5),1)", 1.1);
  te_number(false, scope, "decimal(modulo(-10.1,4.5),1)", 3.4);
  te_number(false, scope, "decimal(modulo(10.1,-4.5),1)", -3.4);
  te_number(false, scope, "decimal(modulo(-10.1,-4.5),1)", -1.1);
}

#[test]
fn test_not() {
  let scope = &te_scope("{ On time: true, Too late: false }");
  te_bool(false, scope, "not(true)", false);
  te_bool(false, scope, " not  (  true  ) ", false);
  te_bool(false, scope, "not(false)", true);
  te_bool(false, scope, " not  \n (  \t  false \r  ) \n  ", true);
  te_bool(false, scope, "not(On time)", false);
  te_bool(false, scope, "not(Too late)", true);
}

#[test]
fn test_odd() {
  let scope = &te_scope("{ even number: 20, odd number: 21 }");
  te_bool(false, scope, "odd(2)", false);
  te_bool(false, scope, "odd(-2)", false);
  te_bool(false, scope, "odd(1)", true);
  te_bool(false, scope, "odd(-1)", true);
  te_bool(false, scope, "odd(0)", false);
  te_bool(false, scope, "odd(-0)", false);
  te_null(false, scope, "odd()", "");
  te_null(false, scope, "odd(4,4)", "");
  te_bool(false, scope, "odd(number:4)", false);
  te_null(false, scope, "odd(n:4)", "");
  te_bool(false, scope, "odd(even number)", false);
  te_bool(false, scope, "odd(odd number)", true);
}

#[test]
fn test_remove() {
  let scope = &te_scope("{}");
  te_null(false, scope, "remove([1,2,3,4,5],0)", "");
  te_null(false, scope, "remove([1,2,3,4,5],6)", "");
  te_null(false, scope, "remove([1,2,3,4,5],-6)", "");
  te_null(false, scope, "remove([1,2,3,4,5])", "");
  te_null(false, scope, "remove(6)", "");
  te_null(false, scope, "remove([1,2,3,4,5],true)", "");
  te_null(false, scope, "remove(l:[1,2,3,4,5],position:1)", "");
  te_null(false, scope, "remove(list:[1,2,3,4,5],p:1)", "");
  te_be_value(false, scope, "remove([1,2,3,4,5],1)", "[2,3,4,5]");
  te_be_value(false, scope, "remove([1,2,3,4,5],2)", "[1,3,4,5]");
  te_be_value(false, scope, "remove([1,2,3,4,5],3)", "[1,2,4,5]");
  te_be_value(false, scope, "remove([1,2,3,4,5],4)", "[1,2,3,5]");
  te_be_value(false, scope, "remove([1,2,3,4,5],5)", "[1,2,3,4]");
  te_be_value(false, scope, "remove([1,2,3,4,5],-5)", "[2,3,4,5]");
  te_be_value(false, scope, "remove([1,2,3,4,5],-4)", "[1,3,4,5]");
  te_be_value(false, scope, "remove([1,2,3,4,5],-3)", "[1,2,4,5]");
  te_be_value(false, scope, "remove([1,2,3,4,5],-2)", "[1,2,3,5]");
  te_be_value(false, scope, "remove([1,2,3,4,5],-1)", "[1,2,3,4]");
  te_be_value(false, scope, "remove(list:[1,2,3,4,5],position:1)", "[2,3,4,5]");
}

#[test]
fn test_replace() {
  let scope = &te_scope("{}");
  te_string(false, scope, r##"replace("abcd","(ab)|(a)","[1=$1][2=$2]")"##, r##"[1=ab][2=]cd"##);
  te_string(false, scope, r##"replace("a","[b-z]","#")"##, r##"a"##);
  te_string(false, scope, r##"replace("a","[a-z]","#")"##, r##"#"##);
  te_string(false, scope, r##"replace("abc","def","#")"##, r##"abc"##);
  te_string(false, scope, r##"replace("abc","e","#")"##, r##"abc"##);
  te_string(false, scope, r##"replace("foobar","^fo*b*","#")"##, r##"#ar"##);
  te_string(false, scope, r##"replace("abc",".^[d-z]","#")"##, r##"abc"##);
  te_string(false, scope, r##"replace("abracadabra","bra","*")"##, r##"a*cada*"##);
  te_string(false, scope, r##"replace("abracadabra","a.*a","*")"##, r##"*"##);
  te_string(false, scope, r##"replace("abracadabra","a.*?a","*")"##, r##"*c*bra"##);
  te_string(false, scope, r##"replace("abracadabra","a","")"##, r##"brcdbr"##);
  te_string(false, scope, r##"replace("AAAA","A+","b")"##, r##"b"##);
  te_string(false, scope, r##"replace("AAAA","A+?","b")"##, r##"bbbb"##);
  te_string(false, scope, r##"replace("darted","^(.*?)d(.*)$","$1$2")"##, r##"arted"##);
  te_string(false, scope, r##"replace("darted","^(.*?)d(.*)$","$1c$2")"##, r##"carted"##);
  te_string(false, scope, r##"replace("reluctant","r.*?t","X")"##, r##"Xant"##);
  te_string(
    false,
    scope,
    r##"replace("0123456789","(\d{3})(\d{3})(\d{4})","($1) $2-$3")"##,
    r##"(012) 345-6789"##,
  );
  te_string(false, scope, r##"replace("abc","[a-z]","#","")"##, r##"###"##);
  te_string(false, scope, r##"replace("a.b.c.","[a-z]","#","s")"##, r##"#.#.#."##);
  te_string(false, scope, r##"replace("abc","[A-Z]","#","i")"##, r##"###"##);
  te_string(false, scope, r##"replace("abc","[a-z]","#","s")"##, r##"###"##);
  te_string(false, scope, r##"replace("a b c d ","[a-z]","#","x")"##, r##"# # # # "##);
  te_string(false, scope, r##"replace("a b c d ","[a-z]","#")"##, r##"# # # # "##);
  te_string(false, scope, r##"replace("abc",".^[d-z]*","smix")"##, r##"abc"##);
  te_string(false, scope, r##"replace(input:"abc",pattern:"[a-z]",replacement:"#")"##, r##"###"##);
  te_string(false, scope, r##"replace(input:"abc",pattern:"[A-Z]",replacement:"#",flags:"")"##, r##"abc"##);
  te_string(false, scope, r##"replace(input:"abc",pattern:"[A-Z]",replacement:"#",flags:"i")"##, r##"###"##);
  te_string(
    false,
    scope,
    r##"replace(input:"abc",pattern:".^[d-z]*",replacement:"#",flags:"smix")"##,
    r##"abc"##,
  );
  te_string(false, scope, r##"replace("a\b\c","\\","\\\\","q")"##, r##"a\\b\\c"##);
  te_string(false, scope, r##"replace("a/b/c","/","$","q")"##, r##"a$b$c"##);
  te_string(false, scope, r##"replace("abc","[A-Z]","#","all unknown but i")"##, r##"###"##);
  te_string(false, scope, r##"replace(replace("anbncn","n","\u000A"),"[A-C]\n","u","i")"##, r##"uuu"##);
  te_string(false, scope, r##"replace("a\u000Ab\u000Ac\u000A","[A-Z]\n","u","i")"##, r##"uuu"##);
}

#[test]
fn test_split() {
  let scope = &te_scope("{}");
  te_be_value(false, scope, r#"split("John Doe","\\s")"#, r#"["John","Doe"]"#);
  te_be_value(false, scope, r#"split("a;b;c;;",";")"#, r#"["a","b","c","",""]"#);
}

#[test]
fn test_sqrt() {
  let scope = &te_scope("{ Area: 144.0 }");
  te_number(false, scope, "sqrt(4)", 2.0);
  te_null(false, scope, "sqrt(-1)", "");
  te_number(false, scope, "sqrt(0)", 0.0);
  te_number(false, scope, "sqrt(0.0)", 0.0);
  te_number(false, scope, "sqrt(-0.0)", 0.0);
  te_null(false, scope, "sqrt()", "");
  te_null(false, scope, "sqrt(4,4)", "");
  te_null(false, scope, "sqrt(null)", "");
  te_null(false, scope, r#"sqrt("4")"#, "");
  te_null(false, scope, "sqrt(n:64.0)", "");
  te_number(false, scope, "sqrt(number:81.0)", 9.0);
  te_number(false, scope, "sqrt(Area)", 12.0);
  te_null(false, scope, "sqrt(n  :Area)", "");
  te_number(false, scope, "sqrt(number : Area)", 12.0);
}

#[test]
fn test_stddev() {
  let scope = &te_scope("{}");
  te_number(false, scope, r#"stddev(2,4,7,5)"#, 2.081_665_999_466_132_6);
  te_number(false, scope, r#"decimal(stddev(2,4,7,5),13)"#, 2.081_665_999_466_1);
  te_number(false, scope, r#"decimal(stddev(2,4,7,5),9)"#, 2.081_665_999);
  te_number(false, scope, r#"stddev([2,4,7,5])"#, 2.081_665_999_466_132_6);
  te_number(false, scope, r#"stddev(list:[2,4,7,5])"#, 2.081_665_999_466_132_6);
  te_number(false, scope, r#"stddev(5,6,8,9)"#, 1.825_741_858_350_553_8);
}

#[test]
fn test_string() {
  let scope = &te_scope("{}");
  te_string(false, scope, "string(null)", "null");
  te_string(false, scope, "string(1.1)", "1.1");
  te_string(false, scope, "string(true)", "true");
  te_string(false, scope, "string(false)", "false");
  te_string(false, scope, "string({a:  1.0, b: 2.0})", "{a:1,b:2}");
  te_string(false, scope, "string(from:null)", "null");
  te_string(false, scope, "string(from:1.1)", "1.1");
  te_string(false, scope, r#"string(from:"Adam")"#, r#"Adam"#);
}

#[test]
fn test_string_length() {
  let scope = &te_scope("{}");
  te_number(false, scope, r#"string length("engos")"#, 5.0);
  te_number(false, scope, r#"string length("\u0009")"#, 1.0);
  te_number(false, scope, r#"string length("\\u0009")"#, 6.0);
  te_number(false, scope, r#"string length("\U000009")"#, 1.0);
  te_number(false, scope, r#"string length("\uD83D\uDC0E")"#, 1.0);
  te_number(false, scope, r#"string length("🐎")"#, 1.0);
  te_number(false, scope, r#"string length("🐎😀")"#, 2.0);
}

#[test]
fn test_substring() {
  let scope = &te_scope("{}");
  te_string(false, scope, r#"substring("foobar",3)"#, "obar");
  te_string(false, scope, r#"substring("foobar",3,3)"#, "oba");
  te_string(false, scope, r#"substring("foobar",-2,1)"#, "a");
  te_string(false, scope, r#"substring("\U01F40Eab",2)"#, "ab");
}

#[test]
fn test_substring_after() {
  let scope = &te_scope("{}");
  te_string(false, scope, r#"substring after("foobar","ob")"#, "ar");
  te_string(false, scope, r#"substring after("","a")"#, "");
}

#[test]
fn test_substring_before() {
  let scope = &te_scope("{}");
  te_string(false, scope, r#"substring before("foobar","bar")"#, "foo");
  te_string(false, scope, r#"substring before("foobar","xyz")"#, "");
}

#[test]
fn test_time() {
  let scope = &te_scope("{}");
  te_time(false, scope, r#"time("23:59:00")"#, FeelTime::local(23, 59, 0, 0));
  te_time(false, scope, r#"  time  (  "23:59:00"         )   "#, FeelTime::local(23, 59, 0, 0));
  te_time(false, scope, r#"time("23:59:00Z")"#, FeelTime::utc(23, 59, 0, 0));
  te_time(false, scope, r#"time("23:59:00z")"#, FeelTime::utc(23, 59, 0, 0));
  te_time(false, scope, r#"time("11:22:33-00:00")"#, FeelTime::utc(11, 22, 33, 0));
  te_time(false, scope, r#"time("11:22:33+00:00")"#, FeelTime::utc(11, 22, 33, 0));
  te_time(false, scope, r#"time(time("11:00:00"))"#, FeelTime::local(11, 0, 0, 0));
  te_time(false, scope, r#"time(date and time("2019-12-06T18:34:12"))"#, FeelTime::local(18, 34, 12, 0));
  te_time(false, scope, r#"time(date and time("2019-12-06T11:00:00Z"))"#, FeelTime::utc(11, 0, 0, 0));
  te_time(false, scope, r#"time(date and time("2019-12-06T11:00:00z"))"#, FeelTime::utc(11, 0, 0, 0));
  te_time(false, scope, r#"time(date("2019-12-06"))"#, FeelTime::utc(0, 0, 0, 0));
  te_bool(false, scope, r#"time("12:21:12") in [time("12:21:12")..time("12:21:12")]"#, true);
  te_bool(false, scope, r#"time("12:21:11") in [time("12:21:12")..time("12:21:12")]"#, false);
  te_bool(false, scope, r#"time("12:21:13") in [time("12:21:12")..time("12:21:12")]"#, false);
  te_bool(false, scope, r#"time("12:21:12") in (time("12:21:11")..time("12:21:13"))"#, true);
  te_null(false, scope, r#"time("22:63:12")"#, "");
  te_null(false, scope, r#"time("22:10:12+15:00")"#, "");
  te_null(false, scope, r#"time("22:10:12-15:00")"#, "");
  te_null(false, scope, r#"time(24,59,45,null)"#, "");
  te_null(false, scope, r#"time(23,60,45,null)"#, "");
  te_null(false, scope, r#"time(23,59,60,null)"#, "");
  te_null(false, scope, r#"time(-12,12,12,null)"#, "");
  te_null(false, scope, r#"time(12,-12,12,null)"#, "");
  te_null(false, scope, r#"time(12,12,-12,null)"#, "");
}

#[test]
fn test_years_and_months_duration() {
  let scope = &te_scope("{}");
  te_years_and_months_duration(
    false,
    scope,
    r#"years and months duration(date("2013-08-24"),date and time("2017-12-15T00:59:59"))"#,
    4,
    3,
  );
  te_years_and_months_duration(
    false,
    scope,
    r#"years and months duration(date and time("2017-02-28T23:59:59"),date("2019-07-23"))"#,
    2,
    4,
  );
}
