/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL and DMN model evaluator.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use dmntk_feel::values::{Value, VALUE_NULL};
use dmntk_feel::{AstNode, FeelDate, FeelDateTime, FeelDaysAndTimeDuration, FeelTime, FeelYearsAndMonthsDuration, Scope};

mod built_in_functions;
mod textual_expression;
mod unary_tests;

const SECONDS_IN_DAY: i64 = 86_400;
const SECONDS_IN_HOUR: i64 = 3_600;
const SECONDS_IN_MINUTE: i64 = 60;

/// Utility function that tests evaluation of boolean value.
pub fn te_bool(scope: &Scope, s: &str, expected: bool) {
  textual_expression(scope, s, Value::Boolean(expected));
}

/// Utility function that tests evaluation of boolean value, with tracing messages.
pub fn te_bool_trace(scope: &Scope, s: &str, expected: bool) {
  textual_expression_trace(scope, s, Value::Boolean(expected));
}

/// Utility function that tests evaluation of date value.
pub fn te_date(scope: &Scope, s: &str, year: i32, month: u8, day: u8) {
  textual_expression(scope, s, Value::Date(FeelDate::new(year, month, day)));
}

/// Utility function that tests evaluation of date value, with tracing messages.
pub fn te_date_trace(scope: &Scope, s: &str, year: i32, month: u8, day: u8) {
  textual_expression_trace(scope, s, Value::Date(FeelDate::new(year, month, day)));
}

/// Utility function that tests evaluation of date and time value.
pub fn te_date_time(scope: &Scope, s: &str, expected: FeelDateTime) {
  textual_expression(scope, s, Value::DateTime(expected));
}

/// Utility function that tests evaluation of date and time value.
pub fn te_date_time_local(scope: &Scope, s: &str, year: i32, month: u8, day: u8, hour: u8, minute: u8, second: u8, nanos: u64) {
  textual_expression(scope, s, Value::DateTime(FeelDateTime::local(year, month, day, hour, minute, second, nanos)));
}

/// Utility function that creates scope from specified input.
pub fn te_scope(input: &str) -> Scope {
  let scope = Scope::default();
  match dmntk_feel_parser::parse_context(&scope, input, false) {
    Ok(node) => match crate::feel::evaluate(&scope, &node) {
      Ok(value) => match value {
        Value::Context(ctx) => ctx.into(),
        _ => {
          println!("ERROR (INVALID VALUE TYPE): {:?}", value);
          panic!("te_scope failed");
        }
      },
      Err(reason) => {
        println!("{}", reason);
        panic!("te_scope failed");
      }
    },
    Err(reason) => {
      println!("ERROR (REASON): {}", reason);
      panic!("te_scope failed");
    }
  }
}

/// Utility function that creates scope from specified input.
pub fn te_scope_trace(input: &str) -> Scope {
  let parent_scope = Scope::default();
  match dmntk_feel_parser::parse_context(&parent_scope, input, false) {
    Ok(node) => match crate::feel::evaluate(&parent_scope, &node) {
      Ok(value) => match value {
        Value::Context(ctx) => ctx.into(),
        _ => {
          println!("ERROR (INVALID VALUE TYPE): {:?}", value);
          panic!("te_scope_trace failed");
        }
      },
      Err(reason) => {
        println!("{}", reason);
        panic!("te_scope_trace failed");
      }
    },
    Err(reason) => {
      println!("ERROR (REASON): {}", reason);
      panic!("te_scope_trace failed");
    }
  }
}

/// Utility function that tests evaluation of numeric values.
pub fn te_number(scope: &Scope, s: &str, expected: f64) {
  textual_expression(scope, s, Value::Number(expected));
}

/// Utility function that tests evaluation of numeric values, with tracing messages.
pub fn te_number_trace(scope: &Scope, s: &str, expected: f64) {
  textual_expression_trace(scope, s, Value::Number(expected));
}

/// Utility function that tests evaluation to null value.
fn te_null(scope: &Scope, s: &str) {
  textual_expression(scope, s, VALUE_NULL);
}

/// Utility function that tests evaluation to null value with expected tracing message.
fn te_null_trace(scope: &Scope, s: &str, t: &str) {
  textual_expression(scope, s, Value::Null(Some(t.to_string())));
}

/// Utility function that tests evaluation to an error result.
pub fn te_none(scope: &Scope, s: &str) {
  assert!(dmntk_feel_parser::parse_textual_expression(scope, s, false).is_err());
}

/// Utility function that tests evaluation to string value.
fn te_string(scope: &Scope, s: &str, expected: &str) {
  textual_expression(scope, s, Value::String(expected.to_string()));
}

/// Utility function that tests evaluation to string value, with tracing messages.
pub fn te_string_trace(scope: &Scope, s: &str, expected: &str) {
  textual_expression_trace(scope, s, Value::String(expected.to_string()));
}

/// Utility function that tests evaluation of year and months duration.
pub fn te_years_and_months_duration(scope: &Scope, s: &str, years: i64, months: i64) {
  textual_expression(scope, s, Value::YearsAndMonthsDuration(FeelYearsAndMonthsDuration::new_ym(years, months)));
}

/// Utility function that tests evaluation of days and time duration.
pub fn te_days_and_time_duration(scope: &Scope, s: &str, sec: i64, nano: u64) {
  textual_expression(scope, s, Value::DaysAndTimeDuration(FeelDaysAndTimeDuration::new(sec, nano)));
}

/// Utility function that tests evaluation of time.
pub fn te_time(scope: &Scope, s: &str, expected: FeelTime) {
  textual_expression(scope, s, Value::Time(expected));
}

/// Utility function that tests evaluation to specified value.
pub fn te_value(scope: &Scope, actual: &str, expected: &str) {
  match dmntk_feel_parser::parse_textual_expression(scope, expected, false) {
    Ok(node) => match crate::feel::evaluate(scope, &node) {
      Ok(value) => textual_expression(scope, actual, value),
      Err(reason) => {
        println!("{}", reason);
        panic!("te_value failed");
      }
    },
    Err(reason) => {
      println!("ERROR (REASON): {}", reason);
      panic!("te_value failed");
    }
  }
}

/// Utility function that tests evaluation to specified value, with tracing messages
pub fn te_value_trace(scope: &Scope, actual: &str, expected: &str) {
  match dmntk_feel_parser::parse_textual_expression(scope, expected, true) {
    Ok(node) => match crate::feel::evaluate(scope, &node) {
      Ok(value) => textual_expression_trace(scope, actual, value),
      Err(reason) => {
        println!("{}", reason);
        panic!("te_value_trace failed");
      }
    },
    Err(reason) => {
      println!("ERROR (REASON): {}", reason);
      panic!("te_value_trace failed");
    }
  }
}

/// Utility function that tests evaluation to specified value represented by boxed expression.
pub fn te_be_value(scope: &Scope, actual: &str, expected: &str) {
  match dmntk_feel_parser::parse_boxed_expression(scope, expected, false) {
    Ok(node) => match crate::feel::evaluate(scope, &node) {
      Ok(value) => textual_expression(scope, actual, value),
      Err(reason) => {
        println!("{}", reason);
        panic!("te_value failed");
      }
    },
    Err(reason) => {
      println!("ERROR (REASON): {}", reason);
      panic!("te_value failed");
    }
  }
}

/// Utility function that tests evaluation to specified value represented by boxed expression.
pub fn te_be_value_trace(scope: &Scope, actual: &str, expected: &str) {
  match dmntk_feel_parser::parse_boxed_expression(scope, expected, true) {
    Ok(node) => match crate::feel::evaluate(scope, &node) {
      Ok(value) => textual_expression_trace(scope, actual, value),
      Err(reason) => {
        println!("{}", reason);
        panic!("te_value failed");
      }
    },
    Err(reason) => {
      println!("ERROR (REASON): {}", reason);
      panic!("te_value failed");
    }
  }
}

/// Utility function that tests evaluation to specified value represented by boxed expression.
pub fn be_be_value(scope: &Scope, actual: &str, expected: &str) {
  match dmntk_feel_parser::parse_boxed_expression(scope, expected, false) {
    Ok(node) => match crate::feel::evaluate(scope, &node) {
      Ok(value) => boxed_expression(scope, actual, value),
      Err(reason) => {
        println!("{}", reason);
        panic!("te_value failed");
      }
    },
    Err(reason) => {
      println!("ERROR (REASON): {}", reason);
      panic!("te_value failed");
    }
  }
}

/// Utility function that tests evaluation to specified value represented by boxed expression.
pub fn be_be_value_trace(scope: &Scope, actual: &str, expected: &str) {
  match dmntk_feel_parser::parse_boxed_expression(scope, expected, true) {
    Ok(node) => match crate::feel::evaluate(scope, &node) {
      Ok(value) => boxed_expression(scope, actual, value),
      Err(reason) => {
        println!("{}", reason);
        panic!("te_value failed");
      }
    },
    Err(reason) => {
      println!("ERROR (REASON): {}", reason);
      panic!("te_value failed");
    }
  }
}

/// Utility function that takes a text parameter, evaluates the boxed expression
/// represented by this text and compares the result with provided expected value.
/// The result must be equal to expected value, otherwise an error is reported.
pub fn boxed_expression(scope: &Scope, text: &str, expected: Value) {
  match dmntk_feel_parser::parse_boxed_expression(scope, text, false) {
    Ok(node) => match crate::feel::evaluate(scope, &node) {
      Ok(value) => assert_eq!(value, expected),
      Err(reason) => {
        println!("{}", reason);
        panic!("boxed_expression failed");
      }
    },
    Err(reason) => {
      println!("ERROR: {}", reason);
      panic!("boxed_expression failed");
    }
  }
}

/// Utility function that takes a text parameter, evaluates the textual expression
/// represented by this text and compares the result with provided expected value.
/// The result must be equal to expected value, otherwise an error is reported.
fn textual_expression(scope: &Scope, text: &str, expected: Value) {
  match dmntk_feel_parser::parse_textual_expression(scope, text, false) {
    Ok(node) => match crate::feel::evaluate(scope, &node) {
      Ok(value) => assert_eq!(value, expected),
      Err(reason) => {
        println!("{}", reason);
        panic!("textual_expression failed");
      }
    },
    Err(reason) => {
      println!("ERROR: {}", reason);
      panic!("textual_expression failed");
    }
  }
}

/// Utility function that takes a text parameter, evaluates the textual expression
/// represented by this text and compares the result with provided expected value.
/// The result must be equal to expected value, otherwise an error is reported.
/// Additionally the full trace of parsing and AST is displayed.
fn textual_expression_trace(scope: &Scope, text: &str, expected: Value) {
  match dmntk_feel_parser::parse_textual_expression(scope, text, true) {
    Ok(node) => match crate::feel::evaluate(scope, &node) {
      Ok(value) => assert_eq!(value, expected),
      Err(reason) => {
        println!("ERROR: {}", reason);
        panic!("textual_expression_trace failed");
      }
    },
    Err(reason) => {
      println!("ERROR: {}", reason);
      panic!("textual_expression_trace failed");
    }
  }
}

/// Utility function that checks if unary tests are correctly parsed.
pub fn valid_unary_tests(scope: &Scope, text: &str) {
  match dmntk_feel_parser::parse_unary_tests(scope, text, false) {
    Ok(_) => {}
    Err(reason) => {
      println!("ERROR: {}", reason);
      panic!("valid_unary_tests failed");
    }
  }
}

/// Utility function that checks if unary tests are correctly parsed.
/// Additionally the full trace of parsing and AST is displayed.
pub fn valid_unary_tests_trace(scope: &Scope, text: &str) {
  match dmntk_feel_parser::parse_unary_tests(scope, text, true) {
    Ok(_) => {}
    Err(reason) => {
      println!("ERROR: {}", reason);
      panic!("valid_unary_tests_trace failed");
    }
  }
}

pub fn satisfies(scope: &Scope, input_expression: &str, input_values: &str, input_entry: &str, expected: bool) {
  let input_expression_node = dmntk_feel_parser::parse_textual_expression(scope, input_expression, false).unwrap();
  let input_entry_node = dmntk_feel_parser::parse_unary_tests(&scope, input_entry, false).unwrap();
  let node = if !input_values.is_empty() {
    let input_values_node = dmntk_feel_parser::parse_unary_tests(scope, input_values, false).unwrap();
    let left = AstNode::In(Box::new(input_expression_node.clone()), Box::new(input_values_node));
    let right = AstNode::In(Box::new(input_expression_node), Box::new(input_entry_node));
    AstNode::And(Box::new(left), Box::new(right))
  } else {
    AstNode::In(Box::new(input_expression_node), Box::new(input_entry_node))
  };
  match crate::feel::evaluate(scope, &node) {
    Ok(value) => assert_eq!(value, Value::Boolean(expected)),
    Err(reason) => {
      println!("ERROR: {}", reason);
      panic!("`satisfies` failed");
    }
  }
}
