/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL and DMN model evaluator.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use crate::model::errors::*;
use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::context::FeelContext;
use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::dmntk_common::{Result, Stringify};
use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::values::{Value, VALUE_NULL};
use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::Name;
use dmntk_recognizer::dmntk_model::model::{Decision, Definitions, DmnElement, InputData, NamedElement};
use std::collections::HashMap;

///
type NameValueExtractor = Box<dyn Fn(&Value) -> Result<(Name, Value)>>;

///
type ValueExtractor = Box<dyn Fn(&Value) -> Result<Value>>;

///
#[derive(Default)]
pub struct ModelEvaluator {
  ///
  input_data_extractors: HashMap<String, NameValueExtractor>,
}

impl ModelEvaluator {
  ///
  pub fn new(definitions: &Definitions) -> Result<Self> {
    let mut input_data_extractors = HashMap::new();
    for input_data in definitions.input_data() {
      let input_data_id = input_data.id().as_ref().ok_or_else(empty_identifier)?.clone();
      input_data_extractors.insert(input_data_id, build_extractor_from_input_data(input_data)?);
    }
    Ok(Self { input_data_extractors })
  }
  ///
  pub fn evaluate_decision(&self, decision: &Decision, ctx: &FeelContext) -> Result<Value> {
    let expression_instance = decision.decision_logic().as_ref().ok_or_else(empty_decision_logic)?;
    println!("AAA:DECISION_LOGIC = {:?}", expression_instance);

    // if let Some(expression_instance) = decision.decision_logic() {
    //   println!("AAA:DECISION_LOGIC = {:?}", expression_instance);
    //   let evaluator = ModelEvaluator::new(&definitions)?;
    //
    //   let item_definitions = ItemDefinitions::new(definitions);
    //   evaluate_context(ctx, decision, definitions, &item_definitions)?;
    // // values evaluated in requirements may be overwritten by input parameters
    // evl.zip(ext);
    // let scope: Scope = evl.into();
    // println!("DECISION_SCOPE={}", scope.stringify());
    // let result = eval_expression_instance(&scope, expression_instance)?;
    // println!("DECISION_RESULT={}", result.stringify());
    // println!("DECISION_RESULT_TYPE={:?}", result.type_of());
    // let output_type = decision.variable().feel_type();
    // println!("DECISION_OUTPUT_TYPE={:?}", output_type);
    // return if result.type_of().is_conformant(output_type) {
    //   Ok(result)
    // } else {
    //   // coercion of singleton list into single value
    //   if let Value::List(items) = &result {
    //     if items.len() == 1 {
    //       if items[0].type_of().is_conformant(output_type) {
    //         return Ok(items[0].clone());
    //       }
    //     }
    //   }
    //   // coercion of single value into singleton list
    //   if let FeelType::List(expected_inner_type) = output_type {
    //     if result.type_of().is_conformant(expected_inner_type.borrow()) {
    //       return Ok(Value::List(Values::new(vec![result])));
    //     }
    //   }
    //   println!("COERCION");
    //   Ok(null_with_trace!("coercion"))
    // };
    //   Ok(VALUE_NULL)
    // } else {
    //   Ok(null_with_trace!("no decision logic"))
    // }
    self.evaluate_context(ctx, decision)?;
    Ok(VALUE_NULL)
  }
  /// Evaluates local context for specified [Decision].
  fn evaluate_context(&self, ctx: &FeelContext, decision: &Decision) -> Result<FeelContext> {
    println!("AAA:EXTERNAL_CONTEXT = {}", ctx.stringify());
    let mut evaluated_context: FeelContext = Default::default();
    //eval_knowledge_requirements(definitions, ctx, decision.knowledge_requirements(), &mut evaluated_ctx)?;

    for information_requirement in decision.information_requirements() {
      // if let Some(href) = information_requirement.required_decision() {
      //   if let Some(id) = href.strip_hash() {
      //     eval_required_decision_by_id(definitions, id, ctx, &mut evaluated_ctx)?;
      //   }
      // }
      if let Some(href) = information_requirement.required_input() {
        let (name, value) = self.evaluate_input_data(href.into(), &Value::Context(ctx.clone()))?;
        evaluated_context.set_entry(&name, value);
      }
    }
    println!("AAA:EVALUATED_CONTEXT = {}", evaluated_context.stringify());
    Ok(evaluated_context)
  }
  ///
  fn evaluate_input_data(&self, input_data_id: &str, value: &Value) -> Result<(Name, Value)> {
    // find the extractor function for specified input data identifier
    let extractor = self
      .input_data_extractors
      .get(input_data_id)
      .ok_or_else(|| input_data_with_specified_id_not_found(input_data_id))?;
    // execute an extractor and evaluate the input data
    extractor(value)
  }
}

///
pub fn build_extractor_from_something(_input_data: &InputData) -> Result<ValueExtractor> {
  Ok(Box::new(move |_: &Value| Ok(VALUE_NULL)))
}

///
pub fn build_extractor_from_input_data(input_data: &InputData) -> Result<NameValueExtractor> {
  let input_data_id = input_data.id().as_ref().ok_or_else(empty_identifier)?;
  let type_ref = input_data
    .variable()
    .type_ref()
    .as_ref()
    .ok_or_else(|| input_data_without_type_reference(input_data_id))?;
  let name = input_data.variable().feel_name().as_ref().ok_or_else(empty_feel_name)?.clone();
  Ok(match type_ref.as_str() {
    "string" => Box::new(move |value: &Value| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::String(_) = v {
            return Ok((name.clone(), v.clone()));
          }
        }
      }
      Ok((name.clone(), VALUE_NULL))
    }),
    "number" => Box::new(move |value: &Value| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::Number(_) = v {
            return Ok((name.clone(), v.clone()));
          }
        }
      }
      Ok((name.clone(), VALUE_NULL))
    }),
    "boolean" => Box::new(move |value: &Value| {
      if let Value::Context(ctx) = value {
        if let Some(v) = ctx.get_entry(&name) {
          if let Value::Boolean(_) = v {
            return Ok((name.clone(), v.clone()));
          }
        }
      }
      //Ok((name.clone(), VALUE_NULL))
      Err(empty_feel_name())
    }),
    //TODO add more basic types
    _ => {
      let mut a = vec![];
      a.push(build_extractor_from_something(input_data)?);
      a.push(build_extractor_from_something(input_data)?);
      a.push(build_extractor_from_something(input_data)?);
      Box::new(move |value: &Value| {
        for ve in &a {
          ve(&value)?;
        }
        Ok((name.clone(), VALUE_NULL))
      })
    }
  })
}

#[cfg(test)]
mod tests {
  use crate::model::eval::ModelEvaluator;
  use dmntk_recognizer::dmntk_model;
  use dmntk_recognizer::dmntk_model::dmntk_examples;
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::context::FeelContext;
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::values::{Value, VALUE_NULL};
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::Name;

  #[test]
  pub fn test_build_extractor_from_input_data_string() {
    let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0001, "file:///2_0001.dmn").unwrap();
    let evaluator = ModelEvaluator::new(&definitions).unwrap();
    let mut ctx: FeelContext = Default::default();
    let name_a = Name::new(&["Full", "Name"]);
    let value_a = Value::String("John".to_owned());
    ctx.set_entry(&name_a, value_a.clone());
    let value_ctx = Value::Context(ctx);
    let input_data_id = "_cba86e4d-e91c-46a2-9176-e9adf88e15db";
    assert_eq!((name_a, value_a), evaluator.evaluate_input_data(input_data_id, &value_ctx).unwrap());
  }

  #[test]
  pub fn test_build_extractor_from_input_data_number() {
    let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0002, "file:///2_0002.dmn").unwrap();
    let evaluator = ModelEvaluator::new(&definitions).unwrap();
    let mut ctx: FeelContext = Default::default();
    let name_a = Name::new(&["Monthly", "Salary"]);
    let value_a = Value::Number(12000.0);
    ctx.set_entry(&name_a, value_a.clone());
    let value_ctx = Value::Context(ctx);
    let input_data_id = "i_MonthlySalary";
    assert_eq!((name_a, value_a), evaluator.evaluate_input_data(input_data_id, &value_ctx).unwrap());
  }

  #[test]
  pub fn test_build_extractor_from_input_data_error() {
    let definitions = dmntk_model::parse(dmntk_examples::DMN_2_0001, "file:///2_0001.dmn").unwrap();
    let evaluator = ModelEvaluator::new(&definitions).unwrap();
    let mut ctx: FeelContext = Default::default();
    let name_a = Name::new(&["Full", "Name"]);
    let value_a = Value::Number(50.0);
    ctx.set_entry(&name_a, value_a);
    let value_ctx = Value::Context(ctx);
    let input_data_id = "_cba86e4d-e91c-46a2-9176-e9adf88e15db";
    assert_eq!((name_a, VALUE_NULL), evaluator.evaluate_input_data(input_data_id, &value_ctx).unwrap());
  }
}
