/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL and DMN model evaluator.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::dmntk_common::DmntkError;

/// Errors related to model evaluation.
#[derive(Debug, PartialEq)]
pub enum ModelEvaluatorError {
  BusinessKnowledgeModelWithNameNotFound(String),
  DecisionWithNameNotFound(String),
  DecisionServiceWithNameNotFound(String),
  MultipleRulesMatchForUniqueHitPolicy,
  AllMatchingRulesMustHaveTheSameValue,
  NumberOfOutputValuesDifferFromComponentNames,
  AggregatorsNotAllowedForCompoundOutputs,
  /// Error reported when input data with specified identifier was not found in definitions.
  InputDataWithSpecifiedIdNotFound(String),
  /// Error reported when input data has no type reference defined.
  InputDataWithoutTypeReference(String),
  EmptyFeelName,
  EmptyIdentifier,
  EmptyDecisionLogic,
}

impl From<ModelEvaluatorError> for DmntkError {
  fn from(e: ModelEvaluatorError) -> Self {
    DmntkError::new("ModelEvaluatorError", &format!("{}", e))
  }
}

impl std::fmt::Display for ModelEvaluatorError {
  fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
    match self {
      ModelEvaluatorError::BusinessKnowledgeModelWithNameNotFound(s) => {
        write!(f, "business knowledge model with name `{}` was not found", s)
      }
      ModelEvaluatorError::DecisionWithNameNotFound(s) => {
        write!(f, "decision with name `{}` was not found", s)
      }
      ModelEvaluatorError::DecisionServiceWithNameNotFound(s) => {
        write!(f, "decision service with name `{}` was not found", s)
      }
      ModelEvaluatorError::MultipleRulesMatchForUniqueHitPolicy => {
        write!(f, "multiple rules match in decision table with unique hit policy")
      }
      ModelEvaluatorError::AllMatchingRulesMustHaveTheSameValue => {
        write!(f, "all matching rules must have the same value")
      }
      ModelEvaluatorError::NumberOfOutputValuesDifferFromComponentNames => {
        write!(f, "the number of output values differ from component names")
      }
      ModelEvaluatorError::AggregatorsNotAllowedForCompoundOutputs => {
        write!(f, "aggregators are not allowed for compound outputs")
      }
      ModelEvaluatorError::InputDataWithSpecifiedIdNotFound(s) => {
        write!(f, "input data with specified identifier `{}` was not found in definitions", s)
      }
      ModelEvaluatorError::InputDataWithoutTypeReference(s) => {
        write!(f, "input data with id `{}` has no type reference definition", s)
      }
      ModelEvaluatorError::EmptyFeelName => {
        write!(f, "empty FEEL name")
      }
      ModelEvaluatorError::EmptyIdentifier => {
        write!(f, "empty identifier")
      }
      ModelEvaluatorError::EmptyDecisionLogic => {
        write!(f, "empty decision logic")
      }
    }
  }
}

pub fn business_knowledge_model_with_name_not_found(name: &str) -> DmntkError {
  ModelEvaluatorError::BusinessKnowledgeModelWithNameNotFound(name.to_owned()).into()
}

pub fn decision_with_name_not_found(name: &str) -> DmntkError {
  ModelEvaluatorError::DecisionWithNameNotFound(name.to_owned()).into()
}

pub fn decision_service_with_name_not_found(name: &str) -> DmntkError {
  ModelEvaluatorError::DecisionServiceWithNameNotFound(name.to_owned()).into()
}

pub fn multiple_rules_match_in_unique_hit_policy() -> DmntkError {
  ModelEvaluatorError::MultipleRulesMatchForUniqueHitPolicy.into()
}

pub fn all_matching_rules_must_have_the_same_value() -> DmntkError {
  ModelEvaluatorError::AllMatchingRulesMustHaveTheSameValue.into()
}

pub fn number_of_output_values_differ_from_component_names() -> DmntkError {
  ModelEvaluatorError::NumberOfOutputValuesDifferFromComponentNames.into()
}

pub fn aggregators_not_allowed_for_compound_outputs() -> DmntkError {
  ModelEvaluatorError::AggregatorsNotAllowedForCompoundOutputs.into()
}

pub fn input_data_with_specified_id_not_found(s: &str) -> DmntkError {
  ModelEvaluatorError::InputDataWithSpecifiedIdNotFound(s.to_owned()).into()
}

pub fn input_data_without_type_reference(s: &str) -> DmntkError {
  ModelEvaluatorError::InputDataWithoutTypeReference(s.to_owned()).into()
}

pub fn empty_feel_name() -> DmntkError {
  ModelEvaluatorError::EmptyFeelName.into()
}

pub fn empty_identifier() -> DmntkError {
  ModelEvaluatorError::EmptyIdentifier.into()
}

pub fn empty_decision_logic() -> DmntkError {
  ModelEvaluatorError::EmptyDecisionLogic.into()
}
