/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * DMN model and parser
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use self::errors::*;
use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::dmntk_common::Result;
use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::FeelType;
use dmntk_recognizer::dmntk_model::model::*;
use std::borrow::Borrow;
use std::collections::{BTreeMap, HashMap};

pub fn enrich_model(definitions: &mut Definitions) -> Result<()> {
  Enricher::default().enrich(definitions)
}

#[derive(Default)]
struct Enricher {
  /// Mapping of item definition name to evaluated `FEEL` type,
  /// needed for enriching `FEEL` types of item definitions.
  item_definition_types: HashMap<String, FeelType>,
}

impl Enricher {
  ///
  pub fn enrich(&mut self, definitions: &mut Definitions) -> Result<()> {
    self.enrich_feel_type_of_item_definitions(definitions)?;
    for drg_element in definitions.drg_elements_mut() {
      if let DrgElement::InputData(input_data) = drg_element {
        let variable = input_data.variable();
        let _opt_feel_type = self.opt_feel_type(variable.type_ref());
      }
    }
    Ok(())
  }
  ///
  fn enrich_feel_type_of_item_definitions(&mut self, definitions: &mut Definitions) -> Result<()> {
    for item_definition in definitions.item_definitions_mut() {
      self.update_feel_type_of_item_definition(item_definition)?;
    }
    Ok(())
  }
  ///
  fn update_feel_type_of_item_definition(&mut self, item_definition: &mut ItemDefinition) -> Result<()> {
    for item_component in item_definition.item_components_mut() {
      self.update_feel_type_of_item_definition(item_component)?;
    }
    let name = item_definition.name().to_string();
    let item_definition_type = self.item_definition_type(item_definition.borrow())?;
    item_definition.set_item_definition_type(item_definition_type.clone());
    match item_definition_type {
      // SIMPLE TYPE
      ItemDefinitionType::SimpleType(feel_type) => {
        self.item_definition_types.insert(name, feel_type.clone());
        item_definition.set_feel_type(feel_type);
        return Ok(());
      }
      // REFERENCED TYPE
      ItemDefinitionType::ReferencedType(ref_type_name) => {
        if let Some(feel_type) = self.item_definition_types.get(&ref_type_name).cloned() {
          self.item_definition_types.insert(name, feel_type.clone());
          item_definition.set_feel_type(feel_type);
          return Ok(());
        }
      }
      // COMPONENT TYPE
      ItemDefinitionType::ComponentType => {
        let mut type_entries = BTreeMap::new();
        for item_component in item_definition.item_components() {
          let feel_name = item_component.feel_name().as_ref().ok_or_else(|| empty_item_definition_feel_name(&name))?;
          let feel_type = item_component.feel_type().as_ref().ok_or_else(|| empty_item_definition_feel_type(&name))?;
          type_entries.insert(feel_name.clone(), feel_type.clone());
        }
        let context_type = FeelType::Context(type_entries);
        self.item_definition_types.insert(name, context_type.clone());
        item_definition.set_feel_type(context_type);
        return Ok(());
      }
      // COLLECTION OF SIMPLE TYPE
      ItemDefinitionType::CollectionOfSimpleType(feel_type) => {
        let list_type = FeelType::List(Box::new(feel_type));
        self.item_definition_types.insert(name, list_type.clone());
        item_definition.set_feel_type(list_type);
        return Ok(());
      }
      // COLLECTION OF REFERENCES TYPE
      ItemDefinitionType::CollectionOfReferencedType(ref_type_name) => {
        if let Some(feel_type) = self.item_definition_types.get(&ref_type_name).cloned() {
          let list_type = FeelType::List(Box::new(feel_type));
          self.item_definition_types.insert(name, list_type.clone());
          item_definition.set_feel_type(list_type);
          return Ok(());
        }
      }
      // COLLECTION OF COMPONENT TYPE
      ItemDefinitionType::CollectionOfComponentType => {
        let mut type_entries = BTreeMap::new();
        for item_component in item_definition.item_components_mut() {
          let feel_name = item_component.feel_name().as_ref().ok_or_else(|| empty_item_definition_feel_name(&name))?;
          let feel_type = item_component.feel_type().as_ref().ok_or_else(|| empty_item_definition_feel_type(&name))?;
          type_entries.insert(feel_name.clone(), feel_type.clone());
        }
        let context_type = FeelType::Context(type_entries);
        let list_type = FeelType::List(Box::new(context_type));
        self.item_definition_types.insert(name, list_type);
        return Ok(());
      }
    }
    Err(invalid_item_definition_type(item_definition.name()))
  }
  ///
  fn item_definition_type(&self, item_definition: &ItemDefinition) -> Result<ItemDefinitionType> {
    let opt_feel_type = self.opt_feel_type(item_definition.type_ref());
    let condition = (
      item_definition.type_ref().is_some(),
      opt_feel_type.is_some(),
      !item_definition.item_components().is_empty(),
      item_definition.is_collection(),
    );
    match condition {
      (_, true, false, false) => Ok(ItemDefinitionType::SimpleType(opt_feel_type.unwrap())),
      (true, false, false, false) => Ok(ItemDefinitionType::ReferencedType(item_definition.type_ref().as_ref().unwrap().clone())),
      (false, false, true, false) => Ok(ItemDefinitionType::ComponentType),
      (_, true, false, true) => Ok(ItemDefinitionType::CollectionOfSimpleType(opt_feel_type.unwrap())),
      (false, false, true, true) => Ok(ItemDefinitionType::CollectionOfComponentType),
      (true, false, false, true) => Ok(ItemDefinitionType::CollectionOfReferencedType(
        item_definition.type_ref().as_ref().unwrap().clone(),
      )),
      _ => Err(invalid_item_definition_type(item_definition.name())),
    }
  }
  ///
  fn opt_feel_type(&self, type_ref: &Option<String>) -> Option<FeelType> {
    type_ref
      .as_ref()
      .map(|type_ref| match type_ref.trim() {
        "string" => FeelType::String,
        "number" => FeelType::Number,
        "boolean" => FeelType::Boolean,
        "date" => FeelType::Date,
        "dateTime" => FeelType::DateTime,
        "dayTimeDuration" => FeelType::DaysAndTimeDuration,
        "yearMonthDuration" => FeelType::YearsAndMonthsDuration,
        _ => FeelType::Any,
      })
      .filter(|feel_type| *feel_type != FeelType::Any)
  }
}

/// Definition of errors raised by [enricher](crate::model::enricher) module.
mod errors {
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::dmntk_common::DmntkError;

  /// Errors raised by [Enricher].
  #[derive(Debug, PartialEq)]
  pub enum EnricherError {
    InvalidItemDefinitionType(String),
    EmptyItemDefinitionFeelName(String),
    EmptyItemDefinitionFeelType(String),
  }

  impl From<EnricherError> for DmntkError {
    fn from(e: EnricherError) -> Self {
      DmntkError::new("EnricherError", &format!("{}", e))
    }
  }

  impl std::fmt::Display for EnricherError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        EnricherError::InvalidItemDefinitionType(s) => {
          write!(f, "invalid item definition type for `{}`", s)
        }
        EnricherError::EmptyItemDefinitionFeelName(s) => {
          write!(f, "empty item definition FEEL name for `{}`", s)
        }
        EnricherError::EmptyItemDefinitionFeelType(s) => {
          write!(f, "empty item definition FEEL type for `{}`", s)
        }
      }
    }
  }
  ///
  pub fn invalid_item_definition_type(s: &str) -> DmntkError {
    EnricherError::InvalidItemDefinitionType(s.to_owned()).into()
  }
  ///
  pub fn empty_item_definition_feel_name(s: &str) -> DmntkError {
    EnricherError::EmptyItemDefinitionFeelName(s.to_owned()).into()
  }
  ///
  pub fn empty_item_definition_feel_type(s: &str) -> DmntkError {
    EnricherError::EmptyItemDefinitionFeelType(s.to_owned()).into()
  }
}
