/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL and DMN model evaluator.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use super::*;
use dmntk_feel_parser::dmntk_feel::FeelDateTime;

#[test]
fn test_addition() {
  let scope = &te_scope(r#"{Full Name:"John Doe", Employment Status:"EMPLOYED"}"#);
  te_number(false, scope, "1+1", 2.0);
  te_number(false, scope, " 1 + 2 ", 3.0);
  te_number(false, scope, " 5 +2 +1 ", 8.0);
  te_number(false, scope, "20+200+2", 222.0);
  te_number(false, scope, "( 1 + 2 ) + ( 3 + 4 )", 10.0);
  te_number(false, scope, "( ( ( 1 + 2 ) ) )", 3.0);
  te_number(false, scope, "(1+2)*(3+2)", 15.0);
  te_number(false, scope, "1+2*3+2", 9.0);
  te_number(false, scope, ".25 + .2", 0.45);
  te_string(false, scope, r#""Hello " + Full Name"#, "Hello John Doe");
  te_string(false, scope, r#""You are " + Employment Status"#, "You are EMPLOYED");
  te_string(false, scope, r#""Hello " + Full Name"#, "Hello John Doe");
}

#[test]
fn test_at_literal() {
  let scope = &te_scope(r#"{}"#);
  te_value(false, scope, r#"@"2021-01-28""#, r#"date("2021-01-28")"#);
  te_value(false, scope, r#"@"2021-01-28T19:15:00""#, r#"date and time("2021-01-28T19:15:00")"#);
  te_value(false, scope, r#"@"-PT1H""#, r#"duration("-PT1H")"#);
  te_value(false, scope, r#"@"P2Y""#, r#"duration("P2Y")"#);
}

#[test]
fn test_between() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "2 between 1 and 4", true);
  te_bool(false, scope, "1 between 1 and 4", true);
  te_bool(false, scope, "4 between 1 and 4", true);
  te_bool(false, scope, "0.99 between 1 and 4", false);
  te_bool(false, scope, "4.01 between 1 and 4", false);
  te_bool(false, scope, "6 between 1 and 4 + 2", true);
  te_bool(false, scope, "2 between 1 + 1 and 4 + 2", true);
  te_bool(false, scope, "5 - 2 between 1 + 2 and 10.2/2", true);
  te_bool(false, scope, "5 - 2 - 0.1 between 1 + 2 and 10.2/2", false);
  te_bool(false, scope, "true = (2 between 1 and 4)", true);
  te_bool(false, scope, "(2 between 1 and 4) = true", true);
  te_bool(false, scope, "(2 between 1 and 4) = (5 between 3 and 8)", true);
}

#[test]
fn test_comments() {
  let scope = &te_scope("{}");
  te_number(
    false,
    scope,
    r#"  1  // eol comment
         + 1"#,
    2.0,
  );
  te_number(
    false,
    scope,
    r#" 1
          /*
          some intro waffle
          */
          + 1"#,
    2.0,
  );
  te_number(false, scope, r#"1 + /* 1 + */ 1"#, 2.0);
  te_number(
    false,
    scope,
    r#" 1
          /*
          some intro waffle
          */
          + 1 // and stuff
          + 2"#,
    4.0,
  );
}

#[test]
fn test_comparison_in() {
  let scope = &te_scope("{ a: 100.0, b: 99.0, c: 101.0}");
  te_none(false, scope, "2 in ()");
  te_bool(false, scope, "2 in []", false);
  te_bool(false, scope, "2 in [1..5]", true);
  te_bool(false, scope, "99 in <=100", true);
  te_bool(false, scope, "(b) in <=100", true);
  te_bool(false, scope, "b in <=100", true);
  te_bool(false, scope, "100 in <=100", true);
  te_bool(false, scope, "(a) in <=100", true);
  te_bool(false, scope, "a in <=100", true);
  te_bool(false, scope, "101 in <=100", false);
  te_bool(false, scope, "99 in <100", true);
  te_bool(false, scope, "(b) in <100", true);
  te_bool(false, scope, "b in <100", true);
  te_bool(false, scope, "100 in <100", false);
  te_bool(false, scope, "(a) in <100", false);
  te_bool(false, scope, "101 in >=100", true);
  te_bool(false, scope, "100 in >=100", true);
  te_bool(false, scope, "(a) in >=100", true);
  te_bool(false, scope, "99 in >=100", false);
  te_bool(false, scope, "(b) in >=100", false);
  te_bool(false, scope, "b in >=100", false);
  te_bool(false, scope, "101 in >100", true);
  te_bool(false, scope, "100 in >100", false);
  te_bool(false, scope, "(a) in >100", false);
  te_bool(false, scope, "a in >100", false);
  te_bool(false, scope, "2 in (2)", true);
  te_bool(false, scope, "2 in (3)", false);
  te_bool(false, scope, "2 in (1,2,3,4,5)", true);
  te_bool(false, scope, "7 in (1,2,3,4,5)", false);
  te_bool(false, scope, "(a) in (1,2,3,4,5)", false);
  te_bool(false, scope, "2 in (<3)", true);
  te_bool(false, scope, "6 in (>5)", true);
  te_bool(false, scope, "2 in (<3,>5)", true);
  te_bool(false, scope, "3 in (<3,>5)", false);
  te_bool(false, scope, "4.12 in (<3,>5)", false);
  te_bool(false, scope, "5 in (<3,>5)", false);
  te_bool(false, scope, "2 in (>5,<3)", true);
  te_bool(false, scope, "5 in (>5,<3)", false);
  te_bool(false, scope, "4.5 in (>5,<3)", false);
  te_bool(false, scope, "3 in (>5,<3)", false);
  te_bool(false, scope, "2 in (<=3)", true);
  te_bool(false, scope, "2 in (<=3,>=5)", true);
  te_bool(false, scope, "3 in (<=3,>=5)", true);
  te_bool(false, scope, "5 in (<=3,>=5)", true);
  te_bool(false, scope, "4 in (<=3,>=5)", false);
  te_bool(false, scope, "2 in (>=5,<=3)", true);
  te_bool(false, scope, "3 in (>=5,<=3)", true);
  te_bool(false, scope, "5 in (>=5,<=3)", true);
  te_bool(false, scope, "4 in (>=5,<=3)", false);
  te_bool(false, scope, "not(4 in (1,3))", true);
  te_bool(false, scope, "not(5.25 in (1.32,2.45,4.12,5.25))", false);
  te_bool(false, scope, "5 in (<=5)", true);
  te_bool(false, scope, "5 in ((5..10])", false);
  te_bool(false, scope, "5 in ([5..10])", true);
  te_bool(false, scope, "5 in (4,5,6)", true);
  te_bool(false, scope, "5 in (<5,>5)", false);
  te_bool(false, scope, "1 in [2,3,1]", true);
  te_bool(false, scope, r#""k" in ["j".."l"]"#, true);
  te_bool(false, scope, r#""b" in [["f".."h"], ["a".."c"]]"#, true);
  te_bool(false, scope, r#""a" in <= "b""#, true);
  te_bool(false, scope, r#"true in [false, 2, 3]"#, false);
  te_bool(false, scope, r#"true in true"#, true);
  te_bool(
    false,
    scope,
    r#"date("2018-12-08") in [date("2018-12-08"),date("2018-12-09"),date("2018-12-10")]"#,
    true,
  );
  te_bool(false, scope, r#"date("2018-12-04") in <= date("2018-12-05")"#, true);
  te_bool(false, scope, r#"[1,2,3] in [[1,2,3,4], [1,2,3]]"#, true);
  te_bool(false, scope, r#"[1,2,2] in [[1,2,3,4], [1,2,3]]"#, false);
  te_bool(false, scope, r#"{a: "foo"} in [{b: "bar"}, {a: "foo"}]"#, true);
  te_bool(false, scope, r#"duration("P11Y") in [duration("P8Y"),duration("P9Y"),duration("P10Y")]"#, false);
  te_bool(
    false,
    scope,
    r#"duration("P11Y") in [[duration("P5Y") .. duration("P7Y")], [duration("P10Y") .. duration("P12Y")]]"#,
    true,
  );
  te_bool(false, scope, r#"duration("P11Y") in > duration("P10Y")"#, true);
}

#[test]
fn test_conjunction() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "true and true", true);
  te_bool(false, scope, "true and true and true", true);
  te_bool(false, scope, "true and true and false", false);
  te_bool(false, scope, "true and false", false);
  te_null(false, scope, "true and 1", "");
  te_null(false, scope, r#" true and "true" "#, "");
  te_null(false, scope, "true and null", "");
  te_bool(false, scope, "true and (false)", false);
  te_bool(false, scope, "false and true", false);
  te_bool(false, scope, "false and false", false);
  te_bool(false, scope, "false and 1", false);
  te_bool(false, scope, "false and null", false);
  te_bool(false, scope, r#" false and "true" "#, false);
  te_null(false, scope, "1 and true", "");
  te_null(false, scope, "null and true", "");
  te_bool(false, scope, "1 and false", false);
  te_bool(false, scope, "null and false", false);
  te_null(false, scope, "1 and 1", "");
  te_null(false, scope, "null and null", "");
  te_null(false, scope, "10.2 and 10.2", "");
  te_bool(false, scope, "(1 < 2) and (3 > 1)", true);
  te_bool(false, scope, "(1 > 2) and (3 between 1 and 2) or true", true);
}

#[test]
fn test_date_and_time() {
  let scope = &te_scope("{}");
  te_date_time_local(false, scope, r#"date and time("2012-12-24")"#, (2012, 12, 24), (0, 0, 0, 0));
  te_date_time(
    false,
    scope,
    r#"date and time("2012-12-24T23:59:00")"#,
    FeelDateTime::local(2012, 12, 24, 23, 59, 0, 0),
  );
  te_date_time(
    false,
    scope,
    "date    and  \n \t  time  ( \"2012-12-24T23:59:00\"  )   ",
    FeelDateTime::local(2012, 12, 24, 23, 59, 0, 0),
  );
  te_date_time(
    false,
    scope,
    r#"date and time("-2017-02-28T02:02:02")"#,
    FeelDateTime::local(-2017, 2, 28, 2, 2, 2, 0),
  );
  te_date_time(
    false,
    scope,
    r#"date and time("-2016-01-30T09:05:00")"#,
    FeelDateTime::local(-2016, 1, 30, 9, 5, 0, 0),
  );
  te_date_time(
    false,
    scope,
    r#"date and time("2015-12-31T23:59:59.9999999")"#,
    FeelDateTime::local(2015, 12, 31, 23, 59, 59, 999_999_900),
  );
  te_date_time(
    false,
    scope,
    r#"date and time("2018-10-01T12:32:59.111111")"#,
    FeelDateTime::local(2018, 10, 1, 12, 32, 59, 111_111_000),
  );
  te_date_time(
    false,
    scope,
    r#"date and time("2018-10-01T12:32:59.123123123123")"#,
    FeelDateTime::local(2018, 10, 1, 12, 32, 59, 123_123_123),
  );
  te_date_time(
    false,
    scope,
    r#"date and time("2012-12-24T23:59:00Z")"#,
    FeelDateTime::utc(2012, 12, 24, 23, 59, 0, 0),
  );
  te_date_time(
    false,
    scope,
    r#"date and time("2012-12-24T23:59:00z")"#,
    FeelDateTime::utc(2012, 12, 24, 23, 59, 0, 0),
  );
  te_date_time(
    false,
    scope,
    r#"date and time("2016-12-24T23:59:00-08:00")"#,
    FeelDateTime::utc(2016, 12, 25, 7, 59, 0, 0),
  );
  te_bool(
    false,
    scope,
    r#"date and time("2012-12-24T23:59:00") in [date and time("2012-12-24T23:59:00")..date and time("2012-12-24T23:59:00")]"#,
    true,
  );
  te_string(false, scope, r#"string(date and time("2016-12-24T23:59:00"))"#, "2016-12-24T23:59:00");
  te_string(false, scope, r#"string(date and time("2016-12-24T23:59:00Z"))"#, "2016-12-24T23:59:00Z");
  te_string(false, scope, r#"string(date and time("2016-12-24T23:59:00z"))"#, "2016-12-24T23:59:00Z");
  te_string(
    false,
    scope,
    r#"string(date and time("2016-12-24T23:59:00-08:00"))"#,
    "2016-12-24T23:59:00-08:00",
  );
  te_string(
    false,
    scope,
    r#"string(date and time("2016-12-24T23:59:00+02:12"))"#,
    "2016-12-24T23:59:00+02:12",
  );
  te_string(
    false,
    scope,
    r#"string(date and time("2016-12-24T23:59:00+14:59"))"#,
    "2016-12-24T23:59:00+14:59",
  );
  te_string(
    false,
    scope,
    r#"string(date and time("2016-12-24T23:59:00-14:59"))"#,
    "2016-12-24T23:59:00-14:59",
  );
  te_string(
    false,
    scope,
    r#"string(date and time("2016-12-24T23:59:00@Etc/UTC"))"#,
    "2016-12-24T23:59:00@Etc/UTC",
  );
  te_string(
    false,
    scope,
    r#"string(date and time("2016-12-24T23:59:00@Europe/Warsaw"))"#,
    "2016-12-24T23:59:00@Europe/Warsaw",
  );
  te_string(
    false,
    scope,
    r#"string(date and time("999999999-12-31T23:59:59.999999999@Europe/Paris"))"#,
    "999999999-12-31T23:59:59.999999999@Europe/Paris",
  );
  te_string(
    false,
    scope,
    r#"string(date and time("999999999-12-31T23:59:59.999999999@Europe/Paris"))"#,
    "999999999-12-31T23:59:59.999999999@Europe/Paris",
  );
  let scope = &te_scope(r#"{dateTimeString:"2016-12-24T23:59:00-08:00"}"#);
  te_string(false, scope, r#"string(date and time(dateTimeString))"#, "2016-12-24T23:59:00-08:00");
  te_null(false, scope, r#"date and time("2012-13-24T23:59:00")"#, "");
  te_null(false, scope, r#"date and time("2012-12-24T13:65:00")"#, "");
  te_null(false, scope, r#"date and time("2016-12-24T23:59:00+25:00")"#, "");
  te_null(false, scope, r#"date and time("2016-12-24T23:59:00-27:30")"#, "");
  te_null(false, scope, r#"date and time("2017-12-31T13:20:00@xyz/abc")"#, "");
}

#[test]
fn test_disjunction() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "true or true", true);
  te_bool(false, scope, "true or not(false)", true);
  te_bool(false, scope, "true or true or true", true);
  te_bool(false, scope, "true or true or false", true);
  te_bool(false, scope, "true or false", true);
  te_bool(false, scope, "true or 1", true);
  te_bool(false, scope, r#" true or "true" "#, true);
  te_bool(false, scope, "true or null", true);
  te_bool(false, scope, "false or true", true);
  te_bool(false, scope, "false or false", false);
  te_null(false, scope, "false or 1", "");
  te_null(false, scope, "false or null", "");
  te_null(false, scope, r#" false or "true" "#, "");
  te_bool(false, scope, "1 or true", true);
  te_bool(false, scope, "null or true", true);
  te_null(false, scope, "1 or false", "");
  te_null(false, scope, "null or false", "");
  te_null(false, scope, "1 or 1", "");
  te_null(false, scope, "null or null", "");
  te_null(false, scope, "10.2 or 10.2", "");
  te_bool(false, scope, "(1 < 2) or (3 > 1)", true);
  te_bool(false, scope, "((1 < 2) or (3 > 1)) and false", false);
  te_bool(false, scope, "((1 < 2) or (3 > 1)) or false", true);
}

#[test]
fn test_division() {
  let scope = &te_scope("{}");
  te_number(false, scope, "1/1", 1.0);
  te_number(false, scope, " 1 / 2 ", 0.5);
  te_number(false, scope, " 5 / 2 / 4 ", 0.625);
  te_number(false, scope, "10/2/5", 1.0);
  te_number(false, scope, "( 1 / 2 ) / ( 12 / 6 )", 0.25);
  te_number(false, scope, "( ( ( 6 / 3 ) ) )", 2.0);
  te_number(false, scope, "1/3", 1.0 / 3.0);
  te_number(false, scope, "1.01/2", 0.505);
  te_null(false, scope, "0.0 / 0.0", "division by zero");
}

#[test]
fn test_empty_input() {
  let scope = &te_scope("{}");
  te_none(false, scope, "");
  te_none(false, scope, "    ");
  te_none(false, scope, "\n");
  te_none(false, scope, "\u{000A}\u{000B}\u{000C}\u{000D}");
}

#[test]
fn test_equal() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "1=1", true);
  te_bool(false, scope, "100 = null", false);
  te_bool(false, scope, "1 = 1.000", true);
  te_bool(false, scope, "1.276=1.276", true);
  te_bool(false, scope, ".54635=.54635", true);
  te_bool(false, scope, "1=2", false);
  te_bool(false, scope, "(1+1)=2", true);
  te_bool(false, scope, "(1+2)=2", false);
  te_bool(false, scope, " ( 1 + 1 ) = 2", true);
  te_bool(false, scope, " ( 1 + 3 ) = 2", false);
  te_bool(false, scope, "(1+1)=(5-3)", true);
  te_bool(false, scope, "(1*2)=(10/5)", true);
  te_bool(false, scope, "true=true", true);
  te_bool(false, scope, "true=false", false);
  te_bool(false, scope, "false=true", false);
  te_bool(false, scope, "true=true", true);
  te_bool(false, scope, "true=null", false);
  te_bool(false, scope, r#"(10 = 10)"#, true);
  te_null(false, scope, r#"10 = ({c1: {a: {c: "bar", b: "foo"}}})"#, "");
  te_bool(false, scope, r#"{a: {c: "bar", b: "foo"}} = {a: {b: "foo", c: "bar"}}"#, true);
  te_bool(false, scope, r#"{a: {c: "bar", b: "foo"}} = {a: {b: "foo", c: "bars"}}"#, false);
  te_bool(false, scope, r#"[1,2,3] = [1,2,3]"#, true);
}

#[test]
fn test_exponentiation() {
  let scope = &te_scope("{}");
  te_number(false, scope, "2**0", 2_f64.powf(0.0));
  te_number(false, scope, "1**1", 1_f64.powf(1.0));
  te_number(false, scope, " 1 ** 2 ", 1_f64.powf(2.0));
  te_number(false, scope, " 5 **2** 3 ", 5_f64.powf(2.0).powf(3.0));
  te_number(false, scope, "10**2**-2", 10_f64.powf(2.0).powf(-2.0));
  te_number(false, scope, "( 1 ** 2 ) ** ( 3 ** 4 )", 1.0);
  te_number(false, scope, "( ( ( 4 ** 3 ) ) )", 64.0);
  te_number(false, scope, "2**(2+3)", 32.0);
  te_number(false, scope, "2**2+3", 7.0);
  te_number(false, scope, "1 + 3/2*2 - 2**3", -4.0);
  te_number(false, scope, "-3 ** 2", -(3_f64.powf(2.0)));
  te_number(false, scope, "1+3/2*2-2**3", -4.0);
  te_number(false, scope, "3 ** 4 ** 5", 3_f64.powf(4.0).powf(5.0));
  te_null(false, scope, "3 ** (4 ** 5)", "");
  te_null(false, scope, r#""foo" ** 4"#, "");
  te_null(false, scope, "true ** 4", "");
  te_null(false, scope, r#"date("2018-12-10") ** 4"#, "");
  te_null(false, scope, r#"time("10:30:00") ** 4"#, "");
  te_null(false, scope, r#"date and time("2018-12-10") ** 4"#, "");
  te_null(false, scope, r#"duration("P2Y") ** 4"#, "");
}

#[test]
fn test_filter() {
  let scope = &te_scope(r#"{A:[1,3,5],B:["A","B","C","D","E"]}"#);
  te_number(false, scope, "A[1]", 1.0);
  te_number(false, scope, "A[2]", 3.0);
  te_number(false, scope, "A[3]", 5.0);
  te_string(false, scope, "B[1]", "A");
  te_string(false, scope, "B[2]", "B");
  te_string(false, scope, "B[3]", "C");
  te_string(false, scope, "B[4]", "D");
  te_string(false, scope, "B[5]", "E");
  te_string(false, scope, "B[A[1]]", "A");
  te_string(false, scope, "B[A[2]]", "C");
  te_string(false, scope, "B[A[3]]", "E");
  let scope = &te_scope(r#"{l:[{x:1,y:2},{x:null,y:3}]}"#);
  te_be_value(false, scope, "l[x<2]", "{x:1,y:2}");
  te_be_value(false, scope, "[{x:1,y:2},{x:null,y:3}][x<2]", "{x:1,y:2}");
  te_be_value(false, scope, "[{x:1,y:2},{x:null,y:3}][y>2]", "{x:null,y:3}");
  te_be_value(false, scope, "[1,2,3,4,5,6,7,8,9,10][item>=5]", "[5,6,7,8,9,10]");
  let scope = &te_scope(r#"{l:[{x:1,y:2},{x:2,y:4},{x:3,y:6}]}"#);
  te_be_value(false, scope, "l[2]", "{x:2,y:4}");
  te_be_value(false, scope, "l[x=2]", "{x:2,y:4}");
  te_value(false, scope, "l[x=2].y", "4");
  let scope = &te_scope(r#"{}"#);
  te_number(false, scope, "[1,2,3,4,5,6][1]", 1.0);
  te_number(false, scope, "[1,2,3,4,5,6][item=4]", 4.0);
  te_be_value(false, scope, "[1,2,3,4,5,6][item>=4]", "[4,5,6]");
  te_number(false, scope, "[1,2,3,4,5,6][1][1]", 1.0);
  te_number(false, scope, "[1,2,3,4,5,6][2]", 2.0);
  te_number(false, scope, "[1,2,3,4,5,6][3]", 3.0);
  te_number(false, scope, "[1,2,3,4,5,6][4]", 4.0);
  te_number(false, scope, "[1,2,3,4,5,6][5]", 5.0);
  te_number(false, scope, "[1,2,3,4,5,6][6]", 6.0);
  te_number(false, scope, "[1,2,3,4,5,6][-1]", 6.0);
  te_number(false, scope, "[1,2,3,4,5,6][-2]", 5.0);
  te_number(false, scope, "[1,2,3,4,5,6][-3]", 4.0);
  te_number(false, scope, "[1,2,3,4,5,6][-4]", 3.0);
  te_number(false, scope, "[1,2,3,4,5,6][-5]", 2.0);
  te_number(false, scope, "[1,2,3,4,5,6][-6]", 1.0);
  te_null(false, scope, "[1,2,3,4,5,6][0]", "eval_filter_expression: exit");
  te_null(false, scope, "[1,2,3,4,5,6][7]", "eval_filter_expression: exit");
  te_null(false, scope, "[1,2,3,4,5,6][-7]", "eval_filter_expression: exit");
  te_be_value(false, scope, r#"true[true]"#, r#"[true]"#);
  be_be_value(false, scope, r#"[{a: 1}, {a: 2}, {a: 3}]"#, r#"[{a:1},{a:2},{a:3}]"#);
  te_be_value(false, scope, r#"[{a: 1}, {a: 2}, {a: 3}][1]"#, r#"{a:1}"#);
  te_be_value(false, scope, r#"[{item: 1}, {item: 2}, {item: 3}][item >= 2]"#, r#"[{item:2},{item:3}]"#);
  te_be_value(false, scope, r#"[{a: 1}, {a: 2}, {a: 3}][item.a=2]"#, r#"{a:2}"#);
  te_be_value(false, scope, r#"[{a: 1}, {a: 2}, {a: 3}][item.a >= 2]"#, r#"[{a:2},{a:3}]"#);
}

#[test]
fn test_function_invocation() {
  let scope = &te_scope_trace(r#"{add: function (x: number, y: number) x + y }"#);
  te_number(false, scope, r#"add(17.82,32.18)"#, 17.82 + 32.18);
}

#[test]
fn test_greater() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "2>1", true);
  te_bool(false, scope, "1.277>1.276", true);
  te_bool(false, scope, ".54635>-.54635", true);
  te_bool(false, scope, "2>3", false);
  te_bool(false, scope, "(1+1.1)>2.0", true);
  te_bool(false, scope, "(1.1+2)>3.11", false);
  te_bool(false, scope, " ( 1 + 0.99 ) > 1.9", true);
  te_bool(false, scope, " ( ( ( 1.1 + 3.1 ) ) ) > 4.21", false);
  te_bool(false, scope, "(0.9+1)>(5.1-3.3)", true);
  te_bool(false, scope, "(1*2)>(10.0/5.1)", true);
}

#[test]
fn test_greater_or_equal() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "2>=1", true);
  te_bool(false, scope, "1>=1", true);
  te_bool(false, scope, "1.277>=1.276", true);
  te_bool(false, scope, "1.276>=1.276", true);
  te_bool(false, scope, ".5>=-.54635", true);
  te_bool(false, scope, ".57>=-.57", true);
  te_bool(false, scope, "2>=2.1", false);
  te_bool(false, scope, "(1+1.2)>=2.01", true);
  te_bool(false, scope, "(1+1)>=2.0", true);
  te_bool(false, scope, "(1.1+2)>=3.11", false);
  te_bool(false, scope, " ( 1.02 + 0.99 ) >= 2.0", true);
  te_bool(false, scope, " ( 1 + 1.0 ) >= 2.0", true);
  te_bool(false, scope, " ( ( ( 1.1 + 3.1 ) ) ) >= 4.21", false);
  te_bool(false, scope, "(0.9+1)>=(5.1-3.21)", true);
  te_bool(false, scope, "(0.9+1.1)>=(5.0-3)", true);
  te_bool(false, scope, "(1*2)>=(10.0/5.1)", true);
  te_bool(false, scope, "(1*2)>=(10.0/5.0)", true);
}

#[test]
fn test_grouping_with_parentheses() {
  let scope = &te_scope(r#"{ Customer: "Private" }"#);
  te_none(false, scope, "()");
  te_number(false, scope, "(1)", 1.0);
  te_bool(false, scope, "(true)", true);
  te_bool(false, scope, "(false)", false);
  te_string(false, scope, r#"("beta")"#, "beta");
  te_string(false, scope, "(Customer)", "Private");
}

#[test]
fn test_instance_of() {
  let scope = &te_scope(r#"{Order:4.0,Customer:"Business",Delivery status:true}"#);
  te_bool(false, scope, "Order instance of number", true);
  te_bool(false, scope, "Order instance of string", false);
  te_bool(false, scope, "Order instance of boolean", false);
  te_bool(false, scope, "Customer instance of string", true);
  te_bool(false, scope, "Customer instance of number", false);
  te_bool(false, scope, "Customer instance of boolean", false);
  te_bool(false, scope, "Delivery status instance of boolean", true);
  te_bool(false, scope, "Delivery status instance of number", false);
  te_bool(false, scope, "Delivery status instance of string", false);
  let scope = &te_scope(r#"{Orders:[1,2,3,4]}"#);
  te_bool(false, scope, "Orders instance of list<number>", true);
  te_bool(false, scope, "Orders instance of list<string>", false);
  te_bool(false, scope, "Orders instance of list<boolean>", false);
  te_bool(false, scope, "Orders instance of list<date>", false);
  te_bool(false, scope, "Orders instance of list<time>", false);
  te_bool(false, scope, "Orders instance of list<date and time>", false);
  te_bool(false, scope, "Orders instance of list<years and months duration>", false);
  te_bool(false, scope, "Orders instance of list<days and time duration>", false);
  let scope = &te_scope(r#"{Items:[1..10]}"#);
  te_bool(false, scope, "Items instance of range<number>", true);
  te_bool(false, scope, "Items instance of range<string>", false);
  te_bool(false, scope, "Items instance of range<boolean>", false);
  te_bool(false, scope, "Items instance of range<date>", false);
  te_bool(false, scope, "Items instance of range<time>", false);
  te_bool(false, scope, "Items instance of range<date and time>", false);
  te_bool(false, scope, "Items instance of range<years and months duration>", false);
  te_bool(false, scope, "Items instance of range<days and time duration>", false);
  let scope = &te_scope(r#"{Person:{name:"John",age:49}}"#);
  te_bool(false, scope, "Person instance of context<name:string,age:number>", true);
  te_bool(false, scope, "Person instance of context<n:string,age:number>", false);
  te_bool(false, scope, "Person instance of context<name:string,a:number>", false);
  te_bool(false, scope, "Person instance of context<name:string,age:number,car:string>", false);
  te_bool(false, scope, "Person instance of context<name:string>", false);
  let scope = &te_scope(r#"{Person:{name:"John",age:49}}"#);
  te_bool(false, scope, "Person instance of function<string>->string", false);
}

#[test]
fn test_less() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "1<2", true);
  te_bool(false, scope, "1.276<1.277", true);
  te_bool(false, scope, "-.5<.54635", true);
  te_bool(false, scope, "2<1", false);
  te_bool(false, scope, "(1+1)<2.01", true);
  te_bool(false, scope, "(1.1+2)<3.0", false);
  te_bool(false, scope, " ( 1 + 0.99 ) < 2.0", true);
  te_bool(false, scope, " ( ( ( 1.1 + 3.1 ) ) ) < 2.5", false);
  te_bool(false, scope, "(0.9+1)<(5.1-3)", true);
  te_bool(false, scope, "(1*2)<(10.0/4.9)", true);
}

#[test]
fn test_less_or_equal() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "1<=2", true);
  te_bool(false, scope, "1<=1", true);
  te_bool(false, scope, "1.276<1.277", true);
  te_bool(false, scope, "1.276<=1.276", true);
  te_bool(false, scope, "-.5<.54635", true);
  te_bool(false, scope, "-.57<=.57", true);
  te_bool(false, scope, "2<=1", false);
  te_bool(false, scope, "(1+1)<=2.01", true);
  te_bool(false, scope, "(1+1)<=2.0", true);
  te_bool(false, scope, "(1.1+2)<=3.0", false);
  te_bool(false, scope, " ( 1 + 0.99 ) <= 2.0", true);
  te_bool(false, scope, " ( 1 + 1.0 ) <= 2.0", true);
  te_bool(false, scope, " ( ( ( 1.1 + 3.1 ) ) ) <= 2.5", false);
  te_bool(false, scope, "(0.9+1)<=(5.1-3)", true);
  te_bool(false, scope, "(0.9+1.1)<=(5.0-3)", true);
  te_bool(false, scope, "(1*2)<=(10.0/4.9)", true);
  te_bool(false, scope, "(1*2)<=(10.0/5.0)", true);
}

#[test]
fn test_list() {
  let scope = &te_scope("{}");
  satisfies(false, scope, "1", "", "[]", false);
  satisfies(false, scope, "1", "", "[1]", true);
  satisfies(false, scope, "10", "", "[9,10,11]", true);
  satisfies(false, scope, "8", "", "[9,10,11]", false);
  satisfies(false, scope, "12", "", "[9,10,11]", false);
  satisfies(false, scope, "10", "", "[9],[10],[11]", true);
  satisfies(false, scope, "8", "", "[9],[10],[11]", false);
  satisfies(false, scope, "12", "", "[9],[10],[11]", false);
}

#[test]
fn test_multiline() {
  let scope = &te_scope("{}");
  te_number(
    false,
    scope,
    r#"1 + 
    2
  + 3"#,
    6.0,
  );
}

#[test]
fn test_multiplication() {
  let scope = &te_scope("{Monthly Salary:10000}");
  te_number(false, scope, "1*1", 1.0);
  te_number(false, scope, " 1 * 2 ", 2.0);
  te_number(false, scope, " 5 *2 *3", 30.0);
  te_number(false, scope, "10*2*5", 100.0);
  te_number(false, scope, "( 1 * 2 ) * ( 3 * 4 )", 24.0);
  te_number(false, scope, "( ( ( 4 * 3 ) ) )", 12.0);
  te_number(false, scope, "(3*2)+(4*5)", 26.0);
  te_number(false, scope, "3*2+4*5", 26.0);
  te_number(false, scope, "3*(2+4)*5", 90.0);
  te_number(false, scope, ".10 * 30.00", 3.0);
  te_number(false, scope, "1.0*10**3", 1000.0);
  te_number(false, scope, "12 * Monthly Salary", 120000.0);
}

#[test]
fn test_name() {
  let scope = &te_scope(
    r#"{
        thing:2.0,
        one two three four:true,
        one and two:null,
        one or two:"found",
        before: {
          after: 1.11,
          or: {
            after: 1.12
          },
          and: {
            after: 1.13
          },
          between: {
            after: 1.14
          },
          next to between: {
            worm: 1.18
          }
        },
        income/loss:1.15,
        per/month/income/loss:1.17,
        a-b:1.19,
        to-be-or-not-to-be:1.20,
        that's:1.21,
        ok that's:1.22,
        bed+breakfast:1.23,
        night+and+day:1.24,
        fr**n*s:1.26,
        wh*t*v*r:1.27}"#,
  );
  te_number(false, scope, "thing", 2.0);
  te_bool(false, scope, "one two three four", true);
  te_null(false, scope, "one and two", "");
  te_string(false, scope, "one or two", "found");
  te_number(false, scope, "before.after", 1.11);
  te_number(false, scope, "before . next to between . worm", 1.18);
  te_number(false, scope, " before . after ", 1.11);
  te_number(false, scope, "before.and.after", 1.13);
  te_number(false, scope, "before. and. after", 1.13);
  te_number(false, scope, "income/loss", 1.15);
  te_number(false, scope, "income  /   loss", 1.15);
  te_number(false, scope, "per/month/income/loss", 1.17);
  te_number(false, scope, " per/ month / income/loss", 1.17);
  te_number(false, scope, "a-b", 1.19);
  te_number(false, scope, "to-be-or-not-to-be", 1.20);
  te_number(false, scope, "that's", 1.21);
  te_number(false, scope, "ok that's", 1.22);
  te_number(false, scope, "bed+breakfast", 1.23);
  te_number(false, scope, "night+and+day", 1.24);
  te_number(false, scope, "night + and + day", 1.24);
  te_number(false, scope, "fr**n*s", 1.26);
  te_number(false, scope, "wh*t*v*r", 1.27);
}

#[test]
fn test_negation() {
  let scope = &te_scope("{}");
  te_number(false, scope, "-0", 0.0);
  te_number(false, scope, "0", 0.0);
  te_number(false, scope, "--0", 0.0);
  te_number(false, scope, "-1", -1.0);
  te_number(false, scope, "--1", 1.0);
  te_number(false, scope, " -10.2 ", -10.2);
  te_number(false, scope, " - 10 ", -10.0);
  te_number(false, scope, "(-20)", -20.0);
  te_number(false, scope, " ( -21 ) ", -21.0);
  te_number(false, scope, " ( - 22 ) ", -22.0);
  te_number(false, scope, " ((( - 23 ))) ", -23.0);
  te_number(false, scope, " - - 24 ", 24.0);
  te_number(false, scope, " - ( - 25 ) ", 25.0);
}

#[test]
fn test_not_equal() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "1!=1", false);
  te_bool(false, scope, "1.25!=2.11", true);
  te_bool(false, scope, "1!=2", true);
  te_bool(false, scope, "(1+1)!=2", false);
  te_bool(false, scope, "(1+2)!=2", true);
  te_bool(false, scope, " ( 1 + 2 ) != 2", true);
  te_bool(false, scope, " ( 1 + 3 ) != 4", false);
  te_bool(false, scope, "(1+0.3)!=(5-3)", true);
  te_bool(false, scope, "(1*2)!=(10/5.1)", true);
  te_bool(false, scope, "true != false", true);
}

#[test]
fn test_properties() {
  let scope = &te_scope("{}");
  te_value(false, scope, r#"@"P1Y0M""#, r#"duration("P1Y")"#);
  te_number(false, scope, r#"date("2021-02-10").year"#, 2021.0);
  te_number(false, scope, r#"date("2021-02-10").month"#, 02.0);
  te_number(false, scope, r#"date("2021-02-10").day"#, 10.0);
  te_number(false, scope, r#"date("2021-02-10").weekday"#, 3.0);
  te_number(false, scope, r#"date and time("2018-12-10T10:30:01").year"#, 2018.0);
  te_number(false, scope, r#"date and time("2018-12-10T10:30:01").month"#, 12.0);
  te_number(false, scope, r#"date and time("2018-12-10T10:30:01").day"#, 10.0);
  te_number(false, scope, r#"date and time("2018-12-10T11:30:01").hour"#, 11.0);
  te_number(false, scope, r#"date and time("2018-12-10T11:30:01").hour"#, 11.0);
  te_number(false, scope, r#"date and time("2018-12-10T11:30:01").minute"#, 30.0);
  te_number(false, scope, r#"date and time("2018-12-10T11:30:01").second"#, 1.0);
  te_number(false, scope, r#"date and time("2021-02-10").hour"#, 0.0);
  te_number(false, scope, r#"date and time("2021-02-10").minute"#, 0.0);
  te_number(false, scope, r#"date and time("2021-02-10").second"#, 0.0);
  te_value(false, scope, r#"date and time("2018-12-10T10:30:00+05:00").time offset"#, r#"@"PT5H""#);
  te_null(false, scope, r#"date and time("2018-12-10T10:30:00").time offset"#, "");
  te_string(false, scope, r#"date and time("2018-12-10T10:30:00@Etc/UTC").timezone"#, r#"Etc/UTC"#);
  te_null(false, scope, r#"date and time("2018-12-10T10:30:00").timezone"#, "");
  te_number(false, scope, r#"time("08:45:27").hour"#, 8.0);
  te_number(false, scope, r#"time("08:45:27").minute"#, 45.0);
  te_number(false, scope, r#"time("08:45:27").second"#, 27.0);
  te_value(false, scope, r#"time("08:45:27-05:00").time offset"#, r#"@"-PT5H""#);
  te_null(false, scope, r#"time("08:45:27").time offset"#, "");
  te_string(false, scope, r#"time("08:45:27@Etc/UTC").timezone"#, r#"Etc/UTC"#);
  te_null(false, scope, r#"time("08:45:27").timezone"#, "");
  te_number(false, scope, r#"duration("P1Y2M").years"#, 1.0);
  te_number(false, scope, r#"duration("P2M").years"#, 0.0);
  te_number(false, scope, r#"duration("P2M").months"#, 2.0);
  te_number(false, scope, r#"duration("P1Y").months"#, 0.0);
  te_null(false, scope, r#"duration("P1Y").days"#, "no such property in years and months duration");
  te_null(false, scope, r#"duration("P1Y").hours"#, "no such property in years and months duration");
  te_null(false, scope, r#"duration("P1Y").minutes"#, "no such property in years and months duration");
  te_null(false, scope, r#"duration("P1Y").seconds"#, "no such property in years and months duration");
  te_null(false, scope, r#"duration("P1D").years"#, "no such property in days and time duration");
  te_null(false, scope, r#"duration("P1D").months"#, "no such property in days and time duration");
  te_number(false, scope, r#"duration("P1D").days"#, 1.0);
  te_number(false, scope, r#"duration("PT2H").days"#, 0.0);
  te_number(false, scope, r#"duration("PT2H").hours"#, 2.0);
  te_number(false, scope, r#"duration("P1D").hours"#, 0.0);
  te_number(false, scope, r#"duration("P1DT3H").hours"#, 3.0);
  te_number(false, scope, r#"duration("PT2M").minutes"#, 2.0);
  te_number(false, scope, r#"duration("P1D").minutes"#, 0.0);
  te_number(false, scope, r#"duration("PT5S").seconds"#, 5.0);
  te_number(false, scope, r#"duration("P1D").seconds"#, 0.0);
  te_number(false, scope, r#"duration("P3DT15H47M13S").days"#, 3.0);
  te_number(false, scope, r#"duration("P3DT15H47M13S").hours"#, 15.0);
  te_number(false, scope, r#"duration("P3DT15H47M13S").minutes"#, 47.0);
  te_number(false, scope, r#"duration("P3DT15H47M13S").seconds"#, 13.0);
}

#[test]
fn test_range() {
  let scope = &te_scope("{}");
  te_bool(false, scope, "2 in [1..5]", true);
  te_bool(false, scope, "0 in [1..5]", false);
  te_bool(false, scope, "6 in [1..5]", false);
  te_bool(false, scope, "1 in [1..5]", true);
  te_bool(false, scope, "5 in [1..5]", true);
  te_bool(false, scope, "1 in (1..5]", false);
  te_bool(false, scope, "5 in [1..5)", false);
  te_bool(false, scope, "1 in (1..5)", false);
  te_bool(false, scope, "1.01 in (1..5)", true);
  te_bool(false, scope, "5 in (1..5)", false);
  te_bool(false, scope, "4.99 in (1..5)", true);
  te_bool(false, scope, "1 in ]1..5]", false);
  te_bool(false, scope, "5 in [1..5[", false);
  te_bool(false, scope, "1 in ]1..5[", false);
  te_bool(false, scope, "5 in ]1..5[", false);
  te_bool(false, scope, "1 in [1..5] and 5 in [1..5]", true);
  te_bool(false, scope, "1 in [1..5] or 6 in [1..5]", true);
}

#[test]
fn test_subtraction() {
  let scope = &te_scope("{ a: 11.2, b: 0.2, c: 5.351 }");
  te_number(false, scope, "1-1", 0.0);
  te_number(false, scope, " 1 - 2 ", -1.0);
  te_number(false, scope, " 5 -2 -1 ", 2.0);
  te_number(false, scope, "1000-200-2", 798.0);
  te_number(false, scope, "( 1 - 2 ) - ( 3 - 4 )", 0.0);
  te_number(false, scope, "( ( ( 4 - 3 ) ) )", 1.0);
  te_number(false, scope, "a-b", 11.0);
  te_number(false, scope, "a - b", 11.0);
  te_number(false, scope, "a-b-c", 5.649);
  te_number(false, scope, " a  \n -  b \n  -  c", 5.649);
}
