/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL and DMN model evaluator.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Implementation of built-in functions.

/// Built-in functions with named parameters.
pub mod named {
  use crate::trace_named_parameter_not_found;
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::dmntk_common::Result;
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::values::{Value, VALUE_NULL, VALUE_NULL_WITH_TRACE};
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::Name;

  lazy_static! {
    static ref NAME_DATE: Name = Name::from("date");
    static ref NAME_DECIMAL_SEPARATOR: Name = Name::new(&["decimal", "separator"]);
    static ref NAME_DELIMITER: Name = Name::from("delimiter");
    static ref NAME_GROUPING_SEPARATOR: Name = Name::new(&["grouping", "separator"]);
    static ref NAME_DAY: Name = Name::from("day");
    static ref NAME_DIVIDEND: Name = Name::from("dividend");
    static ref NAME_DIVISOR: Name = Name::from("divisor");
    static ref NAME_FLAGS: Name = Name::from("flags");
    static ref NAME_FROM: Name = Name::from("from");
    static ref NAME_HOUR: Name = Name::from("hour");
    static ref NAME_INPUT: Name = Name::from("input");
    static ref NAME_LENGTH: Name = Name::from("length");
    static ref NAME_LIST: Name = Name::from("list");
    static ref NAME_MATCH: Name = Name::from("match");
    static ref NAME_MONTH: Name = Name::from("month");
    static ref NAME_MINUTE: Name = Name::from("minute");
    static ref NAME_N: Name = Name::from("n");
    static ref NAME_NEGAND: Name = Name::from("negand");
    static ref NAME_NUMBER: Name = Name::from("number");
    static ref NAME_OFFSET: Name = Name::from("offset");
    static ref NAME_PATTERN: Name = Name::from("pattern");
    static ref NAME_POSITION: Name = Name::from("position");
    static ref NAME_REPLACEMENT: Name = Name::from("replacement");
    static ref NAME_SCALE: Name = Name::from("scale");
    static ref NAME_SECOND: Name = Name::from("second");
    static ref NAME_START_POSITION: Name = Name::new(&["start", "position"]);
    static ref NAME_STRING: Name = Name::from("string");
    static ref NAME_TIME: Name = Name::from("time");
    static ref NAME_TO: Name = Name::from("to");
    static ref NAME_YEAR: Name = Name::from("year");
  }

  pub fn abs(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_N]) {
      Ok(super::core::abs(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn all(named_parameters: &Value) -> Result<Value> {
    if let Some(Value::List(list)) = get_parameter(named_parameters, &[&NAME_LIST]) {
      return Ok(super::core::all(list.to_vec()));
    }
    trace_named_parameter_not_found!();
    Ok(VALUE_NULL)
  }

  pub fn any(named_parameters: &Value) -> Result<Value> {
    if let Some(Value::List(list)) = get_parameter(named_parameters, &[&NAME_LIST]) {
      return Ok(super::core::any(list.to_vec()));
    }
    trace_named_parameter_not_found!();
    Ok(VALUE_NULL)
  }

  pub fn ceiling(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_N]) {
      Ok(super::core::ceiling(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn contains(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      if let Some(match_string_value) = get_parameter(named_parameters, &[&NAME_MATCH]) {
        Ok(super::core::contains(&input_string_value, &match_string_value))
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn count(named_parameters: &Value) -> Result<Value> {
    if let Some(list) = get_parameter(named_parameters, &[&NAME_LIST]) {
      return Ok(super::core::count(list));
    }
    Ok(VALUE_NULL_WITH_TRACE /*no 'list' parameter*/)
  }

  pub fn date(named_parameters: &Value) -> Result<Value> {
    if let Some(from) = get_parameter(named_parameters, &[&NAME_FROM]) {
      return Ok(super::core::date_1(from));
    }
    if let Some(year) = get_parameter(named_parameters, &[&NAME_YEAR]) {
      if let Some(month) = get_parameter(named_parameters, &[&NAME_MONTH]) {
        if let Some(day) = get_parameter(named_parameters, &[&NAME_DAY]) {
          return Ok(super::core::date_3(year, month, day));
        }
      }
    }
    Ok(VALUE_NULL_WITH_TRACE /*no 'list' parameter*/)
  }

  pub fn date_and_time(named_parameters: &Value) -> Result<Value> {
    if let Some(from_value) = get_parameter(named_parameters, &[&NAME_FROM]) {
      return Ok(super::core::date_and_time_1(from_value));
    }
    if let Some(date_value) = get_parameter(named_parameters, &[&NAME_DATE]) {
      if let Some(time_value) = get_parameter(named_parameters, &[&NAME_TIME]) {
        return Ok(super::core::date_and_time_2(date_value, time_value));
      }
    }
    Ok(VALUE_NULL_WITH_TRACE /*no 'list' parameter*/)
  }

  pub fn decimal(named_parameters: &Value) -> Result<Value> {
    if let Some(number) = get_parameter(named_parameters, &[&NAME_N]) {
      if let Some(scale) = get_parameter(named_parameters, &[&NAME_SCALE]) {
        Ok(super::core::decimal(&number, &scale))
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn duration(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_FROM]) {
      Ok(super::core::duration(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn ends_with(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      if let Some(match_string_value) = get_parameter(named_parameters, &[&NAME_MATCH]) {
        Ok(super::core::ends_with(&input_string_value, &match_string_value))
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn even(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_NUMBER]) {
      Ok(super::core::even(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn exp(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_NUMBER]) {
      Ok(super::core::exp(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn floor(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_N]) {
      Ok(super::core::floor(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn log(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_NUMBER]) {
      Ok(super::core::log(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn lower_case(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      Ok(super::core::lower_case(&input_string_value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn max(named_parameters: &Value) -> Result<Value> {
    if let Some(Value::List(list)) = get_parameter(named_parameters, &[&NAME_LIST]) {
      return Ok(super::core::max(list.to_vec()));
    }
    trace_named_parameter_not_found!();
    Ok(VALUE_NULL)
  }

  pub fn mean(named_parameters: &Value) -> Result<Value> {
    if let Some(Value::List(list)) = get_parameter(named_parameters, &[&NAME_LIST]) {
      return Ok(super::core::median(list.to_vec()));
    }
    Ok(VALUE_NULL_WITH_TRACE /*???*/)
  }

  pub fn median(named_parameters: &Value) -> Result<Value> {
    if let Some(Value::List(list)) = get_parameter(named_parameters, &[&NAME_LIST]) {
      return Ok(super::core::median(list.to_vec()));
    }
    trace_named_parameter_not_found!();
    Ok(VALUE_NULL)
  }

  pub fn min(named_parameters: &Value) -> Result<Value> {
    if let Some(Value::List(list)) = get_parameter(named_parameters, &[&NAME_LIST]) {
      return Ok(super::core::min(list.to_vec()));
    }
    trace_named_parameter_not_found!();
    Ok(VALUE_NULL)
  }

  pub fn modulo(named_parameters: &Value) -> Result<Value> {
    if let Some(dividend) = get_parameter(named_parameters, &[&NAME_DIVIDEND]) {
      if let Some(divisor) = get_parameter(named_parameters, &[&NAME_DIVISOR]) {
        Ok(super::core::modulo(&dividend, &divisor))
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn mode(named_parameters: &Value) -> Result<Value> {
    if let Some(Value::List(list)) = get_parameter(named_parameters, &[&NAME_LIST]) {
      return Ok(super::core::mode(list.to_vec()));
    }
    trace_named_parameter_not_found!();
    Ok(VALUE_NULL)
  }

  pub fn not(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_NEGAND]) {
      Ok(super::core::not(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn number(named_parameters: &Value) -> Result<Value> {
    if let Some(from) = get_parameter(named_parameters, &[&NAME_FROM]) {
      if let Some(grouping_separator) = get_parameter(named_parameters, &[&NAME_GROUPING_SEPARATOR]) {
        if let Some(decimal_separator) = get_parameter(named_parameters, &[&NAME_DECIMAL_SEPARATOR]) {
          Ok(super::core::number(&from, &grouping_separator, &decimal_separator))
        } else {
          trace_named_parameter_not_found!();
          Ok(VALUE_NULL)
        }
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn odd(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_NUMBER]) {
      Ok(super::core::odd(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn remove(named_parameters: &Value) -> Result<Value> {
    if let Some(list) = get_parameter(named_parameters, &[&NAME_LIST]) {
      if let Some(position) = get_parameter(named_parameters, &[&NAME_POSITION]) {
        Ok(super::core::remove(list, position))
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn replace(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_INPUT]) {
      if let Some(pattern_string_value) = get_parameter(named_parameters, &[&NAME_PATTERN]) {
        if let Some(replacement_string_value) = get_parameter(named_parameters, &[&NAME_REPLACEMENT]) {
          return if let Some(flags_string_value) = get_parameter(named_parameters, &[&NAME_FLAGS]) {
            Ok(super::core::replace(
              input_string_value,
              pattern_string_value,
              replacement_string_value,
              flags_string_value,
            ))
          } else {
            Ok(super::core::replace(
              input_string_value,
              pattern_string_value,
              replacement_string_value,
              &VALUE_NULL,
            ))
          };
        }
      }
    }
    trace_named_parameter_not_found!();
    Ok(VALUE_NULL)
  }

  pub fn split(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      if let Some(delimiter_string_value) = get_parameter(named_parameters, &[&NAME_DELIMITER]) {
        Ok(super::core::split(&input_string_value, &delimiter_string_value))
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn sqrt(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_NUMBER]) {
      Ok(super::core::sqrt(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn starts_with(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      if let Some(match_string_value) = get_parameter(named_parameters, &[&NAME_MATCH]) {
        Ok(super::core::starts_with(&input_string_value, &match_string_value))
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn stddev(named_parameters: &Value) -> Result<Value> {
    if let Some(Value::List(list)) = get_parameter(named_parameters, &[&NAME_LIST]) {
      return Ok(super::core::stddev(list.to_vec()));
    }
    trace_named_parameter_not_found!();
    Ok(VALUE_NULL)
  }

  pub fn string(named_parameters: &Value) -> Result<Value> {
    if let Some(value) = get_parameter(named_parameters, &[&NAME_FROM]) {
      Ok(super::core::string(&value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn string_length(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      Ok(super::core::string_length(&input_string_value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn substring(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      if let Some(start_position_value) = get_parameter(named_parameters, &[&NAME_START_POSITION]) {
        if let Some(length_value) = get_parameter(named_parameters, &[&NAME_LENGTH]) {
          Ok(super::core::substring(&input_string_value, &start_position_value, &length_value))
        } else {
          Ok(super::core::substring(&input_string_value, &start_position_value, &VALUE_NULL))
        }
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn substring_after(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      if let Some(input_string_match) = get_parameter(named_parameters, &[&NAME_MATCH]) {
        Ok(super::core::substring_after(&input_string_value, &input_string_match))
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn substring_before(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      if let Some(input_string_match) = get_parameter(named_parameters, &[&NAME_MATCH]) {
        Ok(super::core::substring_before(&input_string_value, &input_string_match))
      } else {
        trace_named_parameter_not_found!();
        Ok(VALUE_NULL)
      }
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn time(named_parameters: &Value) -> Result<Value> {
    if let Some(from_value) = get_parameter(named_parameters, &[&NAME_FROM]) {
      return Ok(super::core::time_1(&from_value));
    }
    if let Some(hour_value) = get_parameter(named_parameters, &[&NAME_HOUR]) {
      if let Some(minute_value) = get_parameter(named_parameters, &[&NAME_MINUTE]) {
        if let Some(second_value) = get_parameter(named_parameters, &[&NAME_SECOND]) {
          return if let Some(offset_value) = get_parameter(named_parameters, &[&NAME_OFFSET]) {
            Ok(super::core::time_4(hour_value, minute_value, second_value, offset_value))
          } else {
            Ok(super::core::time_3(hour_value, minute_value, second_value))
          };
        }
      }
    }
    Ok(VALUE_NULL_WITH_TRACE)
  }

  pub fn upper_case(named_parameters: &Value) -> Result<Value> {
    if let Some(input_string_value) = get_parameter(named_parameters, &[&NAME_STRING]) {
      Ok(super::core::upper_case(&input_string_value))
    } else {
      trace_named_parameter_not_found!();
      Ok(VALUE_NULL)
    }
  }

  pub fn years_and_months_duration(named_parameters: &Value) -> Result<Value> {
    if let Some(from_value) = get_parameter(named_parameters, &[&NAME_FROM]) {
      if let Some(to_value) = get_parameter(named_parameters, &[&NAME_TO]) {
        return Ok(super::core::years_and_months_duration(from_value, to_value));
      }
    }
    trace_named_parameter_not_found!();
    Ok(VALUE_NULL)
  }

  /// Returns reference to the value of the named parameter with specified name.
  /// Checks all names given in parameter `names`. The value of the first name found is returned.  
  fn get_parameter<'a>(named_parameters: &'a Value, names: &'a [&Name]) -> Option<&'a Value> {
    for name in names {
      if let Value::NamedParameters(map) = named_parameters {
        if let Some(a) = map.get(name) {
          return Some(a);
        }
      }
    }
    None
  }
}

/// Built-in functions with positional parameters.
pub mod positional {
  use crate::feel::evaluate;
  use crate::trace_invalid_number_of_parameters;
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::dmntk_common::Result;
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::values::{Value, VALUE_NULL, VALUE_NULL_WITH_TRACE};
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::{AstNode, Scope};

  pub fn abs(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::abs(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn all(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
      1 => match evaluate(scope, &positional_parameters[0])? {
        Value::List(values) => Ok(super::core::all(values.to_vec())),
        other => Ok(super::core::all(&[other])),
      },
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          values.push(evaluate(scope, parameter)?);
        }
        Ok(super::core::all(&values))
      }
    }
  }

  pub fn any(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
      1 => match evaluate(scope, &positional_parameters[0])? {
        Value::List(values) => Ok(super::core::any(values.to_vec())),
        other => Ok(super::core::any(&[other])),
      },
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          values.push(evaluate(scope, parameter)?);
        }
        Ok(super::core::any(&values))
      }
    }
  }

  pub fn append(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    if positional_parameters.len() > 1 {
      let list = &evaluate(scope, &positional_parameters[0])?;
      let mut values = vec![];
      for parameter in positional_parameters.iter().skip(1) {
        values.push(evaluate(scope, parameter)?);
      }
      Ok(super::core::append(list, &values))
    } else {
      trace_invalid_number_of_parameters!();
      Ok(VALUE_NULL)
    }
  }

  pub fn ceiling(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::ceiling(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn concatenate(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          values.push(evaluate(scope, parameter)?);
        }
        Ok(super::core::concatenate(&values))
      }
    }
  }

  pub fn contains(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let match_string_value = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::contains(&input_string_value, &match_string_value))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn count(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    if positional_parameters.len() == 1 {
      return Ok(super::core::count(&evaluate(scope, &positional_parameters[0])?));
    }
    Ok(VALUE_NULL_WITH_TRACE /*invalid number of parameters*/)
  }

  pub fn date(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::date_1(&evaluate(scope, &positional_parameters[0])?)),
      3 => Ok(super::core::date_3(
        &evaluate(scope, &positional_parameters[0])?,
        &evaluate(scope, &positional_parameters[1])?,
        &evaluate(scope, &positional_parameters[2])?,
      )),
      _ => Ok(VALUE_NULL_WITH_TRACE /*invalid number of parameters*/),
    }
  }

  pub fn date_and_time(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::date_and_time_1(&evaluate(scope, &positional_parameters[0])?)),
      2 => Ok(super::core::date_and_time_2(
        &evaluate(scope, &positional_parameters[0])?,
        &evaluate(scope, &positional_parameters[1])?,
      )),
      _ => Ok(VALUE_NULL_WITH_TRACE /*invalid number of parameters*/),
    }
  }

  pub fn decimal(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let number = &evaluate(scope, &positional_parameters[0])?;
        let scale = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::decimal(number, scale))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn distinct_values(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::distinct_values(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn duration(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::duration(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn ends_with(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let match_string_value = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::ends_with(&input_string_value, &match_string_value))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn even(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::even(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn exp(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::exp(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn flatten(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::flatten(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn floor(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::floor(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn index_of(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => Ok(super::core::index_of(
        &evaluate(scope, &positional_parameters[0])?,
        &evaluate(scope, &positional_parameters[1])?,
      )),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn insert_before(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      3 => Ok(super::core::insert_before(
        &evaluate(scope, &positional_parameters[0])?,
        &evaluate(scope, &positional_parameters[1])?,
        &evaluate(scope, &positional_parameters[2])?,
      )),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn list_contains(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => Ok(super::core::list_contains(
        &evaluate(scope, &positional_parameters[0])?,
        &evaluate(scope, &positional_parameters[1])?,
      )),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn log(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::log(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn lower_case(scope: &Scope, parameters: &[AstNode]) -> Result<Value> {
    match parameters.len() {
      1 => Ok(super::core::lower_case(&evaluate(scope, &parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn matches(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let pattern_string_value = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::matches(&input_string_value, &pattern_string_value, &VALUE_NULL))
      }
      3 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let pattern_string_value = &evaluate(scope, &positional_parameters[1])?;
        let flags_string_value = &evaluate(scope, &positional_parameters[2])?;
        Ok(super::core::matches(&input_string_value, &pattern_string_value, &flags_string_value))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn max(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
      1 => {
        // single parameter may be a list of values or a single comparable value
        match evaluate(scope, &positional_parameters[0])? {
          Value::List(values) => Ok(super::core::max(values.to_vec())),
          other => Ok(super::core::max(&[other])),
        }
      }
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          values.push(evaluate(scope, parameter)?);
        }
        Ok(super::core::max(&values))
      }
    }
  }

  pub fn mean(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => Ok(VALUE_NULL_WITH_TRACE /*no parameters*/),
      1 => match evaluate(scope, &positional_parameters[0])? {
        Value::List(values) => Ok(super::core::mean(values.to_vec())),
        other => Ok(super::core::mean(&[other])),
      },
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          let value = evaluate(scope, parameter)?;
          values.push(value);
        }
        Ok(super::core::mean(&values))
      }
    }
  }

  pub fn median(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
      1 => {
        // single parameter should be a list of numbers
        match evaluate(scope, &positional_parameters[0])? {
          Value::List(values) => Ok(super::core::median(values.to_vec())),
          other => Ok(super::core::median(&[other])),
        }
      }
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          let value = evaluate(scope, parameter)?;
          values.push(value);
        }
        Ok(super::core::median(&values))
      }
    }
  }

  pub fn min(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
      1 => {
        // single parameter may be a list of values or a single comparable value
        match evaluate(scope, &positional_parameters[0])? {
          Value::List(values) => Ok(super::core::min(values.to_vec())),
          other => Ok(super::core::min(&[other])),
        }
      }
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          values.push(evaluate(scope, parameter)?);
        }
        Ok(super::core::min(&values))
      }
    }
  }

  pub fn mode(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
      1 => {
        // single parameter should be a list of numbers
        match evaluate(scope, &positional_parameters[0])? {
          Value::List(values) => Ok(super::core::mode(values.to_vec())),
          other => Ok(super::core::mode(&[other])),
        }
      }
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          let value = evaluate(scope, parameter)?;
          values.push(value);
        }
        Ok(super::core::mode(&values))
      }
    }
  }

  pub fn modulo(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let dividend = &evaluate(scope, &positional_parameters[0])?;
        let divisor = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::modulo(dividend, divisor))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn not(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::not(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn number(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      3 => {
        let from = &evaluate(scope, &positional_parameters[0])?;
        let grouping_separator = &evaluate(scope, &positional_parameters[1])?;
        let decimal_separator = &evaluate(scope, &positional_parameters[2])?;
        Ok(super::core::number(from, grouping_separator, decimal_separator))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn odd(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::odd(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn remove(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let list = &evaluate(scope, &positional_parameters[0])?;
        let position = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::remove(list, position))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn replace(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      3 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let pattern_string_value = &evaluate(scope, &positional_parameters[1])?;
        let replacement_string_value = &evaluate(scope, &positional_parameters[2])?;
        Ok(super::core::replace(
          input_string_value,
          pattern_string_value,
          replacement_string_value,
          &VALUE_NULL,
        ))
      }
      4 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let pattern_string_value = &evaluate(scope, &positional_parameters[1])?;
        let replacement_string_value = &evaluate(scope, &positional_parameters[2])?;
        let flags_string_value = &evaluate(scope, &positional_parameters[3])?;
        Ok(super::core::replace(
          input_string_value,
          pattern_string_value,
          replacement_string_value,
          flags_string_value,
        ))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn reverse(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    if positional_parameters.len() == 1 {
      return Ok(super::core::reverse(&evaluate(scope, &positional_parameters[0])?));
    }
    Ok(VALUE_NULL)
  }

  pub fn split(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let delimiter_string_value = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::split(input_string_value, delimiter_string_value))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn sqrt(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::sqrt(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn starts_with(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let match_string_value = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::starts_with(input_string_value, match_string_value))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn stddev(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => Ok(VALUE_NULL),
      1 => match evaluate(scope, &positional_parameters[0])? {
        Value::List(values) => Ok(super::core::stddev(values.to_vec())),
        _ => Ok(VALUE_NULL),
      },
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          values.push(evaluate(scope, parameter)?);
        }
        Ok(super::core::stddev(&values))
      }
    }
  }

  pub fn string(scope: &Scope, parameters: &[AstNode]) -> Result<Value> {
    match parameters.len() {
      1 => Ok(super::core::string(&evaluate(scope, &parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn string_length(scope: &Scope, parameters: &[AstNode]) -> Result<Value> {
    match parameters.len() {
      1 => Ok(super::core::string_length(&evaluate(scope, &parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn sublist(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let list = &evaluate(scope, &positional_parameters[0])?;
        let position = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::sublist2(list, position))
      }
      3 => {
        let list = &evaluate(scope, &positional_parameters[0])?;
        let position = &evaluate(scope, &positional_parameters[1])?;
        let length = &evaluate(scope, &positional_parameters[2])?;
        Ok(super::core::sublist3(list, position, length))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn substring(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let start_position_value = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::substring(&input_string_value, &start_position_value, &VALUE_NULL))
      }
      3 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let start_position_value = &evaluate(scope, &positional_parameters[1])?;
        let length_value = &evaluate(scope, &positional_parameters[2])?;
        Ok(super::core::substring(&input_string_value, &start_position_value, &length_value))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn substring_after(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let match_string_value = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::substring_after(&input_string_value, &match_string_value))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn substring_before(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      2 => {
        let input_string_value = &evaluate(scope, &positional_parameters[0])?;
        let match_string_value = &evaluate(scope, &positional_parameters[1])?;
        Ok(super::core::substring_before(&input_string_value, &match_string_value))
      }
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn sum(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
      1 => {
        // single parameter may be a list of values or a single comparable value
        match evaluate(scope, &positional_parameters[0])? {
          Value::List(values) => Ok(super::core::sum(values.to_vec())),
          other => Ok(super::core::sum(&[other])),
        }
      }
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          let value = evaluate(scope, parameter)?;
          values.push(value);
        }
        Ok(super::core::sum(&values))
      }
    }
  }

  pub fn time(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => {
        let from_value = &evaluate(scope, &positional_parameters[0])?;
        return Ok(super::core::time_1(from_value));
      }
      3 => {
        let hour_value = &evaluate(scope, &positional_parameters[0])?;
        let minute_value = &evaluate(scope, &positional_parameters[1])?;
        let second_value = &evaluate(scope, &positional_parameters[2])?;
        return Ok(super::core::time_3(hour_value, minute_value, second_value));
      }
      4 => {
        let hour_value = &evaluate(scope, &positional_parameters[0])?;
        let minute_value = &evaluate(scope, &positional_parameters[1])?;
        let second_value = &evaluate(scope, &positional_parameters[2])?;
        let offset_value = &evaluate(scope, &positional_parameters[3])?;
        return Ok(super::core::time_4(hour_value, minute_value, second_value, offset_value));
      }
      _ => {}
    }
    Ok(VALUE_NULL_WITH_TRACE)
  }

  pub fn union(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      0 => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
      _ => {
        let mut values = vec![];
        for parameter in positional_parameters {
          let value = evaluate(scope, parameter)?;
          values.push(value);
        }
        Ok(super::core::union(&values))
      }
    }
  }

  pub fn upper_case(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    match positional_parameters.len() {
      1 => Ok(super::core::upper_case(&evaluate(scope, &positional_parameters[0])?)),
      _ => {
        trace_invalid_number_of_parameters!();
        Ok(VALUE_NULL)
      }
    }
  }

  pub fn years_and_months_duration(scope: &Scope, positional_parameters: &[AstNode]) -> Result<Value> {
    if positional_parameters.len() == 2 {
      let from = &evaluate(scope, &positional_parameters[0])?;
      let to = &evaluate(scope, &positional_parameters[1])?;
      return Ok(super::core::years_and_months_duration(from, to));
    }
    Ok(VALUE_NULL_WITH_TRACE)
  }
}

/// Core built-in functions used by named and positional implementations.
pub mod core {
  use crate::feel::evaluate_equals;
  use crate::{trace_invalid_value, trace_invalid_value_type};
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::dmntk_common::Stringify;
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::values::Value::YearsAndMonthsDuration;
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::values::{Value, Values, VALUE_FALSE, VALUE_NULL, VALUE_NULL_WITH_TRACE, VALUE_TRUE};
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::{
    round_half_to_even, FeelDate, FeelDateTime, FeelDaysAndTimeDuration, FeelTime, FeelYearsAndMonthsDuration,
  };
  use regex::Regex;
  use std::convert::TryFrom;

  pub fn abs(value: &Value) -> Value {
    if let Value::Number(v) = value {
      Value::Number(v.abs())
    } else {
      trace_invalid_value_type!("abs");
      VALUE_NULL
    }
  }

  pub fn all(values: &[Value]) -> Value {
    if values.is_empty() {
      return VALUE_TRUE;
    }
    for value in values {
      if let Value::Boolean(v) = value {
        if !v {
          return VALUE_FALSE;
        }
      } else {
        trace_invalid_value_type!("all");
        return VALUE_NULL;
      }
    }
    VALUE_TRUE
  }

  pub fn any(values: &[Value]) -> Value {
    if values.is_empty() {
      return VALUE_FALSE;
    }
    for value in values {
      if let Value::Boolean(v) = value {
        if *v {
          return VALUE_TRUE;
        }
      } else {
        trace_invalid_value_type!("any");
        return VALUE_NULL;
      }
    }
    VALUE_FALSE
  }

  pub fn append(list: &Value, values: &[Value]) -> Value {
    if let Value::List(items) = list {
      let mut appended = items.clone();
      for value in values {
        appended.add(value.clone());
      }
      return Value::List(appended);
    }
    trace_invalid_value_type!("append");
    VALUE_NULL
  }

  pub fn ceiling(value: &Value) -> Value {
    if let Value::Number(v) = value {
      Value::Number(v.ceil())
    } else {
      trace_invalid_value_type!("ceiling");
      VALUE_NULL
    }
  }

  pub fn concatenate(values: &[Value]) -> Value {
    let mut concatenated = vec![];
    for value in values {
      if let Value::List(items) = value {
        for item in items.to_vec() {
          concatenated.push(item.clone());
        }
      } else {
        trace_invalid_value_type!("concatenate");
        return VALUE_NULL;
      }
    }
    Value::List(Values::new(concatenated))
  }

  /// Returns **true** when the input string contains the match.
  pub fn contains(input_string_value: &Value, match_string_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      if let Value::String(match_string) = match_string_value {
        Value::Boolean(input_string.contains(match_string))
      } else {
        trace_invalid_value_type!("contains");
        VALUE_NULL
      }
    } else {
      trace_invalid_value_type!("contains");
      VALUE_NULL
    }
  }

  pub fn count(list: &Value) -> Value {
    if let Value::List(items) = list {
      return Value::Number(items.to_vec().len() as f64);
    }
    VALUE_NULL_WITH_TRACE /*parameter is not a list*/
  }

  pub fn date_1(value: &Value) -> Value {
    match value {
      Value::String(text) => {
        if let Ok(date) = FeelDate::try_from(text.as_str()) {
          Value::Date(date)
        } else {
          VALUE_NULL_WITH_TRACE
        }
      }
      Value::Date(date) => Value::Date(date.clone()),
      Value::DateTime(date_time) => Value::Date(date_time.date()),
      _ => VALUE_NULL_WITH_TRACE,
    }
  }

  pub fn date_3(year_value: &Value, month_value: &Value, day_value: &Value) -> Value {
    if let Value::Number(year) = year_value {
      if let Value::Number(month) = month_value {
        if let Value::Number(day) = day_value {
          if let Ok(date) = FeelDate::try_from((*year, *month, *day)) {
            Value::Date(date)
          } else {
            VALUE_NULL_WITH_TRACE
          }
        } else {
          VALUE_NULL_WITH_TRACE
        }
      } else {
        VALUE_NULL_WITH_TRACE
      }
    } else {
      VALUE_NULL_WITH_TRACE
    }
  }

  pub fn date_and_time_1(value: &Value) -> Value {
    if let Value::String(text) = value {
      if let Ok(date_time) = FeelDateTime::try_from(text.as_str()) {
        return Value::DateTime(date_time);
      }
      if let Ok(date) = FeelDate::try_from(text.as_str()) {
        return Value::DateTime(FeelDateTime::new(date, FeelTime::local(0, 0, 0, 0)));
      }
    }
    VALUE_NULL_WITH_TRACE
  }

  pub fn date_and_time_2(date_value: &Value, time_value: &Value) -> Value {
    match date_value {
      Value::DateTime(date_time) => {
        if let Value::Time(time) = time_value {
          return Value::DateTime(FeelDateTime::new(date_time.date(), time.clone()));
        }
      }
      Value::Date(date) => {
        if let Value::Time(time) = time_value {
          return Value::DateTime(FeelDateTime::new(date.clone(), time.clone()));
        }
      }
      _ => {}
    }
    VALUE_NULL_WITH_TRACE
  }

  /// Returns **n** with given **scale**.
  pub fn decimal(number: &Value, scale: &Value) -> Value {
    if let Value::Number(n) = number {
      if let Value::Number(s) = scale {
        let sc = s.trunc();
        if (-6111.0..6176.0).contains(&sc) {
          Value::Number(round_half_to_even(*n, sc))
        } else {
          trace_invalid_value_type!("decimal");
          VALUE_NULL
        }
      } else {
        trace_invalid_value_type!("decimal");
        VALUE_NULL
      }
    } else {
      trace_invalid_value_type!("decimal");
      VALUE_NULL
    }
  }

  pub fn distinct_values(value: &Value) -> Value {
    let mut result = vec![];
    if let Value::List(items) = value {
      for item in items.to_vec() {
        if result.iter().all(|a| !evaluate_equals(a, item)) {
          result.push(item.clone())
        }
      }
    } else {
      return VALUE_NULL_WITH_TRACE;
    }
    Value::List(Values::new(result))
  }

  /// Converts string value to a days and time or years and months duration.
  pub fn duration(value: &Value) -> Value {
    if let Value::String(s) = value {
      if let Ok(ym_duration) = FeelYearsAndMonthsDuration::try_from(s.as_str()) {
        Value::YearsAndMonthsDuration(ym_duration)
      } else if let Ok(dt_duration) = FeelDaysAndTimeDuration::try_from(s.as_str()) {
        Value::DaysAndTimeDuration(dt_duration)
      } else {
        trace_invalid_value_type!("duration");
        VALUE_NULL
      }
    } else {
      trace_invalid_value_type!("duration");
      VALUE_NULL
    }
  }

  /// Returns **true** when the input string ends with specified match string.
  pub fn ends_with(input_string_value: &Value, match_string_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      if let Value::String(match_string) = match_string_value {
        Value::Boolean(input_string.ends_with(match_string))
      } else {
        trace_invalid_value_type!("ends_with");
        VALUE_NULL
      }
    } else {
      trace_invalid_value_type!("ends_with");
      VALUE_NULL
    }
  }

  /// Returns true if number is even, false if it is odd.
  pub fn even(value: &Value) -> Value {
    if let Value::Number(v) = value {
      Value::Boolean((v.trunc() as i64) % 2 == 0)
    } else {
      trace_invalid_value_type!("even");
      VALUE_NULL
    }
  }

  /// Returns the Euler’s number e raised to the power of **value** given as a parameter.
  pub fn exp(value: &Value) -> Value {
    if let Value::Number(v) = value {
      Value::Number(v.exp())
    } else {
      trace_invalid_value_type!("exp");
      VALUE_NULL
    }
  }

  pub fn flatten(value: &Value) -> Value {
    if let Value::List(_) = value {
      let mut flattened = vec![];
      flatten_value(value, &mut flattened);
      return Value::List(Values::new(flattened));
    }
    trace_invalid_value_type!("flatten");
    VALUE_NULL
  }

  fn flatten_value(value: &Value, flattened: &mut Vec<Value>) {
    if let Value::List(items) = value {
      for item in items.to_vec() {
        if let Value::List(_) = item {
          flatten_value(item, flattened);
        } else {
          flattened.push(item.clone())
        }
      }
    }
  }

  /// Returns greatest **integer** <= **value** specified as a parameter.
  pub fn floor(value: &Value) -> Value {
    if let Value::Number(v) = value {
      Value::Number(v.floor())
    } else {
      trace_invalid_value_type!("floor");
      VALUE_NULL
    }
  }

  pub fn index_of(list: &Value, element: &Value) -> Value {
    if let Value::List(items) = list {
      let mut indexes = vec![];
      for (i, item) in items.to_vec().iter().enumerate() {
        if evaluate_equals(item, element) {
          indexes.push(Value::Number(i as f64));
        }
      }
      return Value::List(Values::new(indexes));
    }
    VALUE_NULL_WITH_TRACE
  }

  pub fn insert_before(list: &Value, position: &Value, new_item: &Value) -> Value {
    if let Value::List(mut items) = list.clone() {
      if let Value::Number(pos) = position {
        let index = pos.trunc() as i64;
        if index > 0 {
          let i = (index as usize) - 1;
          if i < items.len() {
            items.insert(i, new_item.clone());
            return Value::List(items);
          }
        }
        if index < 0 {
          let i = index.abs() as usize;
          if i <= items.to_vec().len() {
            items.insert(items.len() - i, new_item.clone());
            return Value::List(items);
          }
        }
      }
    }
    VALUE_NULL_WITH_TRACE /*probably index is out of range*/
  }

  pub fn list_contains(list: &Value, element: &Value) -> Value {
    if let Value::List(items) = list {
      for item in items.to_vec() {
        if evaluate_equals(item, element) {
          return VALUE_TRUE;
        }
      }
    }
    VALUE_FALSE
  }

  /// Returns the natural logarithm (base **e**) of the number parameter.
  pub fn log(number: &Value) -> Value {
    if let Value::Number(n) = number {
      if *n > 0.0 {
        Value::Number(n.ln())
      } else {
        VALUE_NULL
      }
    } else {
      trace_invalid_value_type!("log");
      VALUE_NULL
    }
  }

  /// Returns lower-cased string.
  pub fn lower_case(input_string_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      Value::String(input_string.to_lowercase())
    } else {
      trace_invalid_value_type!("lower_case");
      VALUE_NULL
    }
  }

  /// Returns `true` when the input matches the regexp pattern.
  pub fn matches(input_string_value: &Value, pattern_string_value: &Value, flags_string_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      if let Value::String(pattern_string) = pattern_string_value {
        if let Value::String(flags_string) = flags_string_value {
          if let Ok(re) = Regex::new(format!("(?{}){}", flags_string, pattern_string).as_str()) {
            return Value::Boolean(re.is_match(input_string));
          }
        } else if let Ok(re) = Regex::new(&pattern_string) {
          return Value::Boolean(re.is_match(input_string));
        }
      }
    }
    trace_invalid_value_type!("matches");
    VALUE_NULL
  }

  /// Returns the maximum value in the collection of comparable values.
  pub fn max(values: &[Value]) -> Value {
    if values.is_empty() {
      return VALUE_NULL;
    }
    return match &values[0] {
      Value::Number(n) => {
        let mut max = *n;
        for value in values.iter().skip(1) {
          match value {
            Value::Number(v) => {
              if *v > max {
                max = *v;
              }
            }
            Value::Null(_) => {}
            other => {
              trace_invalid_value_type!(format!("max: expected value of type number or Null, but encountered: {:?}", other.type_of()));
              return VALUE_NULL_WITH_TRACE;
            }
          }
        }
        Value::Number(max)
      }
      Value::String(s) => {
        let mut max = s.clone();
        for value in values.iter().skip(1) {
          match value {
            Value::String(v) => {
              if *v > max {
                max = v.clone();
              }
            }
            Value::Null(_) => {}
            other => {
              trace_invalid_value_type!(format!("max: expected value of type string or Null, but encountered: {:?}", other.type_of()));
              return VALUE_NULL_WITH_TRACE;
            }
          }
        }
        Value::String(max)
      }
      other => {
        trace_invalid_value_type!(format!("max: unhandled value type: {:?}", other.type_of()));
        VALUE_NULL_WITH_TRACE
      }
    };
  }

  /// Returns the mean of numbers.
  pub fn mean(values: &[Value]) -> Value {
    if values.is_empty() {
      return VALUE_NULL;
    }
    let mut sum = 0.0;
    for value in values {
      if let Value::Number(n) = value {
        sum += n;
      } else {
        return VALUE_NULL_WITH_TRACE/*not a number*/;
      }
    }
    Value::Number(sum / (values.len() as f64))
  }

  /// Returns the median of numbers.
  pub fn median(values: &[Value]) -> Value {
    if values.is_empty() {
      return VALUE_NULL;
    }
    let mut list = vec![];
    for value in values {
      if let Value::Number(n) = value {
        list.push(*n);
      } else {
        trace_invalid_value_type!("median");
        return VALUE_NULL;
      }
    }
    list.sort_by(|x, y| x.partial_cmp(y).unwrap_or(std::cmp::Ordering::Equal));
    let index = values.len() / 2;
    if list.len() % 2 == 0 {
      Value::Number((list[index - 1] + list[index]) / 2.0)
    } else {
      Value::Number(list[index])
    }
  }

  /// Returns the minimum value in the collection of comparable values.
  pub fn min(values: &[Value]) -> Value {
    if values.is_empty() {
      return VALUE_NULL;
    }
    match &values[0] {
      Value::Number(n) => {
        let mut min = *n;
        for value in values.iter().skip(1) {
          if let Value::Number(v) = value {
            if *v < min {
              min = *v;
            }
          } else {
            trace_invalid_value_type!("min");
            return VALUE_NULL;
          }
        }
        return Value::Number(min);
      }
      Value::String(s) => {
        let mut min = s.clone();
        for value in values.iter().skip(1) {
          if let Value::String(v) = value {
            if *v < min {
              min = v.clone();
            }
          } else {
            trace_invalid_value_type!("min");
            return VALUE_NULL;
          }
        }
        return Value::String(min);
      }
      _ => {}
    }
    trace_invalid_value_type!("min");
    VALUE_NULL
  }

  /// Returns the mode of numbers.
  pub fn mode(values: &[Value]) -> Value {
    if values.is_empty() {
      return Value::List(Values::empty());
    }
    // make sure all values are numbers and prepare the list of numbers
    let mut list = vec![];
    for value in values {
      if let Value::Number(n) = value {
        list.push(*n);
      } else {
        trace_invalid_value_type!("mode");
        return VALUE_NULL;
      }
    }
    // sort values in ascending order
    list.sort_by(|x, y| x.partial_cmp(y).unwrap_or(std::cmp::Ordering::Equal));
    // calculate the frequencies of the numbers
    let mut mode: Vec<(usize, f64)> = vec![];
    for x in list {
      if let Some((count, value)) = mode.pop() {
        if (x - value).abs() < f64::EPSILON {
          mode.push((count + 1, value));
        } else {
          mode.push((count, value));
          mode.push((1_usize, x));
        }
      } else {
        mode.push((1_usize, x));
      }
    }
    // sort frequencies in descending order, and when equal then by number in ascending order
    mode.sort_by(|x, y| match x.0.cmp(&y.0).reverse() {
      std::cmp::Ordering::Equal => x.1.partial_cmp(&y.1).unwrap_or(std::cmp::Ordering::Equal),
      other => other,
    });
    // there must be minimum one element in the list but to be sure check it
    if let Some((max, _)) = mode.get(0) {
      // return items with maximum frequency
      return Value::List(Values::new(
        mode
          .iter()
          .filter_map(|(c, v)| if *c == *max { Some(Value::Number(*v)) } else { None })
          .collect(),
      ));
    }
    trace_invalid_value_type!("mode");
    VALUE_NULL
  }

  /// Returns the remainder of the division of dividend by divisor.
  pub fn modulo(dividend_value: &Value, divisor_value: &Value) -> Value {
    if let Value::Number(dividend) = *dividend_value {
      if let Value::Number(divisor) = *divisor_value {
        if divisor.abs() < f64::EPSILON {
          trace_invalid_value!();
          VALUE_NULL
        } else {
          Value::Number(dividend - divisor * (dividend / divisor).floor())
        }
      } else {
        trace_invalid_value_type!("modulo");
        VALUE_NULL
      }
    } else {
      trace_invalid_value_type!("modulo");
      VALUE_NULL
    }
  }

  /// Logical negation.
  pub fn not(negand: &Value) -> Value {
    if let Value::Boolean(v) = negand {
      Value::Boolean(!(*v))
    } else {
      trace_invalid_value_type!("not");
      VALUE_NULL
    }
  }

  /// Converts string to a number.
  /// Grouping...
  pub fn number(from: &Value, grouping_separator: &Value, decimal_separator: &Value) -> Value {
    // function for converting string to Value::Number
    let convert = |value: String| {
      if let Ok(number) = value.parse::<f64>() {
        Value::Number(number)
      } else {
        trace_invalid_value_type!("number");
        VALUE_NULL
      }
    };
    match from {
      Value::String(value) => {
        // prepare grouping separator from Value::String ot VALUE_NULL
        let grouping_sep = match grouping_separator {
          Value::String(s) => match s.as_str() {
            " " | "." | "," => Some((*s).clone()),
            _ => {
              trace_invalid_value!();
              return VALUE_NULL;
            }
          },
          Value::Null(_) => None,
          _ => {
            trace_invalid_value_type!("number");
            return VALUE_NULL;
          }
        };
        // prepare decimal separator from Value::String ot VALUE_NULL
        let decimal_sep = match decimal_separator {
          Value::String(s) => match s.as_str() {
            "." | "," => Some((*s).clone()),
            _ => {
              trace_invalid_value!();
              return VALUE_NULL;
            }
          },
          Value::Null(_) => None,
          _ => {
            trace_invalid_value_type!("number");
            return VALUE_NULL;
          }
        };
        // replace both separators and try to convert
        if let Some(grp_sep) = &grouping_sep {
          if let Some(dec_sep) = &decimal_sep {
            return if *grp_sep != *dec_sep {
              convert(value.replace(grp_sep, "").replace(dec_sep, "."))
            } else {
              trace_invalid_value!();
              VALUE_NULL
            };
          }
        }
        // replace grouping separator and try to convert
        if decimal_sep.is_none() {
          if let Some(sep) = grouping_sep {
            return convert(value.replace(&sep, ""));
          }
        }
        // replace decimal separator and try to convert
        if grouping_sep.is_none() {
          if let Some(sep) = decimal_sep {
            return convert(value.replace(&sep, "."));
          }
        }
        // try to convert an input parameter without replacing
        convert(value.clone())
      }
      _ => {
        trace_invalid_value_type!("number");
        VALUE_NULL
      }
    }
  }

  /// Returns **true** if number is odd, **false** if it is even.
  pub fn odd(value: &Value) -> Value {
    if let Value::Number(v) = value {
      Value::Boolean((v.trunc() as i64) % 2 != 0)
    } else {
      trace_invalid_value_type!("odd");
      VALUE_NULL
    }
  }

  pub fn remove(list: &Value, position: &Value) -> Value {
    if let Value::List(mut items) = list.clone() {
      if let Value::Number(pos) = position {
        let index = pos.trunc() as i64;
        if index > 0 {
          let i = (index as usize) - 1;
          if i < items.to_vec().len() {
            items.remove(i);
            return Value::List(items);
          }
        }
        if index < 0 {
          let i = index.abs() as usize;
          if i <= items.len() {
            items.remove(items.len() - i);
            return Value::List(items);
          }
        }
      }
    }
    VALUE_NULL_WITH_TRACE /*probably index is out of range*/
  }

  pub fn replace(input_string_value: &Value, pattern_string_value: &Value, replacement_string_value: &Value, flags_string_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      if let Value::String(pattern_string) = pattern_string_value {
        if let Value::String(replacement_string) = replacement_string_value {
          // Rust implementation is eager when parsing matching groups, so place numbers in square brackets
          let repl = if let Ok(rg) = Regex::new("\\$([1-9][0-9]*)") {
            rg.replace_all(replacement_string.as_str(), "$${${1}}").to_string()
          } else {
            replacement_string.clone()
          };
          // check and use flags
          if let Value::String(flags_string) = flags_string_value {
            let mut flags = "".to_string();
            let mut flag_q = false;
            let mut clear_flag_q = false;
            for ch in flags_string.chars() {
              if ch == 'q' {
                flag_q = true;
              }
              if matches!(ch, 's' | 'm' | 'i' | 'x') {
                flags.push(ch);
                if ch != 'i' {
                  clear_flag_q = true;
                }
              }
            }
            if clear_flag_q {
              flag_q = false;
            }
            let mut patt = "".to_string();
            for ch in pattern_string.chars() {
              if flag_q {
                patt.push('\\');
              }
              patt.push(ch);
            }
            if flags.is_empty() {
              if let Ok(re) = Regex::new(&patt) {
                let result = re.replace_all(input_string.as_str(), repl.as_str()).to_string();
                return Value::String(result);
              }
            } else if let Ok(re) = Regex::new(format!("(?{}){}", flags, patt).as_str()) {
              let result = re.replace_all(input_string.as_str(), repl.as_str()).to_string();
              return Value::String(result);
            }
          }
          // replace without any flags
          if let Ok(re) = Regex::new(&pattern_string) {
            let result = re.replace_all(input_string.as_str(), repl.as_str()).to_string();
            return Value::String(result);
          }
        }
      }
    }
    trace_invalid_value_type!("replace");
    VALUE_NULL
  }

  pub fn reverse(list: &Value) -> Value {
    if let Value::List(mut items) = list.clone() {
      items.reverse();
      return Value::List(items);
    }
    VALUE_NULL_WITH_TRACE
  }

  ///
  pub fn split(input_string_value: &Value, delimiter_string_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      if let Value::String(delimiter_string) = delimiter_string_value {
        if let Ok(re) = Regex::new(&delimiter_string) {
          return Value::List(Values::new(re.split(input_string).map(|s| Value::String(s.to_string())).collect()));
        }
      }
    }
    trace_invalid_value_type!("split");
    VALUE_NULL
  }

  /// Returns the square root of the given **value** specified as a parameter.
  /// If the given number is negative it returns [VALUE_NULL].
  pub fn sqrt(value: &Value) -> Value {
    if let Value::Number(v) = value {
      if *v >= 0.0 {
        Value::Number(v.sqrt())
      } else {
        VALUE_NULL
      }
    } else {
      trace_invalid_value_type!("sqrt");
      VALUE_NULL
    }
  }

  /// Returns **true** when the input string starts with specified match string.
  pub fn starts_with(input_string_value: &Value, match_string_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      if let Value::String(match_string) = match_string_value {
        Value::Boolean(input_string.starts_with(match_string))
      } else {
        trace_invalid_value_type!("starts_with");
        VALUE_NULL
      }
    } else {
      trace_invalid_value_type!("starts_with");
      VALUE_NULL
    }
  }

  ///
  pub fn stddev(values: &[Value]) -> Value {
    if values.len() < 2 {
      return VALUE_NULL;
    }
    let mut sum = 0.0;
    let mut numbers = vec![];
    for value in values {
      if let Value::Number(x) = *value {
        sum += x;
        numbers.push(x);
      } else {
        trace_invalid_value_type!("stddev");
        return VALUE_NULL;
      }
    }
    let n = numbers.len() as f64;
    let avg = sum / n;
    let mut sum2 = 0.0;
    for number in numbers {
      sum2 += (number - avg).powi(2);
    }
    let stddev = (sum2 / (n - 1.0)).sqrt();
    Value::Number(stddev)
  }

  /// Converts specified value to [Value::String].
  pub fn string(value: &Value) -> Value {
    if let Value::String(s) = value {
      Value::String(s.clone())
    } else {
      Value::String(value.stringify())
    }
  }

  /// Returns the number of characters in string.
  pub fn string_length(input_string_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      Value::Number(input_string.chars().count() as f64)
    } else {
      trace_invalid_value_type!("string_length");
      VALUE_NULL
    }
  }

  /// Returns the sum of values in the collection of numbers.
  pub fn sum(values: &[Value]) -> Value {
    if values.is_empty() {
      return VALUE_NULL;
    }
    if let Value::Number(n) = values[0] {
      let mut sum = n;
      for value in values.iter().skip(1) {
        if let Value::Number(v) = *value {
          sum += v;
        } else {
          trace_invalid_value_type!("sum");
          return VALUE_NULL;
        }
      }
      return Value::Number(sum);
    }

    trace_invalid_value_type!("sum");
    VALUE_NULL
  }

  pub fn sublist2(list: &Value, position: &Value) -> Value {
    if let Value::List(items) = list {
      if let Value::Number(position) = position {
        let position = position.trunc() as i64;
        if position > 0 {
          let index = (position as usize) - 1;
          if index < items.len() {
            return Value::List(Values::new(items.to_vec()[index..].to_vec()));
          }
        }
        if position < 0 {
          let index = position.abs() as usize;
          if index <= items.len() {
            return Value::List(Values::new(items.to_vec()[items.len() - index..].to_vec()));
          }
        }
      }
    }
    VALUE_NULL_WITH_TRACE /*probably index is out of range*/
  }

  pub fn sublist3(list: &Value, position: &Value, length: &Value) -> Value {
    if let Value::List(items) = list {
      if let Value::Number(length) = length {
        let length = length.trunc() as i64;
        if length > 0 {
          if let Value::Number(position) = position {
            let position = position.trunc() as i64;
            if position > 0 {
              let first = (position as usize) - 1;
              let last = first + (length as usize);
              if first < items.len() && last <= items.len() {
                return Value::List(Values::new(items.to_vec()[first..last].to_vec()));
              }
            }
            if position < 0 {
              let first = items.len() - (position.abs() as usize);
              let last = first + (length as usize);
              if first < items.len() && last <= items.len() {
                return Value::List(Values::new(items.to_vec()[first..last].to_vec()));
              }
            }
          }
        }
      }
    }
    VALUE_NULL_WITH_TRACE /*probably index is out of range*/
  }

  /// Returns `length` (or all) characters from string, starting at
  /// `start_position`. First position is 1, last position is -1.
  pub fn substring(input_string_value: &Value, start_position_value: &Value, length_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      if let Value::Number(start_position) = start_position_value {
        let start = start_position.trunc() as isize;
        let input_string_len = input_string.chars().count();
        match length_value {
          Value::Number(length) => {
            if *length < 1.0 {
              trace_invalid_value!();
              return VALUE_NULL;
            }
            let count = length.trunc() as usize;
            if start > 0 {
              let index = (start - 1) as usize;
              if index < input_string_len && index + count <= input_string_len {
                return Value::String(input_string.chars().skip(index).take(count).collect());
              }
            }
            if start < 0 {
              let index = (input_string_len as isize) + start;
              if index >= 0 && index as usize + count <= input_string_len {
                return Value::String(input_string.chars().skip(index as usize).take(count).collect());
              }
            }
            trace_invalid_value!();
            VALUE_NULL
          }
          Value::Null(_) => {
            if start > 0 {
              let index = (start - 1) as usize;
              if index < input_string_len {
                return Value::String(input_string.chars().skip(index).collect());
              }
            }
            if start < 0 {
              let index = (input_string_len as isize) + start;
              if index >= 0 {
                return Value::String(input_string.chars().skip(index as usize).collect());
              }
            }
            trace_invalid_value!();
            VALUE_NULL
          }
          _ => {
            trace_invalid_value_type!("substring");
            VALUE_NULL
          }
        }
      } else {
        trace_invalid_value_type!("substring");
        VALUE_NULL
      }
    } else {
      trace_invalid_value_type!("substring");
      VALUE_NULL
    }
  }

  /// Returns substring of `input_string_value`  after the `match_input_string` in string.
  pub fn substring_after(input_string_value: &Value, match_input_string: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      if let Value::String(match_string) = match_input_string {
        return if let Some(index) = input_string.find(match_string) {
          Value::String(input_string[match_string.len() + index..].to_string())
        } else {
          Value::String("".to_string())
        };
      }
    }
    trace_invalid_value_type!("substring_after");
    VALUE_NULL
  }

  /// Returns substring of `input_string_value`  before the `match_input_string` in string.
  pub fn substring_before(input_string_value: &Value, match_input_string: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      if let Value::String(match_string) = match_input_string {
        return if let Some(index) = input_string.find(match_string) {
          Value::String(input_string[..index].to_string())
        } else {
          Value::String("".to_string())
        };
      }
    }
    trace_invalid_value_type!("substring_before");
    VALUE_NULL_WITH_TRACE
  }

  pub fn time_1(value: &Value) -> Value {
    match value {
      Value::String(text) => {
        if let Ok(time) = text.parse::<FeelTime>() {
          return Value::Time(time);
        }
      }
      Value::Date(_) => return Value::Time(FeelTime::utc(0, 0, 0, 0)),
      Value::DateTime(date_time) => return Value::Time(date_time.time()),
      Value::Time(time) => return Value::Time(time.clone()),
      _ => {}
    }
    VALUE_NULL_WITH_TRACE
  }

  pub fn time_3(hour_value: &Value, minute_value: &Value, second_value: &Value) -> Value {
    if let Value::Number(hour) = hour_value {
      if let Value::Number(minute) = minute_value {
        if let Value::Number(second) = second_value {
          if let Some(feel_time) = FeelTime::new_hms_opt(*hour, *minute, *second) {
            return Value::Time(feel_time);
          }
        }
      }
    }
    VALUE_NULL_WITH_TRACE
  }

  pub fn time_4(hour_value: &Value, minute_value: &Value, second_value: &Value, duration_value: &Value) -> Value {
    if let Value::Number(hour) = hour_value {
      if let Value::Number(minute) = minute_value {
        if let Value::Number(second) = second_value {
          match duration_value {
            Value::DaysAndTimeDuration(duration) => {
              if let Some(feel_time) = FeelTime::new_hmso_opt(*hour, *minute, *second, duration.as_seconds()) {
                return Value::Time(feel_time);
              }
            }
            Value::Null(_) => {
              if let Some(feel_time) = FeelTime::new_hms_opt(*hour, *minute, *second) {
                return Value::Time(feel_time);
              }
            }
            _ => {}
          }
        }
      }
    }
    VALUE_NULL_WITH_TRACE
  }

  pub fn union(lists: &[Value]) -> Value {
    let mut result = vec![];
    for list in lists {
      if let Value::List(items) = list {
        for item in items.to_vec() {
          if result.iter().all(|a| !evaluate_equals(a, item)) {
            result.push(item.clone())
          }
        }
      } else {
        return VALUE_NULL_WITH_TRACE;
      }
    }
    Value::List(Values::new(result))
  }

  /// Returns upper-cased string.
  pub fn upper_case(input_string_value: &Value) -> Value {
    if let Value::String(input_string) = input_string_value {
      Value::String(input_string.to_uppercase())
    } else {
      trace_invalid_value_type!("upper_case");
      VALUE_NULL
    }
  }

  pub fn years_and_months_duration(from_value: &Value, to_value: &Value) -> Value {
    if let Value::Date(from) = from_value {
      if let Value::DateTime(to) = to_value {
        return YearsAndMonthsDuration(to.ym_duration_1(from));
      }
      if let Value::Date(to) = to_value {
        return YearsAndMonthsDuration(to.ym_duration(from));
      }
    }
    if let Value::DateTime(from) = from_value {
      if let Value::DateTime(to) = to_value {
        return YearsAndMonthsDuration(to.ym_duration(from));
      }
      if let Value::Date(to) = to_value {
        return YearsAndMonthsDuration(to.ym_duration(&from.date()));
      }
    }
    VALUE_NULL_WITH_TRACE
  }
}

#[cfg(test)]
mod tests {
  use crate::feel::eval_bif::core::substring;
  use dmntk_recognizer::dmntk_model::dmntk_feel_parser::dmntk_feel::values::{Value, VALUE_NULL};

  #[test]
  fn test_substring() {
    // starting position may be not zero
    eq_substring_null("homeless", 0.0);
    // positive starting position
    eq_substring("homeless", "homeless", 1.0);
    eq_substring("less", "homeless", 5.0);
    eq_substring("ss", "homeless", 7.0);
    eq_substring("s", "homeless", 8.0);
    eq_substring("😀", "foo\u{1F40E}bar\u{1F600}", 8.0);
    eq_substring_null("homeless", 9.0);
    // negative starting position
    eq_substring("s", "homeless", -1.0);
    eq_substring("less", "homeless", -4.0);
    eq_substring("homeless", "homeless", -8.0);
    eq_substring_null("homeless", -9.0);
    // positive starting position with length
    eq_substring_len("homeless", "homeless", 1.0, 8.0);
    eq_substring_len("home", "homeless", 1.0, 4.0);
    eq_substring_len("less", "homeless", 5.0, 4.0);
    eq_substring_len("el", "homeless", 4.0, 2.0);
    eq_substring_len("ss", "homeless", 7.0, 2.0);
    eq_substring_len("s", "homeless", 7.0, 1.0);
    eq_substring_len("s", "homeless", 8.0, 1.0);
    eq_substring_len_null("homeless", 0.0, 4.0);
    eq_substring_len_null("homeless", 1.0, 0.0);
    eq_substring_len_null("homeless", 1.0, 9.0);
    // negative starting position with length
    eq_substring_len("homeless", "homeless", -8.0, 8.0);
    eq_substring_len("home", "homeless", -8.0, 4.0);
    eq_substring_len("less", "homeless", -4.0, 4.0);
    eq_substring_len("el", "homeless", -5.0, 2.0);
    eq_substring_len("ss", "homeless", -2.0, 2.0);
    eq_substring_len("s", "homeless", -2.0, 1.0);
    eq_substring_len("s", "homeless", -1.0, 1.0);
    eq_substring_len("😀", "foo\u{1F40E}bar\u{1F600}", -1.0, 1.0);
    eq_substring_len_null("homeless", -1.0, 0.0);
    eq_substring_len_null("homeless", -3.0, 4.0);
    eq_substring_len_null("homeless", -9.0, 2.0);
  }

  fn eq_substring(expected: &str, input_string: &str, start_position: f64) {
    assert_eq!(
      Value::String(expected.to_string()),
      substring(&Value::String(input_string.to_string()), &Value::Number(start_position), &VALUE_NULL)
    );
  }

  fn eq_substring_null(input_string: &str, start_position: f64) {
    assert_eq!(
      VALUE_NULL,
      substring(&Value::String(input_string.to_string()), &Value::Number(start_position), &VALUE_NULL)
    );
  }

  fn eq_substring_len(expected: &str, input_string: &str, start_position: f64, length: f64) {
    assert_eq!(
      Value::String(expected.to_string()),
      substring(&Value::String(input_string.to_string()), &Value::Number(start_position), &Value::Number(length))
    );
  }

  fn eq_substring_len_null(input_string: &str, start_position: f64, length: f64) {
    assert_eq!(
      VALUE_NULL,
      substring(&Value::String(input_string.to_string()), &Value::Number(start_position), &Value::Number(length))
    );
  }
}
