/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL and DMN model evaluator.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

use crate::model::errors::*;
use dmntk_feel_parser::dmntk_feel::context::FeelContext;
use dmntk_feel_parser::dmntk_feel::dmntk_common::Result;
use dmntk_feel_parser::dmntk_feel::values::{Value, Values, VALUE_NULL_WITH_TRACE};
use dmntk_feel_parser::dmntk_feel::{AstNode, Name, OptAstNode, Scope};
use dmntk_model::model::{BuiltinAggregator, DecisionTable, HitPolicy};

/// Parsed rule of decision table.
struct ParsedRule {
  input_entry_nodes: Vec<AstNode>,
  output_entry_nodes: Vec<AstNode>,
}

/// Parsed decision table.
/// All expressions contained in different parts of the decision
/// table are parsed into AST and stored in this structure.
struct ParsedDecisionTable {
  component_names: Vec<Name>,
  output_values_nodes: Vec<OptAstNode>,
  default_output_values_nodes: Vec<OptAstNode>,
  parsed_rules: Vec<ParsedRule>,
}

/// Evaluated rule of decision table.
struct EvaluatedRule {
  matches: bool,
  output_entry_values: Vec<Value>,
}

/// Evaluated decision table.
/// All ASTs from [ParsedDecisionTable] are evaluated in specified context
/// and stored as [Values](Value) in this structure.
struct EvaluatedDecisionTable {
  component_names: Vec<Name>,
  output_values: Vec<Value>,
  default_output_values: Vec<Value>,
  evaluated_rules: Vec<EvaluatedRule>,
}

impl EvaluatedDecisionTable {
  /// Returns all matching rules in rule order.
  fn get_matching_rules(&self) -> Vec<&EvaluatedRule> {
    self.evaluated_rules.iter().filter(|evaluated_rule| evaluated_rule.matches).collect()
  }
  /// Returns all matching rules in decreasing order of priority.
  fn get_matching_rules_prioritized(&self) -> Vec<&EvaluatedRule> {
    let mut a: Vec<&EvaluatedRule> = self.evaluated_rules.iter().filter(|v| v.matches).collect();
    let compare = |x: &&EvaluatedRule, y: &&EvaluatedRule| {
      for (v1, v2) in x.output_entry_values.iter().zip(y.output_entry_values.iter()) {
        let index1 = self.output_values.iter().position(|o| o == v1);
        let index2 = self.output_values.iter().position(|o| o == v2);
        match (index1, index2) {
          (Some(ix1), Some(ix2)) => {
            if ix1 < ix2 {
              return std::cmp::Ordering::Less;
            }
            if ix1 > ix2 {
              return std::cmp::Ordering::Greater;
            }
          }
          (Some(_), None) => return std::cmp::Ordering::Less,
          (None, Some(_)) => return std::cmp::Ordering::Greater,
          _ => {}
        }
      }
      std::cmp::Ordering::Equal
    };
    a.sort_unstable_by(compare);
    a
  }
  /// Returns a result composed from values taken from evaluated output entries.
  fn get_result(&self, evaluated_rule: &EvaluatedRule) -> Result<Value> {
    if evaluated_rule.output_entry_values.len() > 1 {
      if evaluated_rule.output_entry_values.len() != self.component_names.len() {
        return Err(number_of_output_values_differ_from_component_names());
      }
      let mut result: FeelContext = Default::default();
      for (i, value) in evaluated_rule.output_entry_values.iter().enumerate() {
        result.set_entry(&self.component_names[i], value.clone());
      }
      Ok(Value::Context(result))
    } else {
      Ok(evaluated_rule.output_entry_values[0].clone())
    }
  }
  /// Returns a result composed from values taken from evaluated output entries.
  fn get_results(&self, evaluated_rules: &[&EvaluatedRule]) -> Result<Value> {
    let mut values = vec![];
    for evaluated_rule in evaluated_rules {
      values.push(self.get_result(evaluated_rule)?);
    }
    Ok(Value::List(Values::new(values)))
  }
  ///
  fn evaluate_default_output_value(&self) -> Value {
    match self.default_output_values.len() {
      0 => VALUE_NULL_WITH_TRACE, /* no rules matched, no output value defined */
      1 => self.default_output_values[0].clone(),
      _ => VALUE_NULL_WITH_TRACE,
    }
  }
  ///
  fn evaluate_hit_policy_unique(&self) -> Result<Value> {
    let matching_rules = self.get_matching_rules();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    if matching_rules.len() > 1 {
      return Err(multiple_rules_match_in_unique_hit_policy());
    }
    self.get_result(matching_rules[0])
  }
  ///
  fn evaluate_hit_policy_any(&self) -> Result<Value> {
    let matching_rules = self.get_matching_rules();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    let first_result = self.get_result(matching_rules[0])?;
    for evaluated_rule in matching_rules {
      let result = self.get_result(evaluated_rule)?;
      if result != first_result {
        return Err(all_matching_rules_must_have_the_same_value());
      }
    }
    Ok(first_result)
  }
  ///
  fn evaluate_hit_policy_priority(&self) -> Result<Value> {
    let matching_rules = self.get_matching_rules_prioritized();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    self.get_result(matching_rules[0])
  }
  ///
  fn evaluate_hit_policy_first(&self) -> Result<Value> {
    let matching_rules = self.get_matching_rules();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    self.get_result(matching_rules[0])
  }
  ///
  fn evaluate_hit_policy_rule_order(&self) -> Result<Value> {
    let matching_rules = self.get_matching_rules();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    self.get_results(&matching_rules)
  }
  ///
  fn evaluate_hit_policy_output_order(&self) -> Result<Value> {
    let matching_rules = self.get_matching_rules_prioritized();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    self.get_results(&matching_rules)
  }
  ///
  fn evaluate_hit_policy_collect_list(&self) -> Result<Value> {
    let matching_rules = self.get_matching_rules();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    self.get_results(&matching_rules)
  }
  ///
  fn evaluate_hit_policy_collect_count(&self) -> Value {
    let matching_rules = self.get_matching_rules();
    if matching_rules.is_empty() {
      return self.evaluate_default_output_value();
    }
    Value::Number(matching_rules.len() as f64)
  }
  ///
  fn evaluate_hit_policy_collect_sum(&self) -> Result<Value> {
    if self.component_names.len() > 1 {
      return Err(aggregators_not_allowed_for_compound_outputs());
    }
    let matching_rules = self.get_matching_rules();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    let output_values = matching_rules
      .iter()
      .map(|evaluated_rule| evaluated_rule.output_entry_values[0].clone())
      .collect::<Vec<Value>>();
    Ok(crate::evaluate_sum(output_values))
  }
  ///
  fn evaluate_hit_policy_collect_min(&self) -> Result<Value> {
    if self.component_names.len() > 1 {
      return Err(aggregators_not_allowed_for_compound_outputs());
    }
    let matching_rules = self.get_matching_rules();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    let output_values = matching_rules
      .iter()
      .map(|evaluated_rule| evaluated_rule.output_entry_values[0].clone())
      .collect::<Vec<Value>>();
    Ok(crate::evaluate_min(output_values))
  }
  ///
  fn evaluate_hit_policy_collect_max(&self) -> Result<Value> {
    if self.component_names.len() > 1 {
      return Err(aggregators_not_allowed_for_compound_outputs());
    }
    let matching_rules = self.get_matching_rules();
    if matching_rules.is_empty() {
      return Ok(self.evaluate_default_output_value());
    }
    let output_values = matching_rules
      .iter()
      .map(|evaluated_rule| evaluated_rule.output_entry_values[0].clone())
      .collect::<Vec<Value>>();
    Ok(crate::evaluate_max(output_values))
  }
}

fn parse_dt(scope: &Scope, dt: DecisionTable) -> Result<ParsedDecisionTable> {
  // parse input expressions and input values
  let mut input_expressions_and_values = vec![];
  for input_clause in &dt.input_clauses {
    let input_expression = dmntk_feel_parser::parse_textual_expression(scope, &input_clause.input_expression, false)?;
    let input_values = if let Some(input) = &input_clause.input_values {
      Some(dmntk_feel_parser::parse_unary_tests(scope, input, false)?)
    } else {
      None
    };
    input_expressions_and_values.push((input_expression, input_values))
  }
  // parse output values and output component names
  let mut component_names = vec![];
  let mut output_values_nodes = vec![];
  let mut default_output_values_nodes = vec![];
  for output_clause in &dt.output_clauses {
    if let Some(text) = &output_clause.output_values {
      output_values_nodes.push(Some(dmntk_feel_parser::parse_unary_tests(scope, text, false)?));
    } else {
      output_values_nodes.push(None);
    }
    if let Some(text) = &output_clause.default_output_entry {
      default_output_values_nodes.push(Some(dmntk_feel_parser::parse_unary_tests(scope, text, false)?));
    } else {
      default_output_values_nodes.push(None);
    }
    if let Some(name) = &output_clause.name {
      component_names.push(dmntk_feel_parser::parse_name(scope, name, false)?);
    }
  }
  // parse all rules
  let mut parsed_rules = vec![];
  for rule in &dt.rules {
    // parse input clause
    let mut input_nodes = vec![];
    for (i, (input_expression, input_values)) in input_expressions_and_values.iter().enumerate() {
      let input_entry_node = dmntk_feel_parser::parse_unary_tests(&scope, &rule.input_entries[i].text, false)?;
      if let Some(input_values_node) = input_values {
        let left = AstNode::In(Box::new(input_expression.clone()), Box::new(input_values_node.clone()));
        let right = AstNode::In(Box::new(input_expression.clone()), Box::new(input_entry_node));
        input_nodes.push(AstNode::And(Box::new(left), Box::new(right)));
      } else {
        input_nodes.push(AstNode::In(Box::new(input_expression.clone()), Box::new(input_entry_node)))
      }
    }
    // parse output clause
    let mut output_nodes = vec![];
    for (i, output_values) in output_values_nodes.iter().enumerate() {
      let output_entry_node = dmntk_feel_parser::parse_textual_expression(scope, &rule.output_entries[i].text, false)?;
      if let Some(output_value_node) = output_values {
        output_nodes.push(AstNode::Out(Box::new(output_entry_node), Box::new(output_value_node.clone())))
      } else {
        output_nodes.push(output_entry_node)
      }
    }
    parsed_rules.push(ParsedRule {
      input_entry_nodes: input_nodes,
      output_entry_nodes: output_nodes,
    })
  }
  // return parsed decision table
  Ok(ParsedDecisionTable {
    component_names,
    output_values_nodes,
    default_output_values_nodes,
    parsed_rules,
  })
}

fn evaluate_pdt(scope: &Scope, pdt: ParsedDecisionTable) -> Result<EvaluatedDecisionTable> {
  // evaluate only non-empty output values
  let mut output_values = vec![];
  for output_values_node in pdt.output_values_nodes.iter().flatten() {
    if let Value::CommaList(values) = crate::evaluate(scope, output_values_node)? {
      output_values.append(&mut values.to_vec().to_owned());
    }
  }
  // evaluate only non-empty default output values
  let mut default_output_values = vec![];
  for default_output_values_node in pdt.default_output_values_nodes.iter().flatten() {
    if let Value::CommaList(values) = crate::evaluate(scope, default_output_values_node)? {
      default_output_values.append(&mut values.to_vec().to_owned());
    }
  }
  // evaluate all rules
  let mut evaluated_rules = vec![];
  for parsed_rule in &pdt.parsed_rules {
    let mut input_entry_values = vec![];
    let mut matches = true;
    for input_node in &parsed_rule.input_entry_nodes {
      let input_value = crate::evaluate(scope, input_node)?;
      if !input_value.is_true() {
        matches = false;
      }
      input_entry_values.push(input_value);
    }
    let mut output_entry_values = vec![];
    for output_entry_node in &parsed_rule.output_entry_nodes {
      output_entry_values.push(crate::evaluate(scope, output_entry_node)?);
    }
    evaluated_rules.push(EvaluatedRule { matches, output_entry_values })
  }
  Ok(EvaluatedDecisionTable {
    component_names: pdt.component_names,
    output_values,
    default_output_values,
    evaluated_rules,
  })
}

/// Evaluates the decision table.
pub fn evaluate_decision_table(scope: &Scope, decision_table: DecisionTable) -> Result<Value> {
  // save hit policy for further use
  let hit_policy = decision_table.hit_policy;
  // parse decision table (rules, inputs and outputs) from textual `FEEL` form to AST nodes
  let parsed_decision_table = parse_dt(scope, decision_table)?;
  // evaluate values for each rule
  let evaluated_decision_table = evaluate_pdt(scope, parsed_decision_table)?;
  // evaluate the result depending on hit policy
  match hit_policy {
    HitPolicy::Unique => evaluated_decision_table.evaluate_hit_policy_unique(),
    HitPolicy::Any => evaluated_decision_table.evaluate_hit_policy_any(),
    HitPolicy::Priority => evaluated_decision_table.evaluate_hit_policy_priority(),
    HitPolicy::First => evaluated_decision_table.evaluate_hit_policy_first(),
    HitPolicy::RuleOrder => evaluated_decision_table.evaluate_hit_policy_rule_order(),
    HitPolicy::OutputOrder => evaluated_decision_table.evaluate_hit_policy_output_order(),
    HitPolicy::Collect(aggregator) => match aggregator {
      BuiltinAggregator::List => evaluated_decision_table.evaluate_hit_policy_collect_list(),
      BuiltinAggregator::Count => Ok(evaluated_decision_table.evaluate_hit_policy_collect_count()),
      BuiltinAggregator::Sum => evaluated_decision_table.evaluate_hit_policy_collect_sum(),
      BuiltinAggregator::Min => evaluated_decision_table.evaluate_hit_policy_collect_min(),
      BuiltinAggregator::Max => evaluated_decision_table.evaluate_hit_policy_collect_max(),
    },
  }
}
