/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * FEEL and DMN model evaluator.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! FEEL evaluation engine.

/// Traces invalid type of the value encountered during evaluation.
/// Usually another type is expected in evaluated expression.
#[macro_export]
macro_rules! not_implemented {
  ($l:expr) => {{
    println!("{:?}", $l);
    Value::Null(Some(format!("not implemented: {:?}", $l)))
  }};
}

/// Traces invalid type of the value encountered during evaluation.
/// Usually another type is expected in evaluated expression.
#[macro_export]
macro_rules! trace_invalid_value_type {
  ($l:expr) => {{
    println!("invalid value type: {:?}", $l);
  }};
}

/// Traces invalid value encountered during evaluation.
/// Usually the value can not be converted from one type to another
/// or its value is out of expected range.
#[macro_export]
macro_rules! trace_invalid_value {
  () => {{
    println!("invalid value");
  }};
}

/// Traces invalid number of parameters, usually in functions.
/// Usually, the evaluated function expects another number of input parameters.
#[macro_export]
macro_rules! trace_invalid_number_of_parameters {
  () => {{
    println!("invalid number of parameters");
  }};
}

#[macro_export]
macro_rules! trace_invalid_parameter_type {
  () => {{
    println!("invalid parameter type");
  }};
}

#[macro_export]
macro_rules! trace_invalid_parameter_value {
  () => {{
    println!("invalid parameter value");
  }};
}

#[macro_export]
macro_rules! trace_named_parameter_not_found {
  () => {{
    println!("named parameter not found");
  }};
}

pub mod engine {
  use crate::feel::errors::*;
  use crate::feel::eval_bif;
  use dmntk_feel_parser::dmntk_feel::bif::Bif;
  use dmntk_feel_parser::dmntk_feel::context::FeelContext;
  use dmntk_feel_parser::dmntk_feel::dmntk_common::{null_with_trace, Result, Stringify};
  use dmntk_feel_parser::dmntk_feel::values::{FunctionBody, Value, Values, VALUE_FALSE, VALUE_NULL, VALUE_NULL_WITH_TRACE, VALUE_TRUE};
  use dmntk_feel_parser::dmntk_feel::{
    is_integer, round_half_to_even, subtract, AstNode, FeelDate, FeelDateTime, FeelDaysAndTimeDuration, FeelTime, FeelType, FeelYearsAndMonthsDuration, Name,
    OptAstNode, Scope,
  };
  use std::borrow::Borrow;
  use std::collections::{BTreeMap, HashSet};
  use std::convert::TryFrom;
  use std::ops::Deref;
  use std::str::FromStr;

  /// Evaluates the AST for `FEEL` expression.
  pub fn eval(scope: &Scope, node: &AstNode) -> Result<Value> {
    match node {
      AstNode::Add(lhs, rhs) => eval_addition(scope, lhs, rhs),
      AstNode::And (lhs, rhs) => eval_conjunction(scope, lhs, rhs),
      AstNode::At(rhs) => eval_at_literal(scope, rhs),
      AstNode::Between (lhs, mhs, rhs) => eval_between(scope, lhs, mhs, rhs),
      AstNode::Boolean(lhs) => Ok(Value::Boolean(*lhs)),
      AstNode::Context ( items ) => eval_context(scope, items),
      AstNode::ContextEntry ( key, entry ) => eval_context_entry(scope, key, entry),
      AstNode::ContextEntryKey(name) => Ok(eval_context_entry_key(name)),
      AstNode::ContextType(items) => eval_context_type(scope, items),
      AstNode::Div(lhs, rhs) => eval_division(scope, lhs, rhs),
      AstNode::Eq (lhs, rhs) => eval_equal(scope, lhs, rhs),
      AstNode::Exp (lhs, rhs) => eval_exponentiation(scope, lhs, rhs),
      AstNode::ExpressionList ( items ) => eval_expression_list(scope, items),
      AstNode::FilterExpression (lhs, rhs) => eval_filter_expression(scope, lhs, rhs),
      AstNode::ForExpression (lhs, rhs) => eval_for_expression(scope, lhs, rhs),
      AstNode::FunctionBody ( lhs, external ) => eval_function_body(scope, lhs, *external),
      AstNode::FunctionDefinition(lhs, rhs) => eval_function_definition(scope, lhs, rhs),
      AstNode::FunctionInvocation (lhs, rhs) => eval_function_invocation(scope, lhs, rhs),
      AstNode::Ge (lhs, rhs) => eval_greater_or_equal(scope, lhs, rhs),
      AstNode::Gt (lhs, rhs) => eval_greater(scope, lhs, rhs),
      AstNode::If(condition, when_true, when_false) => eval_if_expression(scope, condition, when_true, when_false),
      AstNode::In ( left, right ) => eval_in(scope, left, right),
      AstNode::InstanceOf (lhs, rhs) => eval_instance_of(scope, lhs, rhs),
      AstNode::IntervalEnd(lhs, closed) => eval_interval_end(scope, lhs, closed),
      AstNode::IntervalStart(lhs, closed) => eval_interval_start(scope, lhs, closed),
      AstNode::Irrelevant => Ok(Value::Irrelevant),
      AstNode::Le (lhs, rhs) => eval_less_or_equal(scope, lhs, rhs),
      AstNode::List(items) => eval_list(scope, items),
      AstNode::ListType(feel_type) => eval_list_type(scope, feel_type),
      AstNode::Lt (lhs, rhs) => eval_less(scope, lhs, rhs),
      AstNode::Mul (lhs, rhs) => eval_multiplication(scope, lhs, rhs),
      AstNode::Name(name) => eval_name(scope, name),
      AstNode::NamedParameter (lhs, rhs) => eval_named_parameter(scope, lhs, rhs),
      AstNode::NamedParameters ( items ) => Ok(eval_named_parameters(scope, items)),
      AstNode::Neg (rhs) => eval_arithmetic_negation(scope, rhs),
      AstNode::Null => Ok(VALUE_NULL),
      AstNode::Numeric(lhs, rhs) => eval_numeric(lhs, rhs),
      AstNode::Nq(lhs, rhs) => eval_not_equal(scope, lhs, rhs),
      AstNode::Or (lhs, rhs) => eval_disjunction(scope, lhs, rhs),
      AstNode::Out (lhs, rhs) => eval_out(scope, lhs, rhs),
      AstNode::ParameterName(name) => Ok(eval_parameter_name(name)),
      AstNode::Path(lhs, rhs) => eval_path(scope, lhs, rhs),
      AstNode::PositionalParameters ( items ) => eval_positional_parameters(scope, items),
      AstNode::QualifiedName(value) => eval_qualified_name(scope, value),
      AstNode::Range ( left,  right ) => eval_interval(scope, left, right),
      AstNode::RangeType(feel_type) => eval_range_type(scope, feel_type),
      AstNode::String(value) => Ok(Value::String(String::from(value))),
      AstNode::Sub (lhs, rhs) => eval_subtraction(scope, lhs, rhs),
      AstNode::UnaryGt (rhs) => eval_unary_greater(scope, rhs),
      AstNode::UnaryGe (rhs) => eval_unary_greater_or_equal(scope, rhs),
      AstNode::UnaryLt (rhs) => eval_unary_less(scope, rhs),
      AstNode::UnaryLe (rhs) => eval_unary_less_or_equal(scope, rhs),
     _=> {
       Ok(VALUE_NULL)
     }
      // AstNode::IterationClauses { .. } => Ok(not_implemented!("AstNode::IterationClauses")),
      // AstNode::IterationContextSingle { .. } => Ok(not_implemented!("AstNode::IterationContextSingle")),
      // AstNode::IterationContextRange { .. } => Ok(not_implemented!("AstNode::IterationContextRange")),
      // AstNode::NegatedCommaList { items } => eval_negated_comma_list(scope, items),
      // AstNode::QuantifiedClauses { .. } => Ok(not_implemented!("AstNode::QuantifiedClauses")),
      // AstNode::QuantifiedExpressionEvery { .. } => Ok(not_implemented!("AstNode::QuantifiedExpressionEvery")),
      // AstNode::QuantifiedExpressionSome { .. } => Ok(not_implemented!("AstNode::QuantifiedExpressionSome")),
    }
  }

  /// Returns `Value::ParameterName` built from specified `name`.
  fn eval_parameter_name(name: &Name) -> Value {
    Value::ParameterName(name.clone())
  }

  /// Evaluates temporal expression after `@` (at) literal.
  fn eval_at_literal(_scope: &Scope, text: &str) -> Result<Value> {
    if let Ok(date) = FeelDate::try_from(text) {
      return Ok(Value::Date(date));
    }
    if let Ok(date_time) = FeelDateTime::try_from(text) {
      return Ok(Value::DateTime(date_time));
    }
    if let Ok(time) = text.parse::<FeelTime>() {
      return Ok(Value::Time(time));
    }
    if let Ok(ym_duration) = FeelYearsAndMonthsDuration::try_from(text) {
      return Ok(Value::YearsAndMonthsDuration(ym_duration));
    }
    if let Ok(dt_duration) = FeelDaysAndTimeDuration::try_from(text) {
      return Ok(Value::DaysAndTimeDuration(dt_duration));
    }
    Ok(null_with_trace!("eval_at_literal: expected string literal"))
  }

  fn eval_named_parameter(scope: &Scope, lhs: &AstNode, rhs: &AstNode) -> Result<Value> {
    let lhv = eval(scope, lhs)?;
    let rhv = eval(scope, rhs)?;
    Ok(Value::NamedParameter(Box::new(lhv), Box::new(rhv)))
  }

  fn eval_named_parameters(scope: &Scope, items: &[AstNode]) -> Value {
    let mut named_parameters = BTreeMap::new();
    for item in items {
      if let Ok(Value::NamedParameter(name, value)) = eval(scope, item) {
        if let Value::ParameterName(name) = *name {
          named_parameters.insert(name, *value);
        }
      }
    }
    Value::NamedParameters(named_parameters)
  }

  fn eval_positional_parameters(scope: &Scope, items: &[AstNode]) -> Result<Value> {
    let mut values = vec![];
    for item in items {
      let value = eval(scope, item)?;
      values.push(value)
    }
    Ok(Value::PositionalParameters(values))
  }

  /// Evaluates division operator `/`.
  fn eval_division(scope: &Scope, lhs: &AstNode, rhs: &AstNode) -> Result<Value> {
    let lhv = eval(scope, lhs)?;
    let rhv = eval(scope, rhs)?;
    match lhv {
      Value::Number(lh) => match rhv {
        Value::Number(rh) => {
          if rh.abs() < f64::EPSILON {
            Ok(null_with_trace!("division by zero"))
          } else {
            Ok(Value::Number(lh / rh))
          }
        }
        _ => Ok(null_with_trace!("divisor is not a number: `{}`", rhv.stringify())),
      },
      _ => Ok(null_with_trace!("dividend is not a number: `{}`", lhv.stringify())),
    }
  }

  fn eval_name(scope: &Scope, name: &Name) -> Result<Value> {
    if let Some(value) = scope.get_entry(name) {
      return Ok(value);
    }
    if let Ok(bif) = Bif::from_str(&name.to_string()) {
      return Ok(Value::BuiltInFunction(bif));
    }
    Ok(null_with_trace!("context has no value for key `{}`", name.to_string()))
  }

  /// Evaluates path expression.
  fn eval_path(scope: &Scope, lhs: &AstNode, rhs: &AstNode) -> Result<Value> {
    let lhv = eval(scope, lhs)?;
    if let AstNode::Name(name) = rhs {
      match lhv {
        Value::Context(context) => {
          if let Some(value) = context.get_entry(name) {
            return Ok(value.clone());
          }
        }
        Value::List(items) => {
          let mut result = vec![];
          for item in items.to_vec() {
            if let Value::Context(context) = item {
              if let Some(value) = context.get_entry(name) {
                result.push(value.clone());
              }
            } else {
              return Ok(null_with_trace!("eval_path_expression: no context in list"));
            }
          }
          return Ok(Value::List(Values::new(result)));
        }
        Value::Date(date) => {
          return match name.to_string().as_str() {
            "year" => Ok(Value::Number(date.year() as f64)),
            "month" => Ok(Value::Number(date.month() as f64)),
            "day" => Ok(Value::Number(date.day() as f64)),
            "weekday" => {
              if let Some(day_num) = date.weekday() {
                Ok(Value::Number(day_num as f64))
              } else {
                Ok(null_with_trace!("could not retrieve weekday for date"))
              }
            }
            _ => Ok(null_with_trace!("no such property in date")),
          }
        }
        Value::DateTime(date_time) => {
          return match name.to_string().as_str() {
            "year" => Ok(Value::Number(date_time.year() as f64)),
            "month" => Ok(Value::Number(date_time.month() as f64)),
            "day" => Ok(Value::Number(date_time.day() as f64)),
            "weekday" => {
              if let Some(day_num) = date_time.weekday() {
                Ok(Value::Number(day_num as f64))
              } else {
                Ok(null_with_trace!("could not retrieve weekday for date and time"))
              }
            }
            "hour" => Ok(Value::Number(date_time.hour() as f64)),
            "minute" => Ok(Value::Number(date_time.minute() as f64)),
            "second" => Ok(Value::Number(date_time.second() as f64)),
            "time offset" => {
              if let Some(offset) = date_time.feel_time_offset() {
                return Ok(Value::DaysAndTimeDuration(FeelDaysAndTimeDuration::new(offset as i64, 0)));
              } else {
                Ok(VALUE_NULL)
              }
            }
            "timezone" => {
              if let Some(feel_time_zone) = date_time.feel_time_zone() {
                return Ok(Value::String(feel_time_zone));
              } else {
                Ok(VALUE_NULL)
              }
            }
            _ => Ok(null_with_trace!("no such property in date and time")),
          }
        }
        Value::Time(time) => {
          return match name.to_string().as_str() {
            "hour" => Ok(Value::Number(time.hour() as f64)),
            "minute" => Ok(Value::Number(time.minute() as f64)),
            "second" => Ok(Value::Number(time.second() as f64)),
            "time offset" => {
              if let Some(offset) = time.feel_time_offset() {
                return Ok(Value::DaysAndTimeDuration(FeelDaysAndTimeDuration::new(offset as i64, 0)));
              } else {
                Ok(VALUE_NULL)
              }
            }
            "timezone" => {
              if let Some(feel_time_zone) = time.feel_time_zone() {
                return Ok(Value::String(feel_time_zone));
              } else {
                Ok(VALUE_NULL)
              }
            }
            _ => Ok(null_with_trace!("no such property in date and time")),
          }
        }
        Value::DaysAndTimeDuration(dt_duration) => {
          return match name.to_string().as_str() {
            "days" => Ok(Value::Number(dt_duration.days() as f64)),
            "hours" => Ok(Value::Number(dt_duration.hours() as f64)),
            "minutes" => Ok(Value::Number(dt_duration.minutes() as f64)),
            "seconds" => Ok(Value::Number(dt_duration.seconds() as f64)),
            _ => Ok(null_with_trace!("no such property in days and time duration")),
          }
        }
        Value::YearsAndMonthsDuration(ym_duration) => {
          return match name.to_string().as_str() {
            "years" => Ok(Value::Number(ym_duration.years() as f64)),
            "months" => Ok(Value::Number(ym_duration.months() as f64)),
            _ => Ok(null_with_trace!("no such property in years and months duration")),
          }
        }
        _ => {}
      }
    }
    Ok(null_with_trace!(
      "eval_path_expression: no context (or list of contexts) on the left or no name on the right"
    ))
  }

  fn eval_qualified_name(scope: &Scope, names: &[Name]) -> Result<Value> {
    scope.search_deep(names).ok_or_else(|| context_has_no_value_for_key(names[0].stringify()))
  }

  fn eval_addition(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lvalue = eval(scope, left)?;
    let rvalue = eval(scope, right)?;
    match lvalue {
      Value::Number(lh) => match rvalue {
        Value::Number(rh) => return Ok(Value::Number(lh + rh)),
        value @ Value::Null(_) => return Ok(value),
        _ => {}
      },
      Value::String(mut lh) => {
        if let Value::String(rh) = rvalue {
          lh.push_str(&rh);
          return Ok(Value::String(lh));
        }
      }
      Value::DaysAndTimeDuration(l) => {
        if let Value::DaysAndTimeDuration(r) = rvalue {
          return Ok(Value::DaysAndTimeDuration(l + r));
        }
      }
      value @ Value::Null(_) => return Ok(value),
      _ => {}
    }
    trace_invalid_value_type!("eval_addition");
    Ok(VALUE_NULL)
  }

  fn eval_range_type(_scope: &Scope, feel_type: &FeelType) -> Result<Value> {
    Ok(Value::BuiltInType(FeelType::Range(Box::new(feel_type.clone()))))
  }

  fn eval_subtraction(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lvalue = eval(scope, left)?;
    let rvalue = eval(scope, right)?;
    match lvalue {
      Value::Number(l) => {
        if let Value::Number(r) = rvalue {
          return Ok(Value::Number(l - r));
        }
      }
      Value::DateTime(l) => {
        if let Value::DateTime(r) = rvalue {
          if let Some(a) = subtract(&l, &r) {
            return Ok(Value::DaysAndTimeDuration(FeelDaysAndTimeDuration::new(
              a / 1_000_000_000,
              a as u64 % 1_000_000_000,
            )));
          }
        }
      }
      _ => {}
    }
    trace_invalid_value_type!("eval_subtraction");
    Ok(VALUE_NULL)
  }

  fn eval_multiplication(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lvalue = eval(scope, left)?;
    let rvalue = eval(scope, right)?;
    match lvalue {
      Value::Number(l) => match rvalue {
        Value::Number(r) => Ok(Value::Number(l * r)),
        _ => {
          trace_invalid_value_type!("eval_multiplication");
          Ok(VALUE_NULL)
        }
      },
      _ => {
        trace_invalid_value_type!("eval_multiplication");
        Ok(VALUE_NULL)
      }
    }
  }

  fn eval_exponentiation(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lhv = eval(scope, left)?;
    let rhv = eval(scope, right)?;
    if let Value::Number(l) = lhv {
      if let Value::Number(r) = rhv {
        let result = l.powf(r);
        if result.is_finite() {
          return Ok(Value::Number(l.powf(r)));
        }
      }
    }
    Ok(VALUE_NULL)
  }

  /// Evaluates a list of values `Value::List` from `AstNode::ExpressionList`.
  fn eval_expression_list(scope: &Scope, items: &[AstNode]) -> Result<Value> {
    let mut values = vec![];
    for item in items {
      values.push(eval(scope, item)?);
    }
    Ok(Value::List(Values::new(values)))
  }

  fn eval_arithmetic_negation(scope: &Scope, right: &AstNode) -> Result<Value> {
    let rvalue = eval(scope, right)?;
    match rvalue {
      Value::Number(r) => Ok(Value::Number(-r)),
      _ => {
        trace_invalid_value_type!("eval_arithmetic_negation");
        Ok(VALUE_NULL)
      }
    }
  }

  fn eval_equal(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lv = eval(scope, left)?;
    let rv = eval(scope, right)?;
    if let Some(result) = eval_ternary_equality(&lv, &rv) {
      Ok(Value::Boolean(result))
    } else {
      println!("ERR = `{:?}` `{:?}`", lv, rv);
      Ok(VALUE_NULL)
    }
  }

  fn eval_not_equal(scope: &Scope, lhs: &AstNode, rhs: &AstNode) -> Result<Value> {
    let lhv = eval(scope, lhs)?;
    let rhv = eval(scope, rhs)?;
    if let Some(result) = eval_ternary_equality(&lhv, &rhv) {
      Ok(Value::Boolean(!result))
    } else {
      Ok(VALUE_NULL)
    }
  }

  pub fn eval_ternary_equality(lhs: &Value, rhs: &Value) -> Option<bool> {
    match lhs {
      Value::Boolean(ls) => match rhs {
        Value::Boolean(rs) => Some(*ls == *rs),
        Value::Null(_) => Some(false),
        _ => None,
      },
      Value::Number(ls) => match rhs {
        Value::Number(rs) => Some((*ls - *rs).abs() < f64::EPSILON),
        Value::Null(_) => Some(false),
        _ => None,
      },
      Value::String(ls) => match rhs {
        Value::String(rs) => Some(*ls == *rs),
        Value::Null(_) => Some(false),
        _ => None,
      },
      Value::Context(ls) => match rhs {
        Value::Context(rs) => {
          if ls.keys().len() == rs.keys().len() {
            for (key1, value1) in ls.deref() {
              if let Some(value2) = rs.get_entry(key1) {
                if let Some(equal) = eval_ternary_equality(value1, value2) {
                  if !equal {
                    return Some(false); // values in entries are NOT EQUAL
                  }
                } else {
                  // values in entries can not be compared
                  return None;
                }
              } else {
                // there is no such key in the right-side context
                return Some(false);
              }
            }
            // contexts are EQUAL
            return Some(true);
          }
          // contexts have different number of keys, so they are NOT EQUAL
          Some(false)
        }
        Value::Null(_) => Some(false),
        _ => None,
      },
      Value::Date(ls) => match rhs {
        Value::Date(rs) => Some(*ls == *rs),
        Value::Null(_) => Some(false),
        _ => None,
      },
      Value::Time(ls) => match rhs {
        Value::Time(rs) => ls.equal(rs),
        Value::Null(_) => Some(false),
        _ => None,
      },
      Value::DateTime(ls) => match rhs {
        Value::DateTime(rs) => ls.equal(rs),
        Value::Null(_) => Some(false),
        _ => None,
      },
      Value::DaysAndTimeDuration(ls) => match rhs {
        Value::DaysAndTimeDuration(rs) => Some(*ls == *rs),
        Value::Null(_) => Some(false),
        _ => None,
      },
      Value::YearsAndMonthsDuration(ls) => match rhs {
        Value::YearsAndMonthsDuration(rs) => Some(*ls == *rs),
        Value::Null(_) => Some(false),
        _ => None,
      },
      Value::Null(_) => match rhs {
        Value::Null(_) => Some(true),
        _ => None,
      },
      Value::List(ls) => match rhs {
        Value::List(rs) => {
          if ls.to_vec().len() == rs.to_vec().len() {
            for (l, r) in ls.to_vec().iter().zip(rs.to_vec().iter()) {
              if let Some(true) = eval_ternary_equality(l, r) {
              } else {
                return Some(false);
              }
            }
            Some(true)
          } else {
            Some(false)
          }
        }
        Value::Null(_) => Some(false),
        _ => None,
      },
      _ => None,
    }
  }

  fn eval_less(scope: &Scope, lhs: &AstNode, rhs: &AstNode) -> Result<Value> {
    let lhv = eval(scope, lhs)?;
    let rhv = eval(scope, rhs)?;
    if let Value::Number(l) = lhv {
      if let Value::Number(r) = rhv {
        return Ok(Value::Boolean(l < r));
      }
    }
    Ok(null_with_trace!("eval_less: lhv={:?} rhv={:?}", lhv, rhv))
  }

  fn eval_unary_less(scope: &Scope, right: &AstNode) -> Result<Value> {
    let value = eval(scope, right)?;
    Ok(Value::UnaryLess(Box::from(value)))
  }

  fn eval_less_or_equal(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lvalue = eval(scope, left)?;
    let rvalue = eval(scope, right)?;
    match lvalue {
      Value::Number(l) => match rvalue {
        Value::Number(r) => Ok(Value::Boolean(l <= r)),
        _ => {
          trace_invalid_value_type!("eval_less_or_equal");
          Ok(VALUE_NULL)
        }
      },
      _ => {
        trace_invalid_value_type!("eval_less_or_equal");
        Ok(VALUE_NULL)
      }
    }
  }

  fn eval_unary_less_or_equal(scope: &Scope, right: &AstNode) -> Result<Value> {
    let value = eval(scope, right)?;
    Ok(Value::UnaryLessOrEqual(Box::from(value)))
  }

  fn eval_greater(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lvalue = eval(scope, left)?;
    let rvalue = eval(scope, right)?;
    match lvalue {
      Value::Number(l) => match rvalue {
        Value::Number(r) => Ok(Value::Boolean(l > r)),
        _ => {
          trace_invalid_value_type!("eval_greater: right argument is not a number");
          Ok(VALUE_NULL)
        }
      },
      Value::Date(l) => match rvalue {
        Value::Date(r) => {
          if let Some(result) = l.after(&r) {
            Ok(Value::Boolean(result))
          } else {
            trace_invalid_value_type!("eval_greater: date comparison returned None");
            Ok(VALUE_NULL)
          }
        }
        _ => {
          trace_invalid_value_type!("eval_greater");
          Ok(VALUE_NULL)
        }
      },
      other => {
        trace_invalid_value_type!(format!("eval_greater: {:?}", other));
        Ok(VALUE_NULL)
      }
    }
  }

  fn eval_unary_greater(scope: &Scope, rhs: &AstNode) -> Result<Value> {
    let rhv = eval(scope, rhs)?;
    Ok(Value::UnaryGreater(Box::from(rhv)))
  }

  fn eval_greater_or_equal(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lvalue = eval(scope, left)?;
    let rvalue = eval(scope, right)?;
    match lvalue {
      Value::Number(l) => match rvalue {
        Value::Number(r) => Ok(Value::Boolean(l >= r)),
        _ => {
          trace_invalid_value_type!("eval_greater");
          Ok(VALUE_NULL)
        }
      },
      _ => {
        trace_invalid_value_type!("eval_greater");
        Ok(VALUE_NULL)
      }
    }
  }

  fn eval_unary_greater_or_equal(scope: &Scope, right: &AstNode) -> Result<Value> {
    let value = eval(scope, right)?;
    Ok(Value::UnaryGreaterOrEqual(Box::from(value)))
  }

  /// Semantics of `between`:
  ///
  /// `e1 between e2 and e3` is an equivalent of `e1 >= e2 and e1 <= e3`
  ///
  fn eval_between(scope: &Scope, value: &AstNode, left: &AstNode, right: &AstNode) -> Result<Value> {
    let between = eval(scope, value)?;
    let lvalue = eval(scope, left)?;
    let rvalue = eval(scope, right)?;
    match between {
      Value::Number(b) => {
        if let Value::Number(l) = lvalue {
          if let Value::Number(r) = rvalue {
            return Ok(Value::Boolean(l <= b && b <= r));
          }
        }
      }
      Value::String(b) => {
        if let Value::String(l) = lvalue {
          if let Value::String(r) = rvalue {
            return Ok(Value::Boolean(l <= b && b <= r));
          }
        }
      }
      Value::Date(b) => {
        if let Value::Date(l) = lvalue {
          if let Value::Date(r) = rvalue {
            if let Some(result) = b.between(&l, &r, true, true) {
              return Ok(Value::Boolean(result));
            }
          }
        }
      }
      Value::Time(b) => {
        if let Value::Time(l) = lvalue {
          if let Value::Time(r) = rvalue {
            if let Some(result) = b.between(&l, &r, true, true) {
              return Ok(Value::Boolean(result));
            }
          }
        }
      }
      Value::DateTime(b) => {
        if let Value::DateTime(l) = lvalue {
          if let Value::DateTime(r) = rvalue {
            if let Some(result) = b.between(&l, &r, true, true) {
              return Ok(Value::Boolean(result));
            }
          }
        }
      }
      Value::YearsAndMonthsDuration(b) => {
        if let Value::YearsAndMonthsDuration(l) = lvalue {
          if let Value::YearsAndMonthsDuration(r) = rvalue {
            return Ok(Value::Boolean(l <= b && b <= r));
          }
        }
      }
      Value::DaysAndTimeDuration(b) => {
        if let Value::DaysAndTimeDuration(l) = lvalue {
          if let Value::DaysAndTimeDuration(r) = rvalue {
            return Ok(Value::Boolean(l <= b && b <= r));
          }
        }
      }
      _ => {}
    }
    trace_invalid_value_type!("eval_between");
    Ok(VALUE_NULL)
  }

  ///
  fn eval_out(scope: &Scope, lhs: &AstNode, rhs: &AstNode) -> Result<Value> {
    match eval_in(scope, lhs, rhs)? {
      Value::Boolean(true) => Ok(eval(scope, lhs)?),
      _ => Ok(VALUE_NULL),
    }
  }

  ///
  fn eval_numeric(before: &str, after: &str) -> Result<Value> {
    let text = format!("{}.{}", before, after);
    if let Ok(num) = text.parse::<f64>() {
      Ok(Value::Number(num))
    } else {
      Ok(null_with_trace!("number conversion error"))
    }
  }

  /// Evaluates `IF` expression.
  fn eval_if_expression(scope: &Scope, condition: &AstNode, when_true: &AstNode, when_false: &AstNode) -> Result<Value> {
    match eval(scope, condition)? {
      Value::Boolean(true) => Ok(eval(scope, when_true)?),
      Value::Boolean(false) | Value::Null(_) => Ok(eval(scope, when_false)?),
      _ => Ok(Value::Null(Some("condition in 'if' expression is not a boolean value".to_string()))),
    }
  }

  fn eval_in(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let left = eval(scope, left)?;
    let right = eval(scope, right)?;
    match right {
      inner @ Value::Number(_)
      | inner @ Value::String(_)
      | inner @ Value::Boolean(_)
      | inner @ Value::Date(_)
      | inner @ Value::Time(_)
      | inner @ Value::DateTime(_)
      | inner @ Value::YearsAndMonthsDuration(_)
      | inner @ Value::DaysAndTimeDuration(_)
      | inner @ Value::Context(_) => Ok(eval_in_equal(&left, &inner)),
      Value::Range(_, _, _, _) => Ok(eval_in_range(&left, &right)),
      Value::List(inner) => {
        if let Value::List(_) = left {
          Ok(eval_in_list_in_list(&left, &inner.to_vec()))
        } else {
          eval_in_list(&left, &inner.to_vec())
        }
      }
      Value::CommaList(inner) => eval_in_list(&left, &inner.to_vec()),
      Value::NegatedCommaList(items) => Ok(eval_in_negated_list(&left, &items)),
      Value::UnaryLess(inner) => Ok(eval_in_unary_less(&left, inner.borrow())),
      Value::UnaryLessOrEqual(inner) => Ok(eval_in_unary_less_or_equal(&left, inner.borrow())),
      Value::UnaryGreater(inner) => Ok(eval_in_unary_greater(&left, inner.borrow())),
      Value::UnaryGreaterOrEqual(inner) => Ok(eval_in_unary_greater_or_equal(&left, inner.borrow())),
      Value::Irrelevant => match left {
        Value::Null(_) => Ok(VALUE_FALSE),
        _ => Ok(VALUE_TRUE),
      },
      _ => {
        trace_invalid_value_type!("eval_in");
        Ok(VALUE_NULL)
      }
    }
  }

  fn eval_in_equal(left: &Value, right: &Value) -> Value {
    if let Some(true) = eval_ternary_equality(left, right) {
      VALUE_TRUE
    } else {
      VALUE_FALSE
    }
  }

  fn eval_in_unary_less(left: &Value, right: &Value) -> Value {
    match right {
      Value::Number(r) => {
        if let Value::Number(l) = left {
          return Value::Boolean(*l < *r);
        }
      }
      Value::String(r) => {
        if let Value::String(l) = left {
          return Value::Boolean(l < r);
        }
      }
      Value::Date(r) => {
        if let Value::Date(l) = left {
          if let Some(result) = l.before(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::Time(r) => {
        if let Value::Time(l) = left {
          if let Some(result) = l.before(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::DateTime(r) => {
        if let Value::DateTime(l) = left {
          if let Some(result) = l.before(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::YearsAndMonthsDuration(r) => {
        if let Value::YearsAndMonthsDuration(l) = left {
          return Value::Boolean(l < r);
        }
      }
      Value::DaysAndTimeDuration(r) => {
        if let Value::DaysAndTimeDuration(l) = left {
          return Value::Boolean(l < r);
        }
      }
      _ => {}
    }
    trace_invalid_value_type!("eval_in_unary_less");
    VALUE_NULL
  }

  fn eval_in_unary_less_or_equal(left: &Value, right: &Value) -> Value {
    match right {
      Value::Number(r) => {
        if let Value::Number(l) = left {
          return Value::Boolean(*l <= *r);
        }
      }
      Value::String(r) => {
        if let Value::String(l) = left {
          return Value::Boolean(l <= r);
        }
      }
      Value::Date(r) => {
        if let Value::Date(l) = left {
          if let Some(result) = l.before_or_equal(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::Time(r) => {
        if let Value::Time(l) = left {
          if let Some(result) = l.before_or_equal(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::DateTime(r) => {
        if let Value::DateTime(l) = left {
          if let Some(result) = l.before_or_equal(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::YearsAndMonthsDuration(r) => {
        if let Value::YearsAndMonthsDuration(l) = left {
          return Value::Boolean(l <= r);
        }
      }
      Value::DaysAndTimeDuration(r) => {
        if let Value::DaysAndTimeDuration(l) = left {
          return Value::Boolean(l <= r);
        }
      }
      _ => {}
    }
    trace_invalid_value_type!("eval_in_unary_less_or_equal");
    VALUE_NULL
  }

  fn eval_in_unary_greater(left: &Value, right: &Value) -> Value {
    match right {
      Value::Number(r) => {
        if let Value::Number(l) = left {
          return Value::Boolean(*l > *r);
        }
      }
      Value::String(r) => {
        if let Value::String(l) = left {
          return Value::Boolean(l > r);
        }
      }
      Value::Date(r) => {
        if let Value::Date(l) = left {
          if let Some(result) = l.after(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::Time(r) => {
        if let Value::Time(l) = left {
          if let Some(result) = l.after(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::DateTime(r) => {
        if let Value::DateTime(l) = left {
          if let Some(result) = l.after(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::YearsAndMonthsDuration(r) => {
        if let Value::YearsAndMonthsDuration(l) = left {
          return Value::Boolean(l > r);
        }
      }
      Value::DaysAndTimeDuration(r) => {
        if let Value::DaysAndTimeDuration(l) = left {
          return Value::Boolean(l > r);
        }
      }
      _ => {}
    }
    trace_invalid_value_type!("eval_in_unary_greater");
    VALUE_NULL
  }

  fn eval_in_unary_greater_or_equal(left: &Value, right: &Value) -> Value {
    match right {
      Value::Number(r) => {
        if let Value::Number(l) = left {
          return Value::Boolean(*l >= *r);
        }
      }
      Value::String(r) => {
        if let Value::String(left_value) = left {
          return Value::Boolean(left_value >= r);
        }
      }
      Value::Date(r) => {
        if let Value::Date(l) = left {
          if let Some(result) = l.after_or_equal(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::Time(r) => {
        if let Value::Time(l) = left {
          if let Some(result) = l.after_or_equal(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::DateTime(r) => {
        if let Value::DateTime(l) = left {
          if let Some(result) = l.after_or_equal(&r) {
            return Value::Boolean(result);
          }
        }
      }
      Value::YearsAndMonthsDuration(r) => {
        if let Value::YearsAndMonthsDuration(l) = left {
          return Value::Boolean(l >= r);
        }
      }
      Value::DaysAndTimeDuration(r) => {
        if let Value::DaysAndTimeDuration(l) = left {
          return Value::Boolean(l >= r);
        }
      }
      _ => {}
    }
    trace_invalid_value_type!("eval_in_unary_greater_or_equal");
    VALUE_NULL
  }

  fn eval_in_range(left: &Value, right: &Value) -> Value {
    match left {
      Value::Number(value) => match right {
        Value::Range(l, l_closed, r, r_closed) => match l.borrow() {
          Value::Number(lv) => match r.borrow() {
            Value::Number(rv) => {
              let l_ok = if *l_closed { value >= lv } else { value > lv };
              let r_ok = if *r_closed { value <= rv } else { value < rv };
              Value::Boolean(l_ok && r_ok)
            }
            _ => {
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
          },
          _ => {
            trace_invalid_value_type!("eval_in_range");
            VALUE_NULL
          }
        },
        _ => {
          trace_invalid_value_type!("eval_in_range");
          VALUE_NULL
        }
      },
      Value::String(value) => match right {
        Value::Range(l, l_closed, r, r_closed) => match l.borrow() {
          Value::String(lv) => match r.borrow() {
            Value::String(rv) => {
              let l_ok = if *l_closed { value >= lv } else { value > lv };
              let r_ok = if *r_closed { value <= rv } else { value < rv };
              Value::Boolean(l_ok && r_ok)
            }
            _ => {
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
          },
          _ => {
            trace_invalid_value_type!("eval_in_range");
            VALUE_NULL
          }
        },
        _ => {
          trace_invalid_value_type!("eval_in_range");
          VALUE_NULL
        }
      },
      Value::Date(value) => match right {
        Value::Range(l, l_closed, r, r_closed) => match l.borrow() {
          Value::Date(lv) => match r.borrow() {
            Value::Date(rv) => {
              if let Some(b) = value.between(lv, rv, *l_closed, *r_closed) {
                return Value::Boolean(b);
              }
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
            _ => {
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
          },
          _ => {
            trace_invalid_value_type!("eval_in_range");
            VALUE_NULL
          }
        },
        _ => {
          trace_invalid_value_type!("eval_in_range");
          VALUE_NULL
        }
      },
      Value::Time(value) => match right {
        Value::Range(l, l_closed, r, r_closed) => match l.borrow() {
          Value::Time(lv) => match r.borrow() {
            Value::Time(rv) => {
              if let Some(b) = value.between(lv, rv, *l_closed, *r_closed) {
                return Value::Boolean(b);
              }
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
            _ => {
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
          },
          _ => {
            trace_invalid_value_type!("eval_in_range");
            VALUE_NULL
          }
        },
        _ => {
          trace_invalid_value_type!("eval_in_range");
          VALUE_NULL
        }
      },
      Value::DateTime(value) => match right {
        Value::Range(l, l_closed, r, r_closed) => match l.borrow() {
          Value::DateTime(lv) => match r.borrow() {
            Value::DateTime(rv) => {
              if let Some(b) = value.between(lv, rv, *l_closed, *r_closed) {
                return Value::Boolean(b);
              }
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
            _ => {
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
          },
          _ => {
            trace_invalid_value_type!("eval_in_range");
            VALUE_NULL
          }
        },
        _ => {
          trace_invalid_value_type!("eval_in_range");
          VALUE_NULL
        }
      },
      Value::YearsAndMonthsDuration(value) => match right {
        Value::Range(l, l_closed, r, r_closed) => match l.borrow() {
          Value::YearsAndMonthsDuration(lv) => match r.borrow() {
            Value::YearsAndMonthsDuration(rv) => {
              let l_ok = if *l_closed { value >= lv } else { value > lv };
              let r_ok = if *r_closed { value <= rv } else { value < rv };
              Value::Boolean(l_ok && r_ok)
            }
            _ => {
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
          },
          _ => {
            trace_invalid_value_type!("eval_in_range");
            VALUE_NULL
          }
        },
        _ => {
          trace_invalid_value_type!("eval_in_range");
          VALUE_NULL
        }
      },
      Value::DaysAndTimeDuration(value) => match right {
        Value::Range(l, l_closed, r, r_closed) => match l.borrow() {
          Value::DaysAndTimeDuration(lv) => match r.borrow() {
            Value::DaysAndTimeDuration(rv) => {
              let l_ok = if *l_closed { value >= lv } else { value > lv };
              let r_ok = if *r_closed { value <= rv } else { value < rv };
              Value::Boolean(l_ok && r_ok)
            }
            _ => {
              trace_invalid_value_type!("eval_in_range");
              VALUE_NULL
            }
          },
          _ => {
            trace_invalid_value_type!("eval_in_range");
            VALUE_NULL
          }
        },
        _ => {
          trace_invalid_value_type!("eval_in_range");
          VALUE_NULL
        }
      },
      _ => {
        trace_invalid_value_type!("eval_in_range");
        VALUE_NULL
      }
    }
  }

  fn eval_in_list(left: &Value, items: &[Value]) -> Result<Value> {
    for item in items {
      match item {
        inner @ Value::String(_)
        | inner @ Value::Number(_)
        | inner @ Value::Boolean(_)
        | inner @ Value::Date(_)
        | inner @ Value::Time(_)
        | inner @ Value::DateTime(_)
        | inner @ Value::YearsAndMonthsDuration(_)
        | inner @ Value::DaysAndTimeDuration(_)
        | inner @ Value::Context(_) => {
          if let Value::Boolean(true) = eval_in_equal(&left, inner) {
            return Ok(VALUE_TRUE);
          }
        }
        Value::UnaryLess(inner) => {
          if let Value::Boolean(true) = eval_in_unary_less(&left, inner.borrow()) {
            return Ok(VALUE_TRUE);
          }
        }
        Value::UnaryLessOrEqual(inner) => {
          if let Value::Boolean(true) = eval_in_unary_less_or_equal(&left, inner.borrow()) {
            return Ok(VALUE_TRUE);
          }
        }
        Value::UnaryGreater(inner) => {
          if let Value::Boolean(true) = eval_in_unary_greater(&left, inner.borrow()) {
            return Ok(VALUE_TRUE);
          }
        }
        Value::UnaryGreaterOrEqual(inner) => {
          if let Value::Boolean(true) = eval_in_unary_greater_or_equal(&left, inner.borrow()) {
            return Ok(VALUE_TRUE);
          }
        }
        Value::List(inner) => {
          if let Value::Boolean(true) = eval_in_list(&left, inner.to_vec())? {
            return Ok(VALUE_TRUE);
          }
        }
        inner @ Value::Range(_, _, _, _) => {
          if let Value::Boolean(true) = eval_in_range(&left, inner) {
            return Ok(VALUE_TRUE);
          }
        }
        _ => return Ok(VALUE_NULL),
      }
    }
    Ok(VALUE_FALSE)
  }

  /// Checks if all elements from `list` are present in `items`.
  fn eval_in_list_in_list(list: &Value, items: &[Value]) -> Value {
    if let Value::List(lhs) = list {
      for item in items {
        if let Value::List(rhs) = item {
          let mut available: HashSet<usize> = (0..rhs.to_vec().len()).collect();
          for l in lhs.to_vec() {
            let mut found = false;
            for (index, r) in rhs.to_vec().iter().enumerate() {
              if available.contains(&index) {
                if let Value::Boolean(true) = eval_in_equal(l, r) {
                  available.remove(&index);
                  found = true;
                  break;
                }
              }
            }
            if !found {
              return VALUE_FALSE;
            }
          }
          return VALUE_TRUE;
        }
      }
    }
    VALUE_FALSE
  }

  fn eval_in_negated_list(left: &Value, items: &[Value]) -> Value {
    for item in items {
      match item {
        ref inner @ Value::Number(_) | ref inner @ Value::String(_) => {
          if let Value::Boolean(true) = eval_in_equal(left, inner) {
            return Value::Boolean(false);
          }
        }
        Value::UnaryLess(inner) => {
          if let Value::Boolean(true) = eval_in_unary_less(left, inner.borrow()) {
            return Value::Boolean(false);
          }
        }
        Value::UnaryLessOrEqual(inner) => {
          if let Value::Boolean(true) = eval_in_unary_less_or_equal(left, inner.borrow()) {
            return Value::Boolean(false);
          }
        }
        Value::UnaryGreater(inner) => {
          if let Value::Boolean(true) = eval_in_unary_greater(left, inner.borrow()) {
            return Value::Boolean(false);
          }
        }
        Value::UnaryGreaterOrEqual(inner) => {
          if let Value::Boolean(true) = eval_in_unary_greater_or_equal(left, inner.borrow()) {
            return Value::Boolean(false);
          }
        }
        _ => {
          trace_invalid_value_type!("eval_in_negated_list");
          return VALUE_NULL;
        }
      }
    }
    Value::Boolean(true)
  }

  /// Semantics of conjunction.
  /// ```text
  /// left        right        result
  ///─────────────────────────────────
  /// true        true         true
  /// true        false        false
  /// true        otherwise    null
  /// false       true         false
  /// false       false        false
  /// false       otherwise    false
  /// otherwise   true         null
  /// otherwise   false        false
  /// otherwise   otherwise    null
  /// ```
  fn eval_conjunction(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lvalue = eval(scope, left)?;
    let rvalue = eval(scope, right)?;
    match lvalue {
      Value::Boolean(l) => match rvalue {
        Value::Boolean(r) => Ok(Value::Boolean(l && r)),
        _ => {
          if l {
            Ok(VALUE_NULL)
          } else {
            Ok(VALUE_FALSE)
          }
        }
      },
      _ => match rvalue {
        Value::Boolean(r) => {
          if r {
            Ok(VALUE_NULL)
          } else {
            Ok(VALUE_FALSE)
          }
        }
        _ => Ok(VALUE_NULL),
      },
    }
  }

  /// Semantics of disjunction.
  /// ```text
  /// left        right        result
  ///─────────────────────────────────
  /// true        true         true
  /// true        false        true
  /// true        otherwise    true
  /// false       true         true
  /// false       false        false
  /// false       otherwise    null
  /// otherwise   true         true
  /// otherwise   false        null
  /// otherwise   otherwise    null
  /// ```
  fn eval_disjunction(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let lvalue = eval(scope, left)?;
    let rvalue = eval(scope, right)?;
    match lvalue {
      Value::Boolean(l) => match rvalue {
        Value::Boolean(r) => Ok(Value::Boolean(l || r)),
        _ => {
          if l {
            Ok(VALUE_TRUE)
          } else {
            Ok(VALUE_NULL)
          }
        }
      },
      _ => match rvalue {
        Value::Boolean(r) => {
          if r {
            Ok(VALUE_TRUE)
          } else {
            Ok(VALUE_NULL)
          }
        }
        _ => Ok(VALUE_NULL),
      },
    }
  }

  fn eval_interval_end(scope: &Scope, lhs: &AstNode, closed: &bool) -> Result<Value> {
    let lhv = eval(scope, lhs)?;
    Ok(Value::IntervalEnd(Box::new(lhv), *closed))
  }

  fn eval_interval_start(scope: &Scope, lhs: &AstNode, closed: &bool) -> Result<Value> {
    let lhv = eval(scope, lhs)?;
    Ok(Value::IntervalStart(Box::new(lhv), *closed))
  }

  fn eval_interval(scope: &Scope, lhs: &AstNode, rhs: &AstNode) -> Result<Value> {
    if let Value::IntervalStart(lhv, l_closed) = eval(scope, lhs)? {
      if let Value::IntervalEnd(rhv, r_closed) = eval(scope, rhs)? {
        return Ok(Value::Range(lhv, l_closed, rhv, r_closed));
      }
    }
    Ok(null_with_trace!("invalid_interval"))
  }

  /// Evaluates filter expression.
  fn eval_filter_expression(scope: &Scope, lhs: &AstNode, rhs: &AstNode) -> Result<Value> {
    if let Ok(Value::Boolean(flag)) = eval(scope, rhs) {
      match eval(scope, lhs)? {
        v @ Value::List(_) => return if flag { Ok(v) } else { Ok(Value::List(Values::empty())) },
        v @ Value::Number(_)
        | v @ Value::Boolean(_)
        | v @ Value::String(_)
        | v @ Value::Date(_)
        | v @ Value::DateTime(_)
        | v @ Value::Time(_)
        | v @ Value::DaysAndTimeDuration(_)
        | v @ Value::YearsAndMonthsDuration(_)
        | v @ Value::Context(_) => {
          return if flag {
            Ok(Value::List(Values::new(vec![v])))
          } else {
            Ok(Value::List(Values::empty()))
          }
        }
        _ => {}
      }
    } else if let Ok(Value::Number(index)) = eval(scope, rhs) {
      if !is_integer(index) {
        return Ok(null_with_trace!("eval_filter_expression: index in filter is not an integer"));
      }
      let n = round_half_to_even(index, 0.) as i64;
      match eval(scope, lhs)? {
        Value::List(values) => {
          let count = values.to_vec().len();
          if n > 0 && (n as usize) <= count {
            if let Some(value) = values.to_vec().get((n - 1) as usize) {
              return Ok(value.clone());
            }
          }
          let n = n.abs() as usize;
          if n > 0 && n <= count {
            if let Some(value) = values.to_vec().get(values.to_vec().len() - n) {
              return Ok(value.clone());
            }
          }
        }
        v @ Value::Number(_)
        | v @ Value::Boolean(_)
        | v @ Value::String(_)
        | v @ Value::Date(_)
        | v @ Value::DateTime(_)
        | v @ Value::Time(_)
        | v @ Value::DaysAndTimeDuration(_)
        | v @ Value::YearsAndMonthsDuration(_)
        | v @ Value::Context(_) => {
          if n == 1 {
            return Ok(v);
          }
        }
        _ => {}
      }
    } else if let Value::List(values) = eval(scope, lhs)? {
      let mut result = vec![];
      let name_item = Name::from("item");
      for value in values.to_vec() {
        let (added_local_context, has_item_entry) = if let Value::Context(local_context) = &value {
          scope.push(local_context.clone());
          if local_context.contains_entry(&name_item) {
            (true, true)
          } else {
            (true, false)
          }
        } else {
          (false, false)
        };
        if !has_item_entry {
          let mut special_context = FeelContext::default();
          special_context.set_entry(&Name::from("item"), value.clone());
          scope.push(special_context);
        }
        let rhv = eval(&scope, rhs);
        if let Ok(Value::Boolean(true)) = rhv {
          result.push(value.clone());
        }
        if !has_item_entry {
          scope.pop();
        }
        if added_local_context {
          scope.pop();
        }
      }
      if result.len() == 1 {
        return Ok(result[0].to_owned());
      }
      return Ok(Value::List(Values::new(result)));
    }
    Ok(null_with_trace!("eval_filter_expression: exit"))
  }

  /// [IterationState] represents the current state of single iteration clause.
  struct IterationState {
    /// Flag indicating if the iteration is over the range of values (== `true`) or a list of values (== `false`).
    is_range: bool,
    /// Name of the variable used in iteration.
    name: Name,
    /// Current value of the looping index.
    index: i64,
    /// The step in the iteration.
    step: i64,
    /// Start value.
    start: i64,
    /// End value.
    end: i64,
    /// Collection of values to iterate over (if not a range).
    values: Values,
  }

  /// Evaluates the `FOR` expression.
  fn eval_for_expression(scope: &Scope, iteration_clauses: &[(Name, AstNode, OptAstNode)], right: &AstNode) -> Result<Value> {
    let mut iteration_states = vec![];
    let mut iteration_context = FeelContext::default();
    let name_partial = Name::from("partial");
    for (name, left_node, right_node) in iteration_clauses {
      match (left_node, right_node) {
        (expr_node, None) => {
          if let Value::List(values) = eval(scope, &expr_node)? {
            if values.to_vec().is_empty() {
              iteration_context.set_entry(name, VALUE_NULL)
            } else {
              iteration_states.push(IterationState {
                is_range: false,
                name: name.clone(),
                index: 0,
                step: 1,
                start: 0,
                end: (values.to_vec().len() as i64) - 1,
                values,
              });
            }
          }
        }
        (range_start_node, Some(range_end_node)) => {
          if let Value::Number(range_start) = eval(scope, &range_start_node)? {
            if let Value::Number(range_end) = eval(scope, &range_end_node)? {
              iteration_states.push(IterationState {
                is_range: true,
                name: name.clone(),
                index: range_start.trunc() as i64,
                step: if range_start < range_end { 1_i64 } else { -1_i64 },
                start: range_start.trunc() as i64,
                end: range_end.trunc() as i64,
                values: Values::empty(),
              });
            }
          }
        }
      }
    }
    iteration_states.reverse();
    let mut results = vec![];
    if !iteration_states.is_empty() {
      'outer: loop {
        let mut is_empty_iteration = true;
        for iteration_state in &iteration_states {
          if iteration_state.is_range {
            iteration_context.set_entry(&iteration_state.name, Value::Number(iteration_state.index as f64));
          } else {
            iteration_context.set_entry(&iteration_state.name, iteration_state.values.to_vec()[iteration_state.index as usize].clone());
          }
          is_empty_iteration = false;
        }
        if !is_empty_iteration {
          iteration_context.set_entry(&name_partial, Value::List(Values::new(results.clone())));
          scope.push(iteration_context.clone());
          let iteration_value = eval(scope, &right)?;
          scope.pop();
          results.push(iteration_value);
        }
        let last_iteration_state_index = iteration_states.len() - 1;
        let mut overflow = true;
        'inner: for (i, iteration_state) in iteration_states.iter_mut().enumerate() {
          if overflow {
            // check if the last iteration variable has overflows
            if i == last_iteration_state_index {
              if iteration_state.step > 0 && iteration_state.index + iteration_state.step > iteration_state.end {
                break 'outer;
              }
              if iteration_state.step < 0 && iteration_state.index + iteration_state.step < iteration_state.end {
                break 'outer;
              }
            }
            if iteration_state.step > 0 {
              if iteration_state.index + iteration_state.step <= iteration_state.end {
                iteration_state.index += iteration_state.step;
                overflow = false;
              } else {
                iteration_state.index = iteration_state.start;
                overflow = true;
              }
            }
            if iteration_state.step < 0 {
              if iteration_state.index + iteration_state.step >= iteration_state.end {
                iteration_state.index += iteration_state.step;
                overflow = false;
              } else {
                iteration_state.index = iteration_state.start;
                overflow = true;
              }
            }
            if iteration_state.step == 0 {
              break 'outer; // if there will be the step parametrized in the future then break the loop when no stepping
            }
          } else {
            break 'inner;
          }
        }
      }
    }
    Ok(Value::List(Values::new(results)))
  }

  fn eval_function_body(scope: &Scope, lhs: &AstNode, external: bool) -> Result<Value> {
    if external {
      // prepare function's body built as external function call (usually context that defines what to call)
      let function_body = FunctionBody::External(Box::new(eval(scope, lhs)?));
      Ok(Value::FunctionBody(function_body))
    } else {
      // prepare function's body built from literal expression
      let function_body = FunctionBody::LiteralExpression(lhs.clone());
      Ok(Value::FunctionBody(function_body))
    }
  }

  /// Evaluates function definition.
  ///
  /// # Arguments
  ///
  /// - `scope` evaluation scope,
  /// - `parameters` AstNode defining the list of formal parameters,
  /// - `body` AstNode defining the function's body,
  ///
  fn eval_function_definition(scope: &Scope, lhs: &AstNode, rhs: &AstNode) -> Result<Value> {
    if let Value::FunctionBody(function_body) = eval(scope, rhs)? {
      if let AstNode::FormalParameters(formal_parameters_node) = lhs {
        let mut formal_parameters = vec![];
        for formal_parameter_node in formal_parameters_node {
          if let AstNode::FormalParameter(parameter_name, parameter_type) = formal_parameter_node {
            formal_parameters.push((parameter_name.clone(), parameter_type.clone()));
          }
        }
        return Ok(Value::FunctionDefinition(formal_parameters, function_body));
      }
    }
    Ok(null_with_trace!("evaluation of the function body in function definition failed"))
  }

  fn eval_function_invocation(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    match right {
      AstNode::PositionalParameters(positional_parameters) => {
        let l_value = eval(scope, left)?;
        match l_value {
          Value::BuiltInFunction(bif) => match bif {
            Bif::Abs => eval_bif::positional::abs(scope, positional_parameters),
            Bif::After => Ok(not_implemented!("Bif::After")),
            Bif::All => eval_bif::positional::all(scope, positional_parameters),
            Bif::Any => eval_bif::positional::any(scope, positional_parameters),
            Bif::Append => eval_bif::positional::append(scope, positional_parameters),
            Bif::Before => Ok(not_implemented!("Bif::Before")),
            Bif::Ceiling => eval_bif::positional::ceiling(scope, positional_parameters),
            Bif::Coincides => Ok(not_implemented!("Bif::Coincides")),
            Bif::Concatenate => eval_bif::positional::concatenate(scope, positional_parameters),
            Bif::Contains => eval_bif::positional::contains(scope, positional_parameters),
            Bif::Count => eval_bif::positional::count(scope, positional_parameters),
            Bif::Date => eval_bif::positional::date(scope, positional_parameters),
            Bif::DateAndTime => eval_bif::positional::date_and_time(scope, positional_parameters),
            Bif::DayOfWeek => Ok(not_implemented!("Bif::DayOfWeek")),
            Bif::DayOfYear => Ok(not_implemented!("Bif::DayOfYear")),
            Bif::Decimal => eval_bif::positional::decimal(scope, positional_parameters),
            Bif::DistinctValues => eval_bif::positional::distinct_values(scope, positional_parameters),
            Bif::Duration => eval_bif::positional::duration(scope, positional_parameters),
            Bif::During => Ok(not_implemented!("Bif::During")),
            Bif::EndsWith => eval_bif::positional::ends_with(scope, positional_parameters),
            Bif::Even => eval_bif::positional::even(scope, positional_parameters),
            Bif::Exp => eval_bif::positional::exp(scope, positional_parameters),
            Bif::FinishedBy => Ok(not_implemented!("Bif::FinishedBy")),
            Bif::Finishes => Ok(not_implemented!("Bif::Finishes")),
            Bif::Flatten => eval_bif::positional::flatten(scope, positional_parameters),
            Bif::Floor => eval_bif::positional::floor(scope, positional_parameters),
            Bif::GetEntries => Ok(not_implemented!("Bif::GetEntries")),
            Bif::GetValue => Ok(not_implemented!("Bif::GetValue")),
            Bif::Includes => Ok(not_implemented!("Bif::Includes")),
            Bif::IndexOf => eval_bif::positional::index_of(scope, positional_parameters),
            Bif::InsertBefore => eval_bif::positional::insert_before(scope, positional_parameters),
            Bif::Is => Ok(not_implemented!("Bif::Is")),
            Bif::ListContains => eval_bif::positional::list_contains(scope, positional_parameters),
            Bif::Log => eval_bif::positional::log(scope, positional_parameters),
            Bif::LoweCase => eval_bif::positional::lower_case(scope, positional_parameters),
            Bif::Matches => eval_bif::positional::matches(scope, positional_parameters),
            Bif::Max => eval_bif::positional::max(scope, positional_parameters),
            Bif::Mean => eval_bif::positional::mean(scope, positional_parameters),
            Bif::Meets => Ok(not_implemented!("Bif::Meets")),
            Bif::Median => eval_bif::positional::median(scope, positional_parameters),
            Bif::MetBy => Ok(not_implemented!("Bif::MetBy")),
            Bif::Min => eval_bif::positional::min(scope, positional_parameters),
            Bif::Mode => eval_bif::positional::mode(scope, positional_parameters),
            Bif::Modulo => eval_bif::positional::modulo(scope, positional_parameters),
            Bif::MonthOfYear => Ok(not_implemented!("Bif::MonthOfYear")),
            Bif::Not => eval_bif::positional::not(scope, positional_parameters),
            Bif::Number => eval_bif::positional::number(scope, positional_parameters),
            Bif::Odd => eval_bif::positional::odd(scope, positional_parameters),
            Bif::OverlapsAfter => Ok(not_implemented!("Bif::OverlapsAfter")),
            Bif::OverlapsBefore => Ok(not_implemented!("Bif::OverlapsBefore")),
            Bif::Product => Ok(not_implemented!("Bif::Product")),
            Bif::Remove => eval_bif::positional::remove(scope, positional_parameters),
            Bif::Replace => eval_bif::positional::replace(scope, positional_parameters),
            Bif::Reverse => eval_bif::positional::reverse(scope, positional_parameters),
            Bif::Sort => Ok(not_implemented!("Bif::Sort")),
            Bif::Split => eval_bif::positional::split(scope, positional_parameters),
            Bif::Sqrt => eval_bif::positional::sqrt(scope, positional_parameters),
            Bif::StartedBy => Ok(not_implemented!("Bif::StartedBy")),
            Bif::Starts => Ok(not_implemented!("Bif::Starts")),
            Bif::StartsWith => eval_bif::positional::starts_with(scope, positional_parameters),
            Bif::Stddev => eval_bif::positional::stddev(scope, positional_parameters),
            Bif::String => eval_bif::positional::string(scope, positional_parameters),
            Bif::StringLength => eval_bif::positional::string_length(scope, positional_parameters),
            Bif::Sublist => eval_bif::positional::sublist(scope, positional_parameters),
            Bif::Substring => eval_bif::positional::substring(scope, positional_parameters),
            Bif::SubstringAfter => eval_bif::positional::substring_after(scope, positional_parameters),
            Bif::SubstringBefore => eval_bif::positional::substring_before(scope, positional_parameters),
            Bif::Sum => eval_bif::positional::sum(scope, positional_parameters),
            Bif::Time => eval_bif::positional::time(scope, positional_parameters),
            Bif::Union => eval_bif::positional::union(scope, positional_parameters),
            Bif::UpperCase => eval_bif::positional::upper_case(scope, positional_parameters),
            Bif::WeekOfYear => Ok(not_implemented!("Bif::WeekOfYear")),
            Bif::YearsAndMonthsDuration => eval_bif::positional::years_and_months_duration(scope, positional_parameters),
          },
          Value::FunctionDefinition(parameters, body) => eval_function_definition_with_positional_parameters(scope, positional_parameters, &parameters, &body),
          other => {
            println!("{:?}", other);
            trace_invalid_value_type!("eval_function_invocation_1");
            Ok(VALUE_NULL)
          }
        }
      }
      named_parameters_node @ AstNode::NamedParameters(_) => {
        let l_value = eval(scope, left)?;
        let named_parameters = &eval(scope, named_parameters_node)?;
        match l_value {
          Value::BuiltInFunction(bif) => match bif {
            Bif::Abs => eval_bif::named::abs(named_parameters),
            Bif::After => Ok(not_implemented!("")),
            Bif::All => eval_bif::named::all(named_parameters),
            Bif::Any => eval_bif::named::any(named_parameters),
            Bif::Append => Ok(not_implemented!("")),
            Bif::Before => Ok(not_implemented!("")),
            Bif::Ceiling => eval_bif::named::ceiling(named_parameters),
            Bif::Coincides => Ok(not_implemented!("")),
            Bif::Concatenate => Ok(VALUE_NULL),
            Bif::Contains => eval_bif::named::contains(named_parameters),
            Bif::Count => eval_bif::named::count(named_parameters),
            Bif::Date => eval_bif::named::date(named_parameters),
            Bif::DateAndTime => eval_bif::named::date_and_time(named_parameters),
            Bif::DayOfWeek => Ok(not_implemented!("")),
            Bif::DayOfYear => Ok(not_implemented!("")),
            Bif::Decimal => eval_bif::named::decimal(named_parameters),
            Bif::DistinctValues => Ok(not_implemented!("")),
            Bif::Duration => eval_bif::named::duration(named_parameters),
            Bif::During => Ok(not_implemented!("")),
            Bif::EndsWith => eval_bif::named::ends_with(named_parameters),
            Bif::Even => eval_bif::named::even(named_parameters),
            Bif::Exp => eval_bif::named::exp(named_parameters),
            Bif::FinishedBy => Ok(not_implemented!("")),
            Bif::Finishes => Ok(not_implemented!("")),
            Bif::Flatten => Ok(not_implemented!("")),
            Bif::Floor => eval_bif::named::floor(named_parameters),
            Bif::GetEntries => Ok(not_implemented!("")),
            Bif::GetValue => Ok(not_implemented!("")),
            Bif::Includes => Ok(not_implemented!("")),
            Bif::IndexOf => Ok(not_implemented!("")),
            Bif::InsertBefore => Ok(not_implemented!("")),
            Bif::Is => Ok(not_implemented!("")),
            Bif::ListContains => Ok(not_implemented!("")),
            Bif::Log => eval_bif::named::log(named_parameters),
            Bif::LoweCase => eval_bif::named::lower_case(named_parameters),
            Bif::Matches => Ok(not_implemented!("")),
            Bif::Max => eval_bif::named::max(named_parameters),
            Bif::Mean => eval_bif::named::mean(named_parameters),
            Bif::Meets => Ok(not_implemented!("")),
            Bif::Median => eval_bif::named::median(named_parameters),
            Bif::MetBy => Ok(not_implemented!("")),
            Bif::Min => eval_bif::named::min(named_parameters),
            Bif::Mode => eval_bif::named::mode(named_parameters),
            Bif::Modulo => eval_bif::named::modulo(named_parameters),
            Bif::MonthOfYear => Ok(not_implemented!("")),
            Bif::Not => eval_bif::named::not(named_parameters),
            Bif::Number => eval_bif::named::number(named_parameters),
            Bif::Odd => eval_bif::named::odd(named_parameters),
            Bif::OverlapsAfter => Ok(not_implemented!("")),
            Bif::OverlapsBefore => Ok(not_implemented!("")),
            Bif::Product => Ok(not_implemented!("")),
            Bif::Remove => eval_bif::named::remove(named_parameters),
            Bif::Replace => eval_bif::named::replace(named_parameters),
            Bif::Reverse => Ok(not_implemented!("")),
            Bif::Sort => Ok(not_implemented!("")),
            Bif::Split => eval_bif::named::split(named_parameters),
            Bif::Sqrt => eval_bif::named::sqrt(named_parameters),
            Bif::StartedBy => Ok(not_implemented!("")),
            Bif::Starts => Ok(not_implemented!("")),
            Bif::StartsWith => eval_bif::named::starts_with(named_parameters),
            Bif::Stddev => eval_bif::named::stddev(named_parameters),
            Bif::String => eval_bif::named::string(named_parameters),
            Bif::StringLength => eval_bif::named::string_length(named_parameters),
            Bif::Sublist => Ok(not_implemented!("")),
            Bif::Substring => eval_bif::named::substring(named_parameters),
            Bif::SubstringAfter => eval_bif::named::substring_after(named_parameters),
            Bif::SubstringBefore => eval_bif::named::substring_before(named_parameters),
            Bif::Sum => Ok(VALUE_NULL),
            Bif::Time => eval_bif::named::time(named_parameters),
            Bif::Union => Ok(not_implemented!("")),
            Bif::UpperCase => eval_bif::named::upper_case(named_parameters),
            Bif::WeekOfYear => Ok(not_implemented!("")),
            Bif::YearsAndMonthsDuration => eval_bif::named::years_and_months_duration(named_parameters),
          },
          Value::FunctionDefinition(parameters, body) => eval_function_definition_with_named_parameters(scope, named_parameters, &parameters, &body),
          _ => {
            trace_invalid_value_type!("eval_function_invocation_2");
            Ok(VALUE_NULL_WITH_TRACE)
          }
        }
      }
      _ => {
        trace_invalid_value_type!("eval_function_invocation_3");
        Ok(VALUE_NULL)
      }
    }
  }

  fn eval_function_definition_argument(
    formal_parameter_name: &Name,
    formal_parameter_type: &FeelType,
    argument_value: &Value,
    argument_context: &mut FeelContext,
  ) {
    if argument_value.type_of().is_conformant(formal_parameter_type) {
      if let Ok(derived_value) = formal_parameter_type.retrieve_value_with_type_checking(argument_value) {
        argument_context.set_entry(formal_parameter_name, derived_value);
      } else {
        //FIXME replace println with something else
        println!("DERIVED VALUE IS NULL for value: {:?} and type {:?}", argument_value, formal_parameter_type);
      }
    } else {
      //FIXME replace println with something else
      println!("NON_CONFORMANT: LEFT={:?} RIGHT={:?}", argument_value.type_of(), formal_parameter_type)
    }
  }

  /// Evaluates function definition with named parameters.
  fn eval_function_definition_with_named_parameters(
    scope: &Scope,
    named_parameters: &Value,
    formal_parameters: &[(Name, FeelType)],
    body: &FunctionBody,
  ) -> Result<Value> {
    let mut arguments_context = FeelContext::default();
    if let Value::NamedParameters(map) = named_parameters {
      for (formal_parameter_name, formal_parameter_type) in formal_parameters {
        if let Some(argument_value) = map.get(formal_parameter_name) {
          eval_function_definition_argument(formal_parameter_name, formal_parameter_type, argument_value, &mut arguments_context);
        } else {
          return Ok(VALUE_NULL_WITH_TRACE /*(invalid number of positional arguments)*/);
        }
      }
    }
    eval_function_definition_instance(scope, &arguments_context, body)
  }

  /// Evaluates function definition with positional parameters.
  fn eval_function_definition_with_positional_parameters(
    scope: &Scope,
    positional_parameters: &[AstNode],
    formal_parameters: &[(Name, FeelType)],
    body: &FunctionBody,
  ) -> Result<Value> {
    let mut positional_arguments = vec![];
    for positional_parameter in positional_parameters {
      positional_arguments.push(eval(scope, positional_parameter)?);
    }
    let mut arguments_context = FeelContext::default();
    for (i, (formal_parameter_name, formal_parameter_type)) in formal_parameters.iter().enumerate() {
      if let Some(argument_value) = positional_arguments.get(i) {
        eval_function_definition_argument(formal_parameter_name, formal_parameter_type, argument_value, &mut arguments_context);
      } else {
        return Ok(VALUE_NULL_WITH_TRACE /* (invalid number of positional arguments) */);
      }
    }
    eval_function_definition_instance(scope, &arguments_context, body)
  }

  /// Evaluates function definition with arguments passed as a context.
  pub fn eval_function_definition_instance(scope: &Scope, arguments_context: &FeelContext, body: &FunctionBody) -> Result<Value> {
    scope.push(arguments_context.clone());
    let result = match body {
      // FunctionBody::Context(context) => context.evaluate(&scope),
      // FunctionBody::DecisionTable(decision_table) => decision_table.evaluate(&scope),
      FunctionBody::LiteralExpression(body_node) => eval(&scope, body_node),
      other => Ok(null_with_trace!(format!("unhandled function body: {:?}", other))),
    };
    scope.pop();
    if let Ok(value) = result {
      //FIXME below function result conformance
      //if value.type_of().is_conformant(result_type) {
      return Ok(value);
      //}
    }
    Ok(VALUE_NULL_WITH_TRACE)
  }

  fn eval_list(scope: &Scope, items: &[AstNode]) -> Result<Value> {
    let mut values = vec![];
    for item in items {
      let value = eval(scope, item)?;
      values.push(value)
    }
    Ok(Value::List(Values::new(values)))
  }

  fn eval_list_type(_scope: &Scope, feel_type: &FeelType) -> Result<Value> {
    Ok(Value::BuiltInType(FeelType::List(Box::new(feel_type.clone()))))
  }

  /// Evaluates the type of the context.
  ///
  /// # Arguments
  ///
  /// scope - evaluation scope,
  /// items - collection of tuples name, type.
  ///
  fn eval_context_type(scope: &Scope, lhs: &[AstNode]) -> Result<Value> {
    let mut entries = BTreeMap::new();
    for context_type_entry in lhs {
      if let AstNode::ContextTypeEntry(a, feel_type) = context_type_entry {
        if let Value::ContextEntryName(name) = eval(scope, a)? {
          entries.insert(name.clone(), feel_type.clone());
        }
      }
    }
    Ok(Value::BuiltInType(FeelType::Context(entries)))
  }

  // fn eval_negated_comma_list(scope: &Scope, items: &[AstNode]) -> Result<Value> {
  //   let mut values = vec![];
  //   for item in items {
  //     let value = eval(scope, item)?;
  //     values.push(value)
  //   }
  //   Ok(Value::NegatedCommaList(values))
  // }

  /// Evaluates `instance of` operator.
  fn eval_instance_of(scope: &Scope, lhs: &AstNode, rhs: &FeelType) -> Result<Value> {
    let lhv = eval(scope, lhs)?;
    match lhv {
      Value::Number(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        FeelType::Number => Ok(VALUE_TRUE),
        _ => Ok(VALUE_FALSE),
      },
      Value::String(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        FeelType::String => Ok(VALUE_TRUE),
        _ => Ok(VALUE_FALSE),
      },
      Value::Boolean(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        FeelType::Boolean => Ok(VALUE_TRUE),
        _ => Ok(VALUE_FALSE),
      },
      Value::Date(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        FeelType::Date => Ok(VALUE_TRUE),
        _ => Ok(VALUE_FALSE),
      },
      Value::DateTime(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        FeelType::DateTime => Ok(VALUE_TRUE),
        _ => Ok(VALUE_FALSE),
      },
      Value::Time(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        FeelType::Time => Ok(VALUE_TRUE),
        _ => Ok(VALUE_FALSE),
      },
      Value::YearsAndMonthsDuration(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        FeelType::YearsAndMonthsDuration => Ok(VALUE_TRUE),
        _ => Ok(VALUE_FALSE),
      },
      Value::DaysAndTimeDuration(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        FeelType::DaysAndTimeDuration => Ok(VALUE_TRUE),
        _ => Ok(VALUE_FALSE),
      },
      Value::Null(_) => match rhs {
        FeelType::Null => Ok(VALUE_TRUE),
        _ => Ok(VALUE_FALSE),
      },
      value @ Value::Range(_, _, _, _) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        expected => Ok(Value::Boolean(value.type_of() == *expected)),
      },
      value @ Value::List(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        expected => Ok(Value::Boolean(value.type_of() == *expected)),
      },
      value @ Value::Context(_) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        expected => Ok(Value::Boolean(value.type_of() == *expected)),
      },
      value @ Value::FunctionDefinition(_, _) => match rhs {
        FeelType::Any => Ok(VALUE_TRUE),
        expected => Ok(Value::Boolean(value.type_of() == *expected)),
      },
      other => Ok(null_with_trace!("eval_instance_of: unhandled value type encountered: {:?}", other)),
    }
  }

  /// Evaluates the context from its entries.
  fn eval_context(scope: &Scope, items: &[AstNode]) -> Result<Value> {
    let mut evaluated_ctx = FeelContext::default();
    scope.push(FeelContext::default()); // prepare special context for already evaluated entries
    for item in items {
      if let Value::ContextEntry(name, value) = eval(scope, item)? {
        evaluated_ctx.set_entry(&name, (*value).clone());
        scope.set_entry(&name, *value); // add newly evaluated entry to special context
      }
    }
    scope.pop(); // remove special context
    Ok(Value::Context(evaluated_ctx))
  }

  fn eval_context_entry(scope: &Scope, left: &AstNode, right: &AstNode) -> Result<Value> {
    let key = eval(scope, left)?;
    let value = eval(scope, right)?;
    match key {
      Value::String(s) => Ok(Value::ContextEntry(Name::from(s), Box::new(value))),
      Value::ContextEntryKey(name) => Ok(Value::ContextEntry(name, Box::new(value))),
      other => Ok(other),
    }
  }

  fn eval_context_entry_key(name: &Name) -> Value {
    Value::ContextEntryKey(name.clone())
  }
}
