/*
 * DMNTK - Decision Model and Notation Toolkit
 *
 * Common definitions.
 *
 * Copyright 2018-2021 Dariusz Depta Engos Software <dariusz.depta@engos.software>
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

//! Configuration.

use self::errors::*;
use crate::Result;
use serde_derive::Deserialize;

/// Configuration.
#[derive(Debug, Default, Deserialize)]
pub struct Configuration {
  pub server: ServerConfiguration,
}

/// Server configuration.
#[derive(Debug, Deserialize)]
pub struct ServerConfiguration {
  pub host: String,
  pub port: u16,
}

impl Default for ServerConfiguration {
  fn default() -> Self {
    Self {
      host: "0.0.0.0".to_string(),
      port: 17000,
    }
  }
}

const CONFIGURATION_FILE: &str = include_str!("config.yml");

/// Returns configuration.
pub fn get_configuration() -> Result<Configuration> {
  serde_yaml::from_str(CONFIGURATION_FILE).map_err(|e| parse_error(e.to_string()))
}

/// Definitions of errors reported by configuration module.
mod errors {
  use crate::DmntkError;

  /// Server errors.
  #[derive(Debug, PartialEq)]
  enum ConfigurationError {
    ParseError(String),
  }

  impl From<ConfigurationError> for DmntkError {
    fn from(e: ConfigurationError) -> Self {
      DmntkError::new("ConfigurationError", &format!("{}", e))
    }
  }

  impl std::fmt::Display for ConfigurationError {
    fn fmt(&self, f: &mut std::fmt::Formatter<'_>) -> std::fmt::Result {
      match self {
        ConfigurationError::ParseError(reason) => {
          write!(f, "parsing configuration file failed with reason: {}", reason)
        }
      }
    }
  }

  pub fn parse_error(reason: String) -> DmntkError {
    ConfigurationError::ParseError(reason).into()
  }
}

#[cfg(test)]
mod tests {
  use crate::config::{get_configuration, ServerConfiguration};

  #[test]
  fn test_default_server_configuration() {
    let server_configuration = ServerConfiguration::default();
    assert_eq!("0.0.0.0", server_configuration.host);
    assert_eq!(17000, server_configuration.port);
  }

  #[test]
  fn test_configuration_file() {
    let configuration = get_configuration().unwrap();
    assert_eq!("127.0.0.1", configuration.server.host);
    assert_eq!(12000, configuration.server.port);
  }
}
