#![allow(dead_code)]
mod binding;

use std::ops::{Add, Mul, Sub};

pub use binding::DtStraightPathFlags;

use binding::*;
use thiserror::Error;

pub type Vector = DtVector;

impl Vector {
    pub fn from_xyz(x: f32, y: f32, z: f32) -> Self {
        Self { x, y, z }
    }

    pub fn from_yzx(y: f32, z: f32, x: f32) -> Self {
        Self { x, y, z }
    }

    pub fn dot(&self, other: &Self) -> f32 {
        self.x * other.x + self.y * other.y + self.z * other.z
    }
}

impl Add for Vector {
    type Output = Self;

    fn add(self, other: Self) -> Self {
        Self {
            x: self.x + other.x,
            y: self.y + other.y,
            z: self.z + other.z,
        }
    }
}

impl Sub for Vector {
    type Output = Self;

    fn sub(self, other: Self) -> Self {
        Self {
            x: self.x - other.x,
            y: self.y - other.y,
            z: self.z - other.z,
        }
    }
}

impl Mul<f32> for Vector {
    type Output = Self;

    fn mul(self, scalar: f32) -> Self {
        Self {
            x: self.x * scalar,
            y: self.y * scalar,
            z: self.z * scalar,
        }
    }
}

pub type NavMeshParams = DtNavMeshParams;

pub type PolyRef = DtPolyRef;

pub type TileRef = DtTileRef;

#[derive(Error, Debug)]
pub enum DivertError {
    #[error("detour internal status failure `{0:?}")]
    Failure(DtStatus),
    #[error("detour unexpected null ptr failure")]
    NullPtr(),
}

pub type DivertResult<T> = std::result::Result<T, DivertError>;

pub struct NavMesh<'a>(&'a mut DtNavMesh);

impl<'a> NavMesh<'a> {
    pub fn new(nav_mesh_params: &NavMeshParams) -> DivertResult<Self> {
        let dt_nav_mesh = unsafe {
            match dtNavMesh_alloc().as_mut() {
                Some(mesh) => mesh,
                None => return Err(DivertError::NullPtr()),
            }
        };

        let init_status = unsafe { dtNavMesh_init(dt_nav_mesh, nav_mesh_params) };
        if init_status.is_failed() {
            return Err(DivertError::Failure(init_status));
        }

        Ok(Self(dt_nav_mesh))
    }

    pub fn add_tile(&mut self, input_data: Vec<u8>) -> DivertResult<TileRef> {
        let mut boxed_slice = input_data.into_boxed_slice();
        let data = boxed_slice.as_mut_ptr();
        let data_size = boxed_slice.len();

        let mut tile_ref = TileRef::default();
        let add_tile_status = unsafe {
            dtNavMesh_addTile(
                self.0,
                data,
                data_size as i32,
                1,
                TileRef::default(),
                &mut tile_ref,
            )
        };

        if add_tile_status.is_failed() {
            return Err(DivertError::Failure(add_tile_status));
        }

        std::mem::forget(boxed_slice);
        Ok(tile_ref)
    }
}

impl<'a> Drop for NavMesh<'a> {
    fn drop(&mut self) {
        unsafe { dtNavMesh_free(self.0) }
    }
}

impl AsRef<DtNavMesh> for NavMesh<'_> {
    fn as_ref(&self) -> &DtNavMesh {
        self.0
    }
}

pub struct QueryFilter<'a>(&'a mut DtQueryFilter);

impl<'a> QueryFilter<'a> {
    pub fn new() -> DivertResult<Self> {
        let dt_query_filter = unsafe {
            match dtQueryFilter_alloc().as_mut() {
                Some(query_filter) => query_filter,
                None => return Err(DivertError::NullPtr()),
            }
        };

        Ok(Self(dt_query_filter))
    }

    pub fn set_include_flags(&mut self, include_flags: u16) {
        unsafe {
            dtQueryFilter_setIncludeFlags(self.0, include_flags);
        }
    }

    pub fn get_include_flags(&mut self) -> u16 {
        unsafe { dtQueryFilter_getIncludeFlags(self.0) }
    }

    pub fn set_exclude_flags(&mut self, exclude_flags: u16) {
        unsafe {
            dtQueryFilter_setExcludeFlags(self.0, exclude_flags);
        }
    }

    pub fn get_exclude_flags(&mut self) -> u16 {
        unsafe { dtQueryFilter_getExcludeFlags(self.0) }
    }
}

impl AsRef<DtQueryFilter> for QueryFilter<'_> {
    fn as_ref(&self) -> &DtQueryFilter {
        self.0
    }
}

pub struct NavMeshQuery<'a>(&'a mut DtNavMeshQuery);

impl<'a> NavMeshQuery<'a> {
    pub fn new(nav_mesh: &NavMesh, max_nodes: i32) -> DivertResult<Self> {
        let dt_nav_mesh_query = unsafe {
            match dtNavMeshQuery_alloc().as_mut() {
                Some(nav_mesh_query) => nav_mesh_query,
                None => return Err(DivertError::NullPtr()),
            }
        };

        let init_status =
            unsafe { dtNavMeshQuery_init(dt_nav_mesh_query, nav_mesh.as_ref(), max_nodes) };
        if init_status.is_failed() {
            return Err(DivertError::Failure(init_status));
        }

        Ok(Self(dt_nav_mesh_query))
    }

    pub fn get_poly_height(&mut self, poly_ref: PolyRef, position: &DtVector) -> DivertResult<f32> {
        let mut height: f32 = 0.0;

        let get_poly_height_status =
            unsafe { dtNavMeshQuery_getPolyHeight(self.0, poly_ref, position, &mut height) };

        if get_poly_height_status.is_failed() {
            return Err(DivertError::Failure(get_poly_height_status));
        }

        Ok(height)
    }

    pub fn find_nearest_poly(
        &mut self,
        center: &Vector,
        extents: &Vector,
        filter: &QueryFilter,
    ) -> DivertResult<(PolyRef, Vector)> {
        let mut closest_point = Vector::default();
        let mut nearest_ref = PolyRef::default();

        let nearest_status = unsafe {
            dtNavMeshQuery_findNearestPoly(
                self.0,
                center,
                extents,
                filter.as_ref(),
                &mut nearest_ref,
                &mut closest_point,
            )
        };

        if nearest_status.is_failed() {
            return Err(DivertError::Failure(nearest_status));
        }

        Ok((nearest_ref, closest_point))
    }

    pub fn closest_point_on_poly(
        &mut self,
        poly_ref: PolyRef,
        position: &Vector,
    ) -> DivertResult<(Vector, bool)> {
        let mut closest_point = Vector::default();
        let mut position_over_poly = false;

        let nearest_status = unsafe {
            dtNavMeshQuery_closestPointOnPoly(
                self.0,
                poly_ref,
                position,
                &mut closest_point,
                &mut position_over_poly,
            )
        };

        if nearest_status.is_failed() {
            return Err(DivertError::Failure(nearest_status));
        }

        Ok((closest_point, position_over_poly))
    }

    pub fn closest_point_on_poly_boundary(
        &mut self,
        poly_ref: PolyRef,
        position: &Vector,
    ) -> DivertResult<Vector> {
        let mut closest_point = Vector::default();

        let dt_result = unsafe {
            dtNavMeshQuery_closestPointOnPolyBoundary(
                self.0,
                poly_ref,
                position,
                &mut closest_point,
            )
        };

        if dt_result.is_failed() {
            return Err(DivertError::Failure(dt_result));
        }

        Ok(closest_point)
    }

    pub fn find_path(
        &mut self,
        start_ref: PolyRef,
        end_ref: PolyRef,
        start_pos: &Vector,
        end_pos: &Vector,
        filter: &QueryFilter,
        max_path: i32,
    ) -> DivertResult<Vec<PolyRef>> {
        let mut path_count = 0;
        let mut path: Vec<PolyRef> = Vec::with_capacity(max_path.try_into().unwrap());

        let find_path_status = unsafe {
            dtNavMeshQuery_findPath(
                self.0,
                start_ref,
                end_ref,
                start_pos,
                end_pos,
                filter.as_ref(),
                path.as_mut_ptr(),
                &mut path_count,
                max_path,
            )
        };

        unsafe {
            path.set_len(path_count as usize);
        }

        if find_path_status.is_failed() {
            return Err(DivertError::Failure(find_path_status));
        }

        Ok(path)
    }

    pub fn find_straight_path(
        &mut self,
        start_pos: &Vector,
        end_pos: &Vector,
        poly_path: &[PolyRef],
        max_path: i32,
        options: i32,
    ) -> DivertResult<Vec<(Vector, DtStraightPathFlags, PolyRef)>> {
        let mut straight_path_count = 0;
        let mut straight_path_points: Vec<DtVector> =
            Vec::with_capacity(max_path.try_into().unwrap());
        let mut straight_path_flags: Vec<DtStraightPathFlags> =
            Vec::with_capacity(max_path.try_into().unwrap());
        let mut straight_path_polys: Vec<PolyRef> =
            Vec::with_capacity(max_path.try_into().unwrap());

        let find_path_status = unsafe {
            dtNavMeshQuery_findStraightPath(
                self.0,
                start_pos,
                end_pos,
                poly_path.as_ptr(),
                poly_path.len().try_into().unwrap(),
                straight_path_points.as_mut_ptr(),
                straight_path_flags.as_mut_ptr(),
                straight_path_polys.as_mut_ptr(),
                &mut straight_path_count,
                max_path,
                options,
            )
        };

        unsafe {
            let path_count = straight_path_count as usize;
            straight_path_points.set_len(path_count);
            straight_path_flags.set_len(path_count);
            straight_path_polys.set_len(path_count);
        }

        if find_path_status.is_failed() {
            return Err(DivertError::Failure(find_path_status));
        }

        let path_result = straight_path_points
            .into_iter()
            .zip(straight_path_flags.into_iter())
            .zip(straight_path_polys.into_iter())
            .map(|((pos, flags), poly_ref)| (pos, flags, poly_ref))
            .collect();

        Ok(path_result)
    }

    pub fn move_along_surface(
        &mut self,
        start_ref: PolyRef,
        start_pos: &Vector,
        end_pos: &Vector,
        filter: &QueryFilter,
        max_visit: i32,
    ) -> DivertResult<(Vector, Vec<PolyRef>)> {
        let mut visited_count = 0;
        let mut visited: Vec<PolyRef> = Vec::with_capacity(max_visit.try_into().unwrap());
        let mut result_pos = Vector::default();

        let move_along_surface_result = unsafe {
            dtNavMeshQuery_moveAlongSurface(
                self.0,
                start_ref,
                start_pos,
                end_pos,
                filter.as_ref(),
                &mut result_pos,
                visited.as_mut_ptr(),
                &mut visited_count,
                max_visit,
            )
        };

        if move_along_surface_result.is_failed() {
            return Err(DivertError::Failure(move_along_surface_result));
        }

        unsafe {
            visited.set_len(visited_count as usize);
        }

        Ok((result_pos, visited))
    }
}

impl<'a> Drop for NavMeshQuery<'a> {
    fn drop(&mut self) {
        unsafe { dtNavMeshQuery_free(self.0) }
    }
}

impl<'a> AsRef<DtNavMeshQuery> for NavMeshQuery<'a> {
    fn as_ref(&self) -> &DtNavMeshQuery {
        self.0
    }
}

#[cfg(test)]
mod tests {

    use crate::QueryFilter;

    #[test]
    fn test_query_filter() {
        let filter = QueryFilter::new();
        assert!(filter.is_ok());

        let mut filter = filter.unwrap();

        filter.set_include_flags(1);
        assert_eq!(filter.get_include_flags(), 1);

        filter.set_exclude_flags(1);
        assert_eq!(filter.get_exclude_flags(), 1);
    }
}
