//! Internal / private prelude

#![allow(unused_imports)]

pub(in crate) use ::ffi_sdk::ffi_utils::{c_slice, char_p, repr_c, NonOpaque};
pub(in crate) use ::safer_ffi::prelude::Out;
pub(in crate) use ::std::{
    any::Any,
    borrow::Borrow,
    collections::HashMap,
    convert::{TryFrom, TryInto},
    ops::Not,
    os::raw::c_void,
    path::{Path, PathBuf},
    pin::Pin,
    sync::Arc,
};
pub(in crate) mod arc {
    pub use ::std::sync::Weak;
}
pub(in crate) use ::to_method::To;

pub(in crate) use crate::{
    ditto::Ditto,
    error::{DittoError, ErrorKind},
    prelude::*,
    store::Store,
    utils::{extension_traits::*, Str},
};

pub(in crate) mod marker {
    pub(in crate) use crate::utils::InvariantLifetimeMarker as InvariantLifetime;
}

pub(in crate) trait IntoRustResult: Sized {
    type Ok;
    fn into_rust_result(self) -> Result<Self::Ok, ::core::num::NonZeroI32>;
    fn ok_or(self, error_kind: ErrorKind) -> Result<Self::Ok, DittoError> {
        self.into_rust_result()
            .map_err(|_| DittoError::from_ffi(error_kind))
    }
    fn ok(self) -> Result<Self::Ok, DittoError> {
        self.ok_or(ErrorKind::Internal)
    }
}

impl<T> IntoRustResult for repr_c::Result_Box<T>
where
    repr_c::Box<T>: ::ffi_sdk::ffi_utils::FfiDrop,
{
    type Ok = repr_c::Box<T>;
    fn into_rust_result(self) -> Result<Self::Ok, ::core::num::NonZeroI32> {
        if let Some(err_code) = ::core::num::NonZeroI32::new(self.status_code) {
            Err(err_code)
        } else {
            Ok(self.ok_value.unwrap())
        }
    }
}

impl<T: NonOpaque> IntoRustResult for repr_c::Result_Vec<T> {
    type Ok = repr_c::Vec<T>;
    fn into_rust_result(self) -> Result<Self::Ok, ::core::num::NonZeroI32> {
        if let Some(err_code) = ::core::num::NonZeroI32::new(self.status_code) {
            Err(err_code)
        } else {
            Ok(self.c_vec.unwrap())
        }
    }
}

impl<T: NonOpaque> IntoRustResult for repr_c::Result_BoxedSlice<T> {
    type Ok = c_slice::Box<T>;
    fn into_rust_result(self) -> Result<Self::Ok, ::core::num::NonZeroI32> {
        if let Some(err_code) = ::core::num::NonZeroI32::new(self.status_code) {
            Err(err_code)
        } else {
            Ok(self.c_slice.unwrap())
        }
    }
}

impl<T: NonOpaque> IntoRustResult for repr_c::Result<T> {
    type Ok = T;
    fn into_rust_result(self) -> Result<T, ::core::num::NonZeroI32> {
        if let Some(err_code) = ::core::num::NonZeroI32::new(self.status_code) {
            Err(err_code)
        } else {
            Ok(self.ok_value)
        }
    }
}
