#[macro_use]
extern crate serde_json;

mod common;

// `insert` and `insert_with_strategy` are deprecated but we still want to test
// them while we support them so we put all of the tests related to them in this
// file so that it will be easy to remove when we get to the 2.0.0 release
#[allow(deprecated)]
mod deprecated {
    use super::common::*;

    #[test]
    fn insert_overwrites_existing_content_by_default() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();
        let custom_id = DocumentId::new(&"custom_id".to_string()).unwrap();
        let id1 = collection
            .insert(json!({"some": "value"}), Some(&custom_id), false)
            .unwrap();
        let id2 = collection
            .insert(json!({"another": "thing"}), Some(&custom_id), false)
            .unwrap();
        let doc = collection.find_by_id(custom_id).exec().unwrap();
        assert_eq!(id1, doc.id());
        assert_eq!(id2, doc.id());
        assert_eq!(doc.get::<String>("another").unwrap(), "thing");
        // Expect key-value pair under "some" key to not exist in document after
        // being overwritten, which leads to an error result in the `get` call.
        assert!(doc.get::<String>("some").is_err());
    }

    #[test]
    fn insert_merges_with_existing_content_if_specified_in_options() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();
        let custom_id = DocumentId::new(&"custom_id".to_string()).unwrap();
        //@ditto/snippet-start insert
        let id1 = collection
            .insert(json!({"some": "value"}), Some(&custom_id), false)
            .unwrap();
        //@ditto/snippet-end
        let id2 = collection
            .insert_with_strategy(
                json!({"another": "thing"}),
                Some(&custom_id),
                WriteStrategy::Merge,
            )
            .unwrap();
        let doc = collection.find_by_id(custom_id).exec().unwrap();
        assert_eq!(id1, doc.id());
        assert_eq!(id2, doc.id());
        assert_eq!(doc.get::<String>("another").unwrap(), "thing");
        assert_eq!(doc.get::<String>("some").unwrap(), "value");
    }

    #[test]
    fn inserting_default_data_if_a_document_with_a_matching_id_already_exists_is_a_no_op() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();
        let custom_id = DocumentId::new(&"custom_id".to_string()).unwrap();

        let id1 = collection
            .insert(json!({"some": "value"}), Some(&custom_id), true)
            .unwrap();

        // Immediately try and insert some new default data using both supported
        // methods
        let id2 = collection
            .insert(json!({"different": "default data"}), Some(&custom_id), true)
            .unwrap();
        let id2b = collection
            .insert_with_strategy(
                json!({"other different": "default data"}),
                Some(&custom_id),
                WriteStrategy::InsertDefaultIfAbsent,
            )
            .unwrap();

        // The new default data has been ignored because a document with a matching
        // ID already existed
        let doc = collection.find_by_id(custom_id.clone()).exec().unwrap();
        assert_eq!(doc.get::<String>("some").unwrap(), "value");
        assert!(doc.get::<String>("different").is_err());

        // Perform an insert of a document containing *non* default data
        let id3 = collection
            .insert_with_strategy(json!({"a": "b"}), Some(&custom_id), WriteStrategy::Merge)
            .unwrap();

        // Again try and insert some new default data using both supported methods
        let id4 = collection
            .insert(json!({"different": "default data"}), Some(&custom_id), true)
            .unwrap();
        let id4b = collection
            .insert_with_strategy(
                json!({"other different": "default data"}),
                Some(&custom_id),
                WriteStrategy::InsertDefaultIfAbsent,
            )
            .unwrap();

        let doc = collection.find_by_id(custom_id).exec().unwrap();
        assert_eq!(doc.get::<String>("a").unwrap(), "b");
        assert_eq!(doc.get::<String>("some").unwrap(), "value");
        assert!(doc.get::<String>("different").is_err());
        assert_eq!(id1, doc.id());
        assert_eq!(id2, doc.id());
        assert_eq!(id2b, doc.id());
        assert_eq!(id3, doc.id());
        assert_eq!(id4, doc.id());
        assert_eq!(id4b, doc.id());
    }

    #[test]
    fn inserts_a_document_if_no_preexisting_when_using_appropriate_strategy() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();
        let custom_id = DocumentId::new(&"custom_id".to_string()).unwrap();
        let id1 = collection
            .insert(json!({"some": "value"}), Some(&custom_id), false)
            .unwrap();
        let id2 = collection
            .insert_with_strategy(
                json!({"another": "thing"}),
                Some(&custom_id),
                WriteStrategy::InsertIfAbsent,
            )
            .unwrap();
        let doc = collection.find_by_id(custom_id).exec().unwrap();
        assert_eq!(id1, doc.id());
        assert_eq!(id2, doc.id());
        assert_eq!(doc.get::<String>("some").unwrap(), "value");
        assert!(doc.get::<String>("another").is_err());
    }

    #[test]
    fn insert_with_provided_id() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let content = json!({"hello": "again"});
        let collection = store.collection("test").unwrap();
        let custom_id = DocumentId::new(&"a".to_string()).unwrap();
        let id = collection.insert(content, Some(&custom_id), false).unwrap();
        let doc = collection.find_by_id(custom_id.to_owned()).exec().unwrap();
        assert_eq!(id, custom_id);
        assert_eq!(id, doc.id());
        assert_eq!(doc.get::<String>("hello").unwrap(), "again");
    }

    #[test]
    fn insert_with_id_in_document_content() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();
        let content = json!({"_id": DocumentId::new(&"abc".to_string()).unwrap(), "some": "value"});
        let id = collection.insert(content, None, false).unwrap();
        let expected_id = DocumentId::new(&"abc".to_string()).unwrap();
        assert_eq!(id, expected_id);
        let doc = collection
            .find_by_id(expected_id.to_owned())
            .exec()
            .unwrap();
        assert_eq!(expected_id, doc.id());
        assert_eq!(doc.get::<String>("_id").unwrap(), expected_id.to_string());
    }

    #[test]
    fn insert_with_provided_id_and_id_in_document_content() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();
        let custom_id = DocumentId::new(&"def".to_string()).unwrap();
        let content = json!({"_id": DocumentId::new(&"abc".to_string()).unwrap(), "some": "value"});
        let id = collection.insert(content, Some(&custom_id), false).unwrap();
        assert_eq!(id, custom_id);
        let doc = collection.find_by_id(custom_id.to_owned()).exec().unwrap();
        assert_eq!(custom_id, doc.id());
        assert_eq!(doc.get::<String>("_id").unwrap(), custom_id.to_string());
    }

    #[test]
    fn roundtrip_with_document_ids_of_different_types() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();

        let raw_doc_ids: Vec<serde_json::Value> = vec![
            "boring_old_string".to_string().into(),
            1.into(),
            0.into(),
            123.into(),
            9999.into(),
            false.into(),
            true.into(),
            (0..64)
                .map(|_| rand::random::<u8>())
                .collect::<Vec<u8>>()
                .into(),
            vec!["a", "abc", "z89{{}}@£!fv>?!,[](){{}}000"].into(),
            json!({"a": "b", "__num__@£$%^&{})(|,,./!?": -7123}),
        ];

        raw_doc_ids.iter().for_each(|raw_id| {
            let doc_id = DocumentId::new(raw_id).unwrap();
            let returned_id = collection
                .insert(json!({"hello": "again"}), Some(&doc_id), false)
                .unwrap();
            assert_eq!(doc_id, returned_id);

            let doc = collection.find_by_id(doc_id.clone()).exec().unwrap();
            assert_eq!(doc.id(), doc_id);
        });
    }

    /// Test round trip insertion and deserialization correctly preserves fields
    /// of various types
    #[test]
    fn serialize_insert_roundtrip() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let collection = store.collection("test").unwrap();
        let custom_id = DocumentId::new(&"a".to_string()).unwrap();
        let content = TestType::random_with_id(Some(custom_id.clone()));
        let id = collection
            .insert(content.to_owned(), Some(&custom_id), false)
            .unwrap();
        assert_eq!(id, custom_id);
        let doc = collection.find_by_id(id).exec().unwrap();
        let returned_content = doc.typed::<TestType>().unwrap();
        assert_eq!(content, returned_content);
    }

    #[test]
    fn inserting_invalid_doc_throws_error() {
        let ditto = get_ditto().unwrap();
        let store = ditto.store();
        let content = json!(["hello", "again"]);
        let collection = store.collection("test").unwrap();
        let result = collection.insert(content, None, false);
        assert!(result.is_err());
    }
}
