pub(in crate) trait RefCounted: Clone {
    fn retain(&self) -> Self {
        Clone::clone(self)
    }

    /// This method is added precisely to make `.clone()` calls ambiguous (in
    /// non-`Clone`-generic context). The idea being that in such contexts,
    /// explicit `.retain()` ought to be more readable than `.clone`-ing.
    ///
    /// We disable `rustfmt` to let the "suggestion" comment be on the same
    /// line, since that's the only visible line in the "ambiguous method"
    /// message.
    #[rustfmt::skip]
    fn clone(&self) -> Self { // Use `.retain()` instead.
        Clone::clone(self)
    }
}
impl<T: ?Sized> RefCounted for ::std::rc::Rc<T> {}
impl<T: ?Sized> RefCounted for ::std::sync::Arc<T> {}

/// Polyfills of unstable stdlib functionality.
pub(in crate) mod polyfills {
    use ::std::{
        cell::Cell,
        mem::MaybeUninit,
        sync::{Arc, Weak},
    };

    /// Polyfill of `arc_new_cyclic` https://github.com/rust-lang/rust/issues/75861.
    #[extension(pub trait ArcNewCyclic)]
    impl<T> Arc<T> {
        fn new_cyclic(constructor: impl FnOnce(&Weak<MaybeUninit<T>>) -> T) -> Arc<T> {
            let arc = Arc::new(Cell::new(MaybeUninit::<T>::uninit()));
            let weak = &Arc::downgrade(&arc);
            #[rustfmt::skip]
            let weak = unsafe {
                // Safety:
                //   - same layout,
                //   - note: removing the MaybeUninit is only sound if the weak
                //     isn't upgraded, so we let `constructor` assert that.
                transmute_by_ref::<
                    Weak<Cell<MaybeUninit<T>>>,
                    Weak<MaybeUninit<T>>,
                >(weak)
            };
            let instance: T = constructor(weak);
            arc.set(MaybeUninit::new(instance));
            unsafe { Arc::from_raw(Arc::into_raw(arc).cast()) }
        }
    }

    /// Same as `transmute::<&T, &U>`, but returning a *bounded lifetime*.
    ///
    /// That is, the following compiles when it "shouldn't":
    ///
    /// ```rust, no_run
    /// let unsigned = unsafe {
    ///     let ft = 42_i32;
    ///     let signed = &ft;
    ///     ::core::mem::transmute::<&i32, &u32>(signed)
    /// };
    /// println!("{}", *unsigned); // UB!
    /// ```
    ///
    /// but the following doesn't:
    ///
    /// ```rust, compile_fail
    /// unsafe fn transmute_by_ref<T, U>(it: &'_ T) -> &'_ U {
    ///     ::core::mem::transmute(it)
    /// }
    /// let unsigned = unsafe {
    ///     let ft = 42_i32;
    ///     let signed = &ft;
    ///     transmute_by_ref::<i32, u32>(signed)
    /// };
    /// println!("{}", *unsigned); // Error, `ft` has been dropped ✅
    /// ```
    pub(in crate) unsafe fn transmute_by_ref<T, U>(it: &'_ T) -> &'_ U {
        ::core::mem::transmute(it)
    }
}
