#[macro_use]
extern crate serde_json;

use common::*;
use dittolive_ditto::store::batch::ScopedStore;

mod common;

#[test]
fn exec_with_query_args() {
    env_logger::init();
    let ditto = common::get_ditto().unwrap();
    let store = ditto.store();
    let collection_name = "test";
    let collection = store.collection(collection_name).unwrap();

    // Insert some documents that will match, and some that won't
    collection
        .insert(
            json!({"make": "Honda", "color": "red"}),
            Some(&DocumentId::new(&"honda".to_string()).unwrap()),
            false,
        )
        .unwrap();
    collection
        .insert(
            json!({"make": "Ford", "color": "blue"}),
            Some(&DocumentId::new(&"ford".to_string()).unwrap()),
            false,
        )
        .unwrap();
    collection
        .insert(
            json!({"make": "Aston Martin", "color": "silver"}),
            Some(&DocumentId::new(&"aston_martin".to_string()).unwrap()),
            false,
        )
        .unwrap();
    collection
        .insert(
            json!({"make": "Ferrari", "color": "red"}),
            Some(&DocumentId::new(&"ferrari".to_string()).unwrap()),
            false,
        )
        .unwrap();

    let _ = store.with_batched_write(|mut tx: ScopedStore<'_>| {
        let mut col_tx = tx.collection(collection_name);

        let docs = col_tx
            .find_with_args("color == $args.color", json!({"color": "red"}))
            .exec()
            .unwrap();

        assert_eq!(docs.len(), 2);
        assert_eq!(docs[0].get::<String>("make").unwrap(), "Honda");
        assert_eq!(docs[1].get::<String>("make").unwrap(), "Ferrari");

        tx.commit_changes()
    });
}

#[test]
fn evict_and_remove_with_args() {
    let ditto = common::get_ditto().unwrap();
    let store = ditto.store();
    let collection_name = "test";
    let collection = store.collection(collection_name).unwrap();
    let honda_id = collection
        .insert(json!({"make": "Honda", "color": "red"}), None, false)
        .unwrap();
    collection
        .insert(json!({"make": "Ford", "color": "blue"}), None, false)
        .unwrap();
    let ferrari_id = collection
        .insert(json!({"make": "Ferrari", "color": "red"}), None, false)
        .unwrap();
    let lamborghini_id = collection
        .insert(
            json!({"make": "Lamborghini", "color": "yellow"}),
            None,
            false,
        )
        .unwrap();

    let docs = collection
        .find_with_args(
            "contains([$args.color1, $args.color2], color)",
            json!({"color1": "red", "color2": "yellow"}),
        )
        .exec()
        .unwrap();
    assert_eq!(docs.len(), 3);

    let _ = store.with_batched_write(|mut tx: ScopedStore<'_>| {
        let mut col_tx = tx.collection(collection_name);

        //@ditto/snippet-start remove-query
        let removed_ids = col_tx
            .find_with_args("color == $args.color", json!({"color": "yellow"}))
            .remove()
            .unwrap();
        //@ditto/snippet-end
        assert_eq!(removed_ids.len(), 1);
        assert_eq!(removed_ids[0], lamborghini_id);

        let sort_param = ffi_sdk::COrderByParam {
            query_c_str: c!("make"),
            direction: ffi_sdk::QuerySortDirection::Descending,
        };
        let evicted_ids = col_tx
            .find_with_args("$args.color == color", json!({"color": "red"}))
            .sort(vec![sort_param])
            .evict()
            .unwrap();
        assert_eq!(evicted_ids.len(), 2);
        assert_eq!(evicted_ids[0], honda_id);
        assert_eq!(evicted_ids[1], ferrari_id);

        let docs = col_tx
            .find_with_args(
                "contains([$args.color1, $args.color2], color)",
                json!({"color1": "red", "color2": "yellow"}),
            )
            .exec()
            .unwrap();
        assert_eq!(docs.len(), 0);

        tx.commit_changes()
    });

    let docs = collection
        .find_with_args(
            "contains([$args.color1, $args.color2], color)",
            json!({"color1": "red", "color2": "yellow"}),
        )
        .exec()
        .unwrap();
    assert_eq!(docs.len(), 0);
}
