use_prelude!();
pub mod builder;

use std::{env, mem::ManuallyDrop, sync::RwLock};

use crossbeam_utils::atomic::AtomicCell;

use ffi_sdk::BoxedDitto;
use uuid::Uuid;

use self::builder::DittoBuilder;
use crate::{
    auth::{DittoAuthenticator, ValidityListener},
    error::{DittoError, ErrorKind, LicenseError},
    transport::{
        peers_observer::PeersObserver,
        presence_manager_v2::{PresenceManagerV2Context, V2Presence},
        TransportConfig, Transports,
    },
    utils::prelude::*,
};

/// The entry point for accessing Ditto-related functionality
/// This struct is generally a handle and interface to ditto-functionality
/// which operates in background threads.
pub struct Ditto {
    // Note, we `get_mut.unwrap()` this on `Drop` so make sure to only extract
    // weak references out of this handle.
    //
    // We use a `ManuallyDrop` to get owned access to the `Arc` in `Ditto`'s
    // `Drop` glue (through `ManuallyDrop::take()`).
    // FIXME(Daniel): have a macro do this to reduce the chance of human error.
    fields: ManuallyDrop<Arc<DittoFields>>,
}

impl std::ops::Deref for Ditto {
    type Target = DittoFields;

    #[inline]
    fn deref(&'_ self) -> &'_ DittoFields {
        &*self.fields
    }
}

pub struct DittoFields {
    // FIXME: (Ham & Daniel) - ideally we'd use this in the same way as we do
    // with the `fields` on `Ditto` (only ever extracting weak references)
    ditto: Arc<ffi_sdk::BoxedDitto>,
    ditto_root: Arc<dyn DittoRoot>, // Arc since Identity will need a copy
    auth: Option<DittoAuthenticator>,
    #[allow(dead_code)]
    validity_listener: Option<Arc<ValidityListener>>, // may not need to hang on to this
    store: Store,
    activated: AtomicCell<bool>,
    site_id: SiteId,
    transports: Arc<RwLock<Transports>>,
    presence_manager_v2: Arc<PresenceManagerV2Context>,
}

impl Ditto {
    fn drop_fields(fields: Arc<DittoFields>) {
        // We use this pattern to ensure `self` is not used
        // after `ManuallyDrop::take()`-ing its fields.
        impl Drop for Ditto {
            fn drop(&mut self) {
                // stop all transports
                self.stop_sync();
                // stop all servers and advertisers
                // the idea is to maximize the chance we
                // can get an exclusive lock
                unsafe {
                    ffi_sdk::ditto_shutdown(&self.ditto);
                }
                unsafe {
                    Ditto::drop_fields(ManuallyDrop::take(&mut self.fields));
                }
            }
        }
        ::log::debug!("Dropping Ditto instance");

        // Now with everything shut down try to get a unique access (`mut` or
        // owned). The most likely remaining references are the LiveQueries.
        let mut fields = Arc::try_unwrap(fields)
            .ok()
            .expect("outstanding strong back reference to Ditto");

        if let Some(_witness_of_unicity) = Arc::get_mut(&mut fields.ditto) {
            // Our own strong counts are equal to 1;
            // so we implicitly drop the `BoxedDitto` which will call
            // `ditto_drop(…)`
        } else {
            // We do NOT want to drop the referent (ffi::Ditto)
            // as LiveQueries are still running. The user is responsible for
            // cleaning these LQs up.
            let count = Arc::strong_count(&fields.ditto);
            ::log::debug!("Attempting to drop Ditto with {} live references", count);
        }
    }
}

// Public interface for modifying Transport configuration
impl Ditto {
    /// Start syncing on all configured transports
    /// May panic if misconfigured or license is not yet validated
    pub fn start_sync(&self) {
        self.try_start_sync()
            .expect("Error with License or TransportConfig found")
    }

    /// Start syncing on all configured transports
    pub fn try_start_sync(&self) -> Result<(), DittoError> {
        // the License must be active or else sync will immediately fail
        if self.activated.load().not() {
            return Err(ErrorKind::NotActivated.into());
        }

        // We need a write lock because this will internally modify the transports
        // by activating those configured but not currently active
        match self.transports.write() {
            Ok(mut transports) => transports.try_start_sync(),
            Err(e) => {
                let rust_error = format!("{:?}", e); // needed as PoinsonedLockError is not Send
                Err(DittoError::new(ErrorKind::Internal, rust_error))
            }
        }
    }

    /// Stop syncing on all transports
    pub fn stop_sync(&self) {
        if let Ok(mut transports) = self.transports.write() {
            transports.stop_sync()
        }
    }

    /// Set a new `TransportConfig` and being syncing over these transports
    /// Any change to start or stop a specific transport should proceed via
    /// providing a modified configuration to this method
    pub fn set_transport_config(&self, config: TransportConfig) {
        if let Ok(mut transports) = self.transports.write() {
            transports.set_transport_config(config);
        }
    }

    /// Returns a snapshot of the currently configured transports
    pub fn current_transport_config(&self) -> Result<TransportConfig, DittoError> {
        match self.transports.read() {
            Ok(t) => Ok(t.current_config().clone()),
            Err(e) => {
                let msg = format!("The transport config is corrupted an inaccessible. {:?}", e);
                Err(DittoError::new(ErrorKind::Internal, msg))
            }
        }
    }
}

impl Ditto {
    pub fn with_sdk_version<R>(ret: impl FnOnce(&'_ str) -> R) -> R {
        ret(unsafe { ffi_sdk::ditto_get_sdk_version().to_str() })
    }
}

impl Ditto {
    pub fn set_logging_enabled(enabled: bool) {
        unsafe { ffi_sdk::ditto_logger_enabled(enabled) }
    }

    pub fn get_logging_enabled() -> bool {
        unsafe { ffi_sdk::ditto_logger_enabled_get() }
    }

    pub fn get_emoji_log_level_headings_enabled() -> bool {
        unsafe { ffi_sdk::ditto_logger_emoji_headings_enabled_get() }
    }

    pub fn set_emoji_log_level_headings_enabled(enabled: bool) {
        unsafe {
            ffi_sdk::ditto_logger_emoji_headings_enabled(enabled);
        }
    }

    pub fn get_minimum_log_level() -> ffi_sdk::CLogLevel {
        unsafe { ffi_sdk::ditto_logger_minimum_log_level_get() }
    }

    pub fn set_minimum_log_level(log_level: ffi_sdk::CLogLevel) {
        unsafe {
            ffi_sdk::ditto_logger_minimum_log_level(log_level);
        }
    }
}

impl Ditto {
    /// Activate a Ditto instance by setting a license token. You cannot
    /// sync with Ditto before you have activated it.
    pub fn set_license_token(&self, license_token: &str) -> Result<(), DittoError> {
        use ::safer_ffi::prelude::{AsOut, ManuallyDropMut};
        use ffi_sdk::LicenseVerificationResult;
        let c_license: char_p::Box = char_p::new(license_token);

        let mut err_msg = None;
        let out_err_msg = err_msg.manually_drop_mut().as_out();
        let res = unsafe { ffi_sdk::verify_license(c_license.as_ref(), Some(out_err_msg)) };

        if res == LicenseVerificationResult::LicenseOk {
            self.activated.store(true);
            return Ok(());
        }

        self.activated.store(false);
        let err_msg = err_msg.unwrap();

        ::log::error!("{}", err_msg);

        match res {
            LicenseVerificationResult::LicenseExpired => {
                Err(DittoError::license(LicenseError::LicenseTokenExpired {
                    message: err_msg.as_ref().to_string(),
                }))
            }
            LicenseVerificationResult::VerificationFailed => Err(DittoError::license(
                LicenseError::LicenseTokenVerificationFailed {
                    message: err_msg.as_ref().to_string(),
                },
            )),
            LicenseVerificationResult::UnsupportedFutureVersion => Err(DittoError::license(
                LicenseError::LicenseTokenUnsupportedFutureVersion {
                    message: err_msg.as_ref().to_string(),
                },
            )),
            _ => panic!("Unexpected license verification result {:?}", res),
        }
    }

    /// Activate a Ditto instance by setting a license token. You cannot
    /// sync with Ditto before you have activated it.
    pub fn set_access_license(&self, license_str: &str) {
        self.set_license_token(license_str)
            .expect("Invalid Ditto License provided")
    }

    /// Look for a license token from a given environment variable
    pub fn set_license_from_env(&self, var_name: &str) -> Result<(), DittoError> {
        match env::var(var_name) {
            Ok(token) => self.set_license_token(&token),
            Err(env::VarError::NotPresent) => {
                let msg = format!("No license token found for env var {}", &var_name);
                Err(DittoError::from_str(ErrorKind::Config, msg))
            }
            Err(e) => Err(DittoError::new(ErrorKind::Config, e)),
        }
    }

    /// Returns a reference to the underlying local data store
    pub fn store(&self) -> &Store {
        &self.store
    }

    /// Returns the site ID that the instance of Ditto is using as part of
    /// its identity.
    pub fn site_id(&self) -> u64 {
        self.site_id
    }

    /// Returns a custom identifier for the current device.
    ///
    /// When using observePeers(), each remote peer is represented by a short
    /// UTF-8 “device name”. By default this will be a truncated version of
    /// the device’s hostname. It does not need to be unique among peers.
    /// Configure the device name before calling start(). If it is too long
    /// it will be truncated.
    pub fn device_name(&self) -> &str {
        todo!();
    }

    /// Set a custom identifier for the current device
    pub fn set_device_name(&mut self, _name: String) {
        todo!();
    }

    /// Request bulk status information about the transports. This is mostly
    /// intended for statistical or debugging purposes.
    pub fn transport_diagnostics(&self) -> TransportDiagnostics {
        todo!();
    }

    /// Request information about Ditto peers in range of this device.
    ///
    /// This method returns an observer which should be held as long as updates
    /// are required. A newly registered observer will have a peers update
    /// delivered to it immediately. Then it will be invoked repeatedly when
    /// Ditto devices come and go, or the active connections to them change.
    pub fn observe_peers_v2<H>(&self, handler: H) -> PeersObserver
    where
        H: Fn(V2Presence) + Send + Sync + 'static,
    {
        self.presence_manager_v2
            .add_observer(self.ditto.retain(), handler)
    }

    pub fn root_dir(&self) -> &Path {
        self.ditto_root.root_path()
    }

    pub fn data_dir(&self) -> &Path {
        self.ditto_root.data_path()
    }

    pub fn authenticator(&self) -> Option<DittoAuthenticator> {
        self.auth.clone()
    }
    /// Has this Ditto instance been activated yet with a valid license token
    pub fn is_activated(&self) -> bool {
        self.activated.load()
    }
}

// Constructors
impl Ditto {
    /// Returns a builder for Ditto following the builder pattern
    /// Start constructing a new Ditto instance
    pub fn builder() -> DittoBuilder {
        DittoBuilder::new()
    }

    /// Construct a Ditto instance with sensible defaults
    /// This instance will still need to have a license set
    /// and sync functionality manually started
    pub fn new(app_id: AppId) -> Ditto {
        Ditto::builder()
            .with_root(Arc::new(
                PersistentRoot::from_current_exe().expect("Invalid Ditto Root"),
            ))
            .with_identity(|ditto_root| identity::OfflinePlayground::new(ditto_root, app_id))
            .expect("Invalid Ditto Identity")
            .with_minimum_log_level(CLogLevel::Info)
            .build()
            .expect("Failed to build Ditto Instance")
    }

    pub(crate) fn new_with_fields(fields: Arc<DittoFields>) -> Ditto {
        Ditto {
            fields: ManuallyDrop::new(fields),
        }
    }
}

impl Ditto {
    /// Removes all sync metadata for any remote peers which aren't currently
    /// connected. This method shouldn't usually be called. Manually running
    /// garbage collection often will result in slower sync times. Ditto
    /// automatically runs a garbage a collection process in the background
    /// at optimal times.
    ///
    /// Manually running garbage collection is typically only useful during
    /// testing if large amounts of data are being generated. Alternatively,
    /// if an entire data set is to be evicted and it's clear that
    /// maintaining this metadata isn't necessary, then garbage collection
    /// could be run after evicting the old data.
    pub fn run_garbage_collection(&self) {
        unsafe {
            ffi_sdk::ditto_run_garbage_collection(&self.ditto);
        }
    }
}

pub struct TransportDiagnostics;

pub type SiteId = u64;

#[derive(Clone, Debug)]
// pub struct AppId(uuid::Uuid); // Demo apps still use arbitrary strings
pub struct AppId(String); // neither String nor Vec<u8> are Copy

impl AppId {
    /// Generate a random AppId from a UUIDv4
    pub fn generate() -> Self {
        let uuid = uuid::Uuid::new_v4();
        AppId::from_uuid(uuid)
    }

    pub fn from_uuid(uuid: Uuid) -> Self {
        let id_str = format!("{:x}", &uuid); // lower-case with hypens
        AppId(id_str)
    }

    /// Attempt to grab a specific AppId from some environment variable
    pub fn from_env(var: &str) -> Result<Self, DittoError> {
        let id_str = env::var(var).map_err(|err| DittoError::new(ErrorKind::Config, err))?;
        Ok(AppId(id_str))
    }

    pub fn as_str(&self) -> &str {
        &self.0
    }

    pub fn to_c_string(&self) -> char_p::Box {
        char_p::new(self.0.as_str())
    }

    /// Returns the default auth URL associated to the App ID
    /// this is `https://{app_id}.cloud.ditto.live/`
    // the AppID dictates the base URL by default
    pub fn default_auth_url(&self) -> String {
        format!("https://{}.cloud.ditto.live", self.0)
    }

    /// Returns the default Websocket Sync URL which is
    /// `wss://{app_id}.cloud.ditto.live/`
    pub fn default_sync_url(&self) -> String {
        format!("wss://{}.cloud.ditto.live", self.0)
    }
}

use std::{fmt, fmt::Display, str::FromStr};

impl Display for AppId {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        write!(f, "{}", self.0)
    }
}

impl FromStr for AppId {
    type Err = DittoError;
    fn from_str(s: &str) -> Result<AppId, DittoError> {
        // later s will need to be a valid UUIDv4
        Ok(AppId(s.to_string()))
    }
}
