use_prelude!();

use collection::ScopedCollection;

use super::collection::document_id::DocumentId;
mod collection;

/// Use lifetime parameters as type-level IDs (generative lifetimes),
/// so as to prevent the following anti-pattern from compiling:
///
/// ```rust,compile_fail
/// use ::dittokit::prelude::*;
/// # use ::dittokit::internal::test_helpers::*;
/// # use ::std::path::Path;
///
/// let dir1 = Path::new("...");
/// let mut ditto1 = get_ditto(dir1);
///
/// let dir2 = Path::new("...");
/// let mut ditto2 = get_ditto(dir2);
///
/// ditto1.store().with_batched_write(|batch1| {
///     // API requires that this closure return:
///     // - either `batch.commit_changes()`
///     // - or `batch.revert_changes()`
///
///     // But somebody could do:
///     let mut commit_changes2 = None;
///     ditto2.store().with_batched_write(|batch2| {
///         commit_changes2 = Some(batch2.commit_changes());
///         batch1.revert_changes()
///     });
///     commit_changes2.unwrap()
/// });
/// ```
use marker::InvariantLifetime as TypeLevelId;

// Note, in practice `'batch` is `'txn`, but we hide this from
// the signatures to hide this implementation detail to the users.
pub struct ScopedStore<'batch> {
    // " 'batch == 'txn "
    txn: &'batch mut ffi_sdk::CWriteTransaction,

    // This could be 'store (which is ≥ 'txn), but for all effects and purposes
    // shrinking this to `'txn` is just as useful, and simplifies / reduces
    // the number of lifetime parameters.
    store: &'batch Store, // a Ref to the REAL store not the scoped store

    results: &'batch mut Vec<WriteTransactionResult>,

    // make sure the `'batch` lifetime is invariant so that it can be used as
    // a type-level unique id.
    _txn_id: TypeLevelId<'batch>,
}

pub(super) fn with_batched_write<F>(
    store: &Store, // Has lifetime of 'store
    f: F,
) -> Result<Vec<WriteTransactionResult>, super::DittoError>
where
    for<'batch> F: FnOnce(ScopedStore<'batch>) -> Action<'batch>,
{
    use super::*;

    let mut txn = {
        let mut slot: Option<ffi_sdk::BoxedWriteTransaction> = None;
        let out_txn = slot.manually_drop_mut().as_out();
        let status = unsafe { ffi_sdk::ditto_write_transaction(&*store.ditto, out_txn) };
        if status != 0 {
            return Err(DittoError::from_ffi(ErrorKind::Internal));
        }
        slot.unwrap()
    };
    let mut results = vec![];

    let batch = ScopedStore {
        txn: &mut *txn,
        store, // The &Store contains an Arc to the root Ditto instance
        results: &mut results,
        _txn_id: TypeLevelId::default(),
    };
    match f(batch).0 {
        ActionKind::Commit => {
            let status = unsafe { ffi_sdk::ditto_write_transaction_commit(&*store.ditto, txn) };
            if status != 0 {
                return Err(DittoError::from_ffi(ErrorKind::Internal));
            }
        }
        ActionKind::Rollback => {
            unsafe { ffi_sdk::ditto_write_transaction_rollback(txn) };
        }
    }
    Ok(results)
}

pub struct Action<'batch>(ActionKind, TypeLevelId<'batch>);

enum ActionKind {
    Commit,
    Rollback,
}

impl<'batch> ScopedStore<'batch> {
    pub fn commit_changes(self: ScopedStore<'batch>) -> Action<'batch> {
        Action(ActionKind::Commit, TypeLevelId::default())
    }

    pub fn revert_changes(self: ScopedStore<'batch>) -> Action<'batch> {
        Action(ActionKind::Rollback, TypeLevelId::default())
    }

    pub fn collection<'coll>(
        self: &'coll mut ScopedStore<'batch>,
        collection_name: &'_ str,
    ) -> ScopedCollection<'coll, 'batch> {
        let c_name = char_p::new(collection_name);
        ScopedCollection {
            batch: self,
            collection_name: c_name,
        }
    }
}

pub struct WriteTransactionResult {
    pub collection_name: char_p::Box,

    pub doc_id: DocumentId,

    pub kind: DocChangeKind,
}

#[derive(Debug, PartialEq, Eq)]
pub enum DocChangeKind {
    Inserted,
    Removed,
    Evicted,
    Updated,
}
