//! A basic example showing how to insert Events into a Timeseries using the
//! RustSDK
//!
//! Currently assumes HyDRA subscription server is listening on localhost:4040.
//! This can be arranged with a k8s cluster by running
//! ```shell
//! kubectl port-forward <sub-server pod> 4040
//! ```

extern crate safer_ffi;
extern crate tempdir;

use std::{env, error::Error as ErrorTrait, path::PathBuf};

use chrono::Utc;
use dittolive_ditto::{
    error::{DittoError, ErrorKind},
    prelude::*,
};
use serde::{Deserialize, Serialize};
use uuid::Uuid;

#[derive(Serialize, Deserialize, Debug)]
struct SensorEvent {
    cpu_temp: u8,
}

fn main() -> Result<(), Box<dyn ErrorTrait>> {
    env_logger::init();
    let app_uuid = Uuid::new_v4().to_string();
    let site_id = 123u64;
    // define actor/site ID
    // If absent a random one will be created
    let license_token: String =
        env::var("DITTO_LICENSE").map_err(|e| DittoError::new(ErrorKind::License, e))?;
    let data_dir = env::var_os("DITTO_DB_PATH")
        .map(PathBuf::from)
        .or_else(|| -> Option<PathBuf> { Some(std::env::temp_dir()) })
        .map(|mut path: PathBuf| {
            path.push("dittokit");
            let _ = std::fs::create_dir_all(&path);
            path
        });

    let hydra_host: String =
        env::var("HYDRA_HOST").unwrap_or_else(|_| "wss://hydra.dev.k8s.ditto.live".to_string());

    let identity = Identity::new_development(&app_uuid, site_id, data_dir.as_deref()).unwrap();
    let mut ditto = Ditto::new(identity, data_dir);
    ditto.set_access_license(license_token.as_str());

    let mut transport_config = TransportConfig::new();
    transport_config.connect.websocket_urls.insert(hydra_host);
    ditto.set_transport_config(transport_config);
    ditto.start_sync();

    let store = ditto.store();
    let ts = store.timeseries("readings").unwrap();
    let mut temp = 37;
    loop {
        let now = Utc::now();
        let msg = SensorEvent { cpu_temp: temp };
        ts.insert(&now, msg).unwrap();
        std::thread::sleep(std::time::Duration::from_millis(1000));
        temp += 1;
    }
}
