//! Bounce latency receiver
//!
//! Subscribes to documents inserted by the Bounce Latency Sender
//! and computes the total travel latency.
//!
//! Example Invocation
//! ``` shell
//! RUST_LOG=debug DITTO_APP_NAME=live.ditto.carsapp DITTO_SITE_ID=2 \
//! DITTO_DB_PATH=./receiver cargo run -p dittokit --example bounce_latency_rx
//! ```
//!
//! Note that it is vital you clear the local database files (`rm -rf receiver`
//! in the example above) between runs in order to keep the replication state
//! intact

extern crate log;
extern crate safer_ffi;
extern crate serde;
extern crate serde_cbor;
extern crate tempdir;

use std::{env, path::PathBuf};

use chrono::{DateTime, Duration, NaiveDateTime, Utc};
use dittolive_ditto::prelude::*;
use rand::Rng;
use serde::{Deserialize, Serialize};
use serde_cbor::Value;
use uuid::Uuid;

#[derive(Serialize, Deserialize)]
struct LatencyProbe {
    timestamp: i64,
    subsec_nanos: u32,
}

fn main() -> Result<(), Box<dyn std::error::Error>> {
    env_logger::init();

    // This must match the APP UUID of the bounce_latency_tx instance
    let app_uuid = env::var("DITTO_APP_UUID").unwrap_or_else(|_x| Uuid::new_v4().to_string());
    // define actor/site ID
    let site_id: u64 = env::var("DITTO_SITE_ID")
        .map(|x| x.parse::<u64>().unwrap())
        .unwrap_or_else(|_x| {
            // recall the site ID's 0 and 1 have special meaning
            ::rand::thread_rng().gen_range(2..SiteId::MAX)
        });
    let license_str: String = std::env::var("DITTO_LICENSE").expect("No License Env Var provided");
    let data_dir = env::var_os("DITTO_DB_PATH")
        .map(PathBuf::from)
        .or_else(|| -> Option<PathBuf> { Some(std::env::temp_dir()) })
        .map(|mut path: PathBuf| {
            path.push("dittokit");
            let _ = std::fs::create_dir_all(&path);
            path
        });

    let hydra_host: String =
        env::var("HYDRA_HOST").unwrap_or_else(|_| "localhost:4040".to_string());

    let identity = Identity::new_development(&app_uuid, site_id, data_dir.as_deref()).unwrap();
    let mut ditto = Ditto::new(identity, data_dir);
    ::log::debug!("Setting Ditto License.");
    ditto.set_access_license(license_str.as_str());
    // To Receive VIA HYDRA uncomment the following
    // and specify the TCP Server as the location of
    // the Hydra Subscription Server
    let mut transport_config = TransportConfig::new();
    transport_config.connect.tcp_servers.insert(hydra_host);
    ditto.set_transport_config(transport_config);
    // this will start the sync operations
    ditto.start_sync();
    ::log::debug!("Ditto Receiver app started!");

    // Use this to communicate directly (no subscription server)
    // ditto.start_tcp_server("127.0.0.1:4040");
    let store = ditto.store();
    let collection = store.collection("testcollection");

    let (tx, rx) = std::sync::mpsc::sync_channel(120);
    // This handler is called every time docs from local or remote sources are
    // committed to the local store which match the associated query.
    // `documents` is a vec of ALL documents matching the query after application of
    // the transaction.
    // `event` can be used to dissect out which of these are insertions.
    let event_handler = move |documents: Vec<ffi_sdk::BoxedDocument>, event| {
        ::log::trace!(
            "Latency Receiver got {:?} with {} updated documents",
            &event,
            documents.len()
        );
        match event {
            LiveQueryEvent::Initial { .. } => {
                // On an initial sync, we can calculate the latency for arrival of the first
                // document
                if let Some(first_doc) = documents.get(0) {
                    let latency = calc_latency(first_doc);
                    ::log::info!("Initial Sync Latency is {} ", latency);
                }
            }
            LiveQueryEvent::Update { insertions, .. } => {
                // We only want to send the newest event
                for idx in insertions.iter() {
                    if let Some(doc) = documents.get(*idx) {
                        let latency = calc_latency(doc);
                        let _ = tx.send(latency);
                    }
                }
            }
        }
    };
    let start_time = Utc::now().timestamp(); // The time of this session
                                             // this will filter for only events in the current test session
    let query: String = format!("timestamp >= {}", start_time);
    // this registers and starts the live query
    let _lq = collection.unwrap().find(&query).observe(event_handler);
    let mut running_stats: Accumulator = Accumulator::default();
    for measurement in rx.iter() {
        // We round down to nearest millisecond precision
        // This can be adjusted if needed
        running_stats.push(measurement.num_milliseconds() as f64);
        let msl = running_stats.mean();
        let stddev = running_stats.std_dev();
        ::log::info!("Latency is {} +/- {} ms", msl, stddev);
    }

    Ok(())
}

/// Parse the Document and get the latency Duration
fn calc_latency(doc: &ffi_sdk::Document) -> Duration {
    let received = Utc::now();
    let mut seconds = 0;
    let mut nanos = 0;
    let ts_val = doc
        .get("timestamp")
        .expect("failed to get value for timestamp");
    if let Value::Integer(ts) = ts_val {
        seconds = ts;
    } else {
        ::log::error!("timestamp is not encoded as an U64");
    }
    let ns_val = doc
        .get("subsec_nanos")
        .expect("Failed to get value for subsec_nanos");
    if let Value::Integer(ns) = ns_val {
        nanos = ns;
    } else {
        ::log::error!("subsec_nanos is not encoded as an U64");
    }
    let sent = DateTime::<Utc>::from_utc(
        NaiveDateTime::from_timestamp(seconds as i64, nanos as u32),
        Utc,
    );
    let latency: Duration = received - sent;
    ::log::info!(
        "Message received at {}, sent at {}, with latency {}",
        &received,
        &sent,
        &latency
    );
    latency
}

/// A univariate statistical accumulator (rolling average)
struct Accumulator {
    /// total number of elements observed
    count: usize,
    moment_1: f64,
    moment_2: f64,
    moment_3: f64,
    moment_4: f64,
    min: f64,
    max: f64,
    /* Median, quartiles, and Mode are more awkward to compute
     * without windowing or discretization */
}

impl Default for Accumulator {
    fn default() -> Self {
        Accumulator {
            count: 0,
            moment_1: 0.0,
            moment_2: 0.0,
            moment_3: 0.0,
            moment_4: 0.0,
            min: 0.0,
            max: 0.0,
        }
    }
}

impl Accumulator {
    /// Push a new observation
    fn push(&mut self, x: f64) {
        let old_count = self.count as f64;
        self.count += 1;
        let n = self.count as f64; // current count as a float
        let delta = x - self.moment_1;
        let delta_n = delta / (self.count as f64);
        let delta_n2 = delta_n * delta_n;
        let term_1 = delta * delta_n * old_count;
        // update our mean
        self.moment_1 += delta_n;

        // Update our kurtosis
        self.moment_4 += term_1 * delta_n2 * (n * n - 3.0 * n + 3.0)
            + 6.0 * delta_n2 * self.moment_2
            - 4.0 * delta_n * self.moment_3;

        // update our skew
        self.moment_3 += term_1 * delta_n * (n - 2.0) - 3.0 * delta_n * self.moment_2;

        // update our variance
        self.moment_2 += term_1;

        if x > self.max {
            self.max = x;
        }
        if x < self.min {
            self.min = x;
        }
    }

    fn mean(&self) -> f64 {
        self.moment_1
    }

    fn variance(&self) -> f64 {
        self.moment_2 / ((self.count - 1) as f64)
    }

    fn std_dev(&self) -> f64 {
        self.variance().sqrt()
    }
}
