//! Tests of live query functionality
#[macro_use]
extern crate serde_json;

use std::sync::{
    atomic::{AtomicBool, Ordering},
    Arc, Mutex,
};

use common::*;
use ditto_test_support::ditto_test;

mod common;

/// Test inserting a document inside of a LQ Callback
/// triggers the correct Initial LiveQueryEvent
#[ditto_test]
fn initial_lq() -> Result<(), Box<dyn std::error::Error>> {
    let test_dir = common::temp_persistence_dir();
    let ditto = common::get_ditto_kit(test_dir.path());
    let store = ditto.store();
    let uuid = uuid::Uuid::new_v4().to_string();
    let collection = store.collection(&uuid).unwrap();

    let doc1 = TestType {
        make: String::from("Honda"),
        color: TestColor::Red,
        ..Default::default()
    };
    let expected_doc = doc1.clone(); // make a copy for the test

    let doc1_id = DocumentId::new(&"red_honda".to_string()).unwrap();

    let doc1_id_ret = collection.insert(doc1, Some(&doc1_id), false).unwrap();
    assert_eq!(doc1_id_ret, doc1_id);

    let doc2 = TestType {
        make: String::from("Honda"),
        color: TestColor::Crimson,
        ..Default::default()
    };
    let _doc2_id = collection.insert(doc2, None, false).unwrap();

    let finished = Arc::new(AtomicBool::new(false));
    let finished_clone = Arc::clone(&finished);
    let counter = Arc::new(Mutex::new(0));
    assert_eq!(*counter.lock().unwrap(), 0);

    let handler = move |docs: Vec<BoxedDocument>, event| {
        let counter_mtx = &*counter; // move (copy) and reborrow
        if let Ok(mut counter) = counter_mtx.lock() {
            *counter += 1;
        }
        match event {
            LiveQueryEvent::Initial => {
                if let Ok(counter) = counter_mtx.lock() {
                    assert_eq!(*counter, 1);
                }
                assert_eq!(docs.len(), 1);
                assert_eq!(doc1_id, docs[0].id());
                let first_doc = docs[0].typed::<TestType>().unwrap();
                assert_eq!(first_doc, expected_doc);
                finished_clone.store(true, Ordering::SeqCst);
            }
            LiveQueryEvent::Update { .. } => {
                unimplemented!("Unexpected Update Received");
            }
        }
    };
    let lq = collection.find("color == \'Red\'").observe(handler)?; // String values in queries must be single quoted

    // wait for the LQ thread to finish
    while !finished.load(Ordering::SeqCst) {
        std::thread::yield_now();
    }

    lq.stop();

    Ok(())
}

/// Test inserting a new document triggers the correct Update LiveQueryEvent
#[ditto_test]
fn insert_lq() -> Result<(), Box<dyn std::error::Error>> {
    let test_dir = common::temp_persistence_dir();
    let ditto = common::get_ditto_kit(test_dir.path());
    let store = ditto.store();
    let uuid = uuid::Uuid::new_v4().to_string();
    let collection = store.collection(&uuid).unwrap(); // Gets moved into Handler

    let finished = Arc::new(AtomicBool::new(false));
    let finished_clone = Arc::clone(&finished);
    let counter = Arc::new(Mutex::new(0));
    assert_eq!(*counter.lock().unwrap(), 0);
    let collection_clone = collection.clone(); // Ref is not allowed as must be 'static

    let handler = move |docs: Vec<BoxedDocument>, event| {
        let counter_mtx = &*counter; // move (copy) and reborrow
        if let Ok(mut counter) = counter_mtx.lock() {
            *counter += 1;
        }
        match event {
            LiveQueryEvent::Initial => {
                // this is a bit overkill but it keeps our Handler Fn
                if let Ok(counter) = counter_mtx.lock() {
                    assert_eq!(*counter, 1);
                }
                assert!(docs.is_empty());
                // insert a matching doc
                let doc1 = TestType {
                    make: String::from("Honda"),
                    color: TestColor::Red,
                    ..Default::default()
                };
                // The Collection here has a reference to the underlying core ditto Instance
                // which may be freed by the time this insert is called unless
                // something holds the test open
                let _id = collection_clone.insert(doc1, None, false).unwrap();
            }
            LiveQueryEvent::Update {
                old_documents,
                insertions,
                ..
            } => {
                let expected_insertions = [0_usize; 1];
                let expected_doc = TestType {
                    make: String::from("Honda"),
                    color: TestColor::Red,
                    ..Default::default()
                };
                if let Ok(counter) = counter_mtx.lock() {
                    assert_eq!(*counter, 2);
                }
                let first_doc = docs[0].typed::<TestType>().unwrap();
                assert_eq!(docs.len(), 1);
                assert!(old_documents.is_empty());
                assert_eq!(insertions, expected_insertions.into());
                assert_eq!(first_doc, expected_doc);
                finished_clone.store(true, Ordering::SeqCst);
            }
        }
    };
    let lq = collection.find("color == \'Red\'").observe(handler)?; // String values in queries must be single quoted

    // wait for the LQ thread to finish
    while !finished.load(Ordering::SeqCst) {
        std::thread::yield_now();
    }

    lq.stop(); // In theory, the clone should drop here
    drop(test_dir);
    drop(collection);
    drop(ditto);

    Ok(())

    // The core ditto ptr is freed here unless something is holding the test
    // open
}

/// Test a live query for a single document, with document IDs of different
/// types.
#[ditto_test]
fn single_document_with_different_ids() {
    let test_dir = common::temp_persistence_dir();
    let ditto = common::get_ditto_kit(test_dir.path());
    let store = ditto.store();
    let uuid = uuid::Uuid::new_v4().to_string();
    let collection = store.collection(&uuid).unwrap();

    let raw_doc_ids: Vec<serde_json::Value> = vec![
        "boring_old_string".to_string().into(),
        1.into(),
        0.into(),
        123.into(),
        9999.into(),
        false.into(),
        true.into(),
        (0..64)
            .map(|_| rand::random::<u8>())
            .collect::<Vec<u8>>()
            .into(),
        vec!["a", "abc", "z89{{}}@£!fv>?!,[](){{}}000"].into(),
        json!({"a": "b", "__num__@£$%^&{})(|,,./!?": -7123}),
    ];

    raw_doc_ids.iter().for_each(|raw_id| {
        let doc_id = DocumentId::new(raw_id).unwrap();

        let doc = TestType {
            make: String::from("Honda"),
            color: TestColor::Red,
            ..Default::default()
        };

        // make copies for the test
        let expected_doc_id = doc_id.clone();
        let expected_doc = doc.clone();

        let returned_id = collection.insert(doc, Some(&doc_id), false).unwrap();
        assert_eq!(returned_id, doc_id);

        let finished = Arc::new(AtomicBool::new(false));
        let finished_clone = Arc::clone(&finished);
        let counter = Arc::new(Mutex::new(0));
        assert_eq!(*counter.lock().unwrap(), 0);

        let handler = move |docs: Vec<BoxedDocument>, event| {
            let counter_mtx = &*counter; // move (copy) and reborrow
            if let Ok(mut counter) = counter_mtx.lock() {
                *counter += 1;
            }
            match event {
                LiveQueryEvent::Initial => {
                    if let Ok(counter) = counter_mtx.lock() {
                        assert_eq!(*counter, 1);
                    }
                    assert_eq!(docs.len(), 1);
                    assert_eq!(expected_doc_id, docs[0].id());
                    let first_doc = docs[0].typed::<TestType>().unwrap();
                    assert_eq!(first_doc, expected_doc);
                    finished_clone.store(true, Ordering::SeqCst);
                }
                LiveQueryEvent::Update { .. } => {
                    unimplemented!("Unexpected Update Received");
                }
            }
        };
        let lq = collection.find_by_id(doc_id).observe(handler).unwrap();

        // wait for the LQ thread to finish
        while !finished.load(Ordering::SeqCst) {
            std::thread::yield_now();
        }

        lq.stop();
    });
}

/// Test inserting a new document triggers the correct Update LiveQueryEvent
#[ditto_test]
fn insert_lq_with_query_args() -> Result<(), Box<dyn std::error::Error>> {
    let test_dir = common::temp_persistence_dir();
    let ditto = common::get_ditto_kit(test_dir.path());
    let store = ditto.store();
    let uuid = uuid::Uuid::new_v4().to_string();
    let collection = store.collection(&uuid).unwrap(); // Gets moved into Handler

    let finished = Arc::new(AtomicBool::new(false));
    let finished_clone = Arc::clone(&finished);
    let counter = Arc::new(Mutex::new(0));
    assert_eq!(*counter.lock().unwrap(), 0);
    let collection_clone = collection.clone(); // Ref is not allowed as must be 'static

    let handler = move |docs: Vec<BoxedDocument>, event| {
        let counter_mtx = &*counter; // move (copy) and reborrow
        if let Ok(mut counter) = counter_mtx.lock() {
            *counter += 1;
        }
        match event {
            LiveQueryEvent::Initial => {
                // this is a bit overkill but it keeps our Handler Fn
                if let Ok(counter) = counter_mtx.lock() {
                    assert_eq!(*counter, 1);
                }
                assert!(docs.is_empty());
                // insert a matching doc
                let doc1 = TestType {
                    make: String::from("Honda"),
                    color: TestColor::Red,
                    ..Default::default()
                };
                // The Collection here has a reference to the underlying core ditto Instance
                // which may be freed by the time this insert is called unless
                // something holds the test open
                let _id = collection_clone.insert(doc1, None, false).unwrap();
            }
            LiveQueryEvent::Update {
                old_documents,
                insertions,
                ..
            } => {
                let expected_insertions = [0_usize; 1];
                let expected_doc = TestType {
                    make: String::from("Honda"),
                    color: TestColor::Red,
                    ..Default::default()
                };
                if let Ok(counter) = counter_mtx.lock() {
                    assert_eq!(*counter, 2);
                }
                let first_doc = docs[0].typed::<TestType>().unwrap();
                assert_eq!(docs.len(), 1);
                assert!(old_documents.is_empty());
                assert_eq!(insertions, expected_insertions.into());
                assert_eq!(first_doc, expected_doc);
                finished_clone.store(true, Ordering::SeqCst);
            }
        }
    };
    let lq = collection
        .find_with_args("color == $args.color", json!({"color": "Red"}))
        .observe(handler)?;

    // wait for the LQ thread to finish
    while !finished.load(Ordering::SeqCst) {
        std::thread::yield_now();
    }

    lq.stop(); // In theory, the clone should drop here
    drop(test_dir);
    drop(collection);
    drop(ditto);

    Ok(())

    // The core ditto ptr is freed here unless something is holding the test
    // open
}
