use std::{collections::HashSet, path::PathBuf};

/// A configuration object specifying which network transports Ditto should use
/// to sync data.
///
/// A `Ditto` object comes with a default transport configuration where all
/// available peer-to-peer transports are enabled. You can customize this by
/// initializing a `TransportConfig`, adjusting its properties, and supplying it
/// to `set_transport_config()` on `Ditto`.
///
/// When you initialize a `TransportConfig` yourself it starts with all
/// transports disabled. You must enable each one directly.
///
/// Peer-to-peer transports will automatically discover peers in the vicinity
/// and create connections without any configuration. These are configured
/// inside the `peer_to_peer` property. To turn each one on, set its `enabled`
/// property to `true`.
///
/// To connect to a peer at a known location, such as a Ditto Big Peer, add its
/// address inside the `connect` configuration. These are either `host:port`
/// strings for raw TCP sync, or a `wss://…` URL for websockets.
///
/// The `listen` configurations are for specific less common data sync
/// scenarios. Please read the documentation on the Ditto website for examples.
/// Incorrect use of `listen` can result in insecure configurations.
#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct TransportConfig {
    pub peer_to_peer: PeerToPeer,
    pub connect: Connect,
    pub listen: Listen,
}

impl TransportConfig {
    pub fn new() -> Self {
        Self {
            peer_to_peer: PeerToPeer {
                bluetooth_le: BluetoothConfig::new(),
            },
            connect: Connect {
                tcp_servers: HashSet::new(),
                websocket_urls: HashSet::new(),
            },
            listen: Listen {
                tcp: TcpListenConfig::new(),
                http: HttpListenConfig::new(),
            },
        }
    }
}

#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct PeerToPeer {
    // For future use - LAN sync etc.
    pub bluetooth_le: BluetoothConfig,
}

#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct Connect {
    pub tcp_servers: HashSet<String>,
    pub websocket_urls: HashSet<String>,
}

#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct Listen {
    pub tcp: TcpListenConfig,
    pub http: HttpListenConfig,
}

#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct HttpListenConfig {
    pub enabled: bool,
    pub interface_ip: String,
    pub port: u16,
    pub static_content_path: Option<PathBuf>,
    pub websocket_sync: bool,
    pub tls_key_path: Option<PathBuf>,
    pub tls_certificate_path: Option<PathBuf>,
}

impl HttpListenConfig {
    pub fn new() -> Self {
        Self {
            enabled: false,
            interface_ip: "[::]".to_string(),
            port: 80,
            static_content_path: None,
            websocket_sync: true,
            tls_key_path: None,
            tls_certificate_path: None,
        }
    }
}

#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct TcpListenConfig {
    pub enabled: bool,
    pub interface_ip: String,
    pub port: u16,
}

impl TcpListenConfig {
    pub fn new() -> Self {
        Self {
            enabled: false,
            interface_ip: "[::]".to_string(),
            port: 4040,
        }
    }
}

#[derive(Clone, Debug, Default, PartialEq, Eq)]
pub struct BluetoothConfig {
    pub enabled: bool,
}

impl BluetoothConfig {
    pub fn new() -> Self {
        Self { enabled: false }
    }
}
