use std::{collections::HashMap, convert::TryFrom};

use serde::Deserialize;

use crate::error::{DittoError, ErrorKind};

#[derive(Debug, Clone, Deserialize)]
#[serde(try_from = "DittoAttachmentTokenStoredFormat")]
pub struct DittoAttachmentToken {
    pub(crate) id: Box<[u8]>,
    pub(crate) len: u64,
    pub(crate) metadata: HashMap<String, String>,
}

impl TryFrom<DittoAttachmentTokenStoredFormat> for DittoAttachmentToken {
    type Error = DittoError;

    fn try_from(value: DittoAttachmentTokenStoredFormat) -> Result<Self, Self::Error> {
        if value._type != "ditto_attachment" {
            return Err(ErrorKind::Internal.into());
        }
        let ret = DittoAttachmentToken {
            id: value._id,
            len: value._len,
            metadata: value._meta,
        };
        Ok(ret)
    }
}

// Helper struct to help deserialize. This is to let us use
// #[derive(Deserialize)] instead of manually implementing it.
// See `impl serde::Serialize for DittoAttachment`
#[derive(Deserialize)]
#[serde(deny_unknown_fields)]
struct DittoAttachmentTokenStoredFormat {
    _type: String,
    _id: Box<[u8]>,
    _len: u64,
    _meta: HashMap<String, String>,
}
