use std::env;

use ffi_sdk::BoxedDitto;

use crate::{
    error::{DittoError, ErrorKind, LicenseError},
    transport::TransportConfig,
    utils::prelude::*,
};

/// The entry point for accessing Ditto-related functionality
/// This struct is generally a handle and interface to ditto-functionality
/// which operates in background threads.
pub struct DittoKit {
    ditto: Arc<ffi_sdk::BoxedDitto>,
    store: Store,
    activated: bool,
    site_id: SiteId,
    /// Handles for the various Transports
    /// operating on the other side of the FFI interface
    tcp_clients: HashMap<String, Box<dyn Any + Send + Sync>>,
    ws_clients: HashMap<String, Box<dyn Any + Send + Sync>>,
    ble_client_transport: Option<Box<dyn Any + Send + Sync>>,
    ble_server_transport: Option<Box<dyn Any + Send + Sync>>,

    sync_active: bool,
    transport_config: TransportConfig,
}

impl DittoKit {
    pub fn new(
        identity: impl Borrow<Identity>,
        persistence_dir: Option<impl Into<PathBuf>>,
    ) -> Self {
        let log_level: ffi_sdk::CLogLevel = env::var("RUST_SDK_LOG_LEVEL")
            .map(|s| match s.as_str() {
                "error" => ffi_sdk::CLogLevel::Error,
                "warning" => ffi_sdk::CLogLevel::Warning,
                "info" => ffi_sdk::CLogLevel::Info,
                "debug" => ffi_sdk::CLogLevel::Debug,
                "verbose" => ffi_sdk::CLogLevel::Verbose,
                _ => ffi_sdk::CLogLevel::Info,
            })
            .unwrap_or(CLogLevel::Debug);

        let identity = identity.borrow();
        let platform = using!(match () {
            use ffi_sdk::Platform;
            | _case if cfg!(target_os = "windows") => Platform::Windows,
            | _case if cfg!(target_os = "android") => Platform::Android,
            | _case if cfg!(target_os = "macos") => Platform::Mac,
            | _case if cfg!(target_os = "ios") => Platform::Ios,
            | _case if cfg!(target_os = "linux") => Platform::Linux,
            | _default => Platform::Unknown,
        });
        let sdk_semver = char_p::new("1.0.0");
        unsafe {
            ffi_sdk::ditto_init_sdk_version(platform, ffi_sdk::Language::Rust, sdk_semver.as_ref());
        }
        unsafe {
            ffi_sdk::ditto_logger_init();
            ffi_sdk::ditto_logger_minimum_log_level(log_level);
            ffi_sdk::ditto_logger_enabled(true);
        }
        let persistence_dir = DittoKit::default_persistence_dir(persistence_dir);
        let working_dir = char_p::new(persistence_dir);
        let uninit_ditto = unsafe { ffi_sdk::uninitialized_ditto_make(working_dir.as_ref()) };
        ::log::debug!("Unitialized Ditto instance started");
        let site_id = unsafe { ffi_sdk::ditto_auth_client_get_site_id(&identity.auth_client) };
        let boxed_ditto = unsafe { ffi_sdk::ditto_make(uninit_ditto, &identity.auth_client) };
        ::log::debug!("Ditto initialized");
        let ditto: Arc<BoxedDitto> = Arc::new(boxed_ditto);
        let store = Store::new(ditto.clone(), site_id);
        ::log::debug!("Local Datastore initialized");

        let transport_config = TransportConfig::new();

        Self {
            ditto,
            store,
            activated: false,
            site_id,
            tcp_clients: HashMap::new(),
            ws_clients: HashMap::new(),
            ble_client_transport: None,
            ble_server_transport: None,
            sync_active: false,
            transport_config,
        }
    }

    /// The default location DittoKit will store data if a persistence directory
    /// isn't supplied.
    pub fn default_persistence_dir(persistence_dir: Option<impl Into<PathBuf>>) -> String {
        let mut persistence_dir: PathBuf = if let Some(it) = persistence_dir {
            it.into()
        } else {
            ::dirs_next::data_local_dir()
                .expect("Failed to retrieve a local data dir on this platform")
                .tap_mut(|it| it.push("DittoKit"))
        };
        persistence_dir.push("dittokit");
        ::log::debug!("Ditto dir: {}", persistence_dir.display());
        persistence_dir
            .into_os_string()
            .into_string()
            .unwrap_or_else(|oss| panic!("Invalid UTF-8 in path {:?}", oss,))
    }
}

impl Drop for DittoKit {
    fn drop(&mut self) {
        // stop all transports
        self.stop_sync();
        // stop all servers and advertisers
        // the idea is to maximize the chance we
        // can get an exclusive lock
        unsafe { ffi_sdk::ditto_shutdown(&self.ditto) };
        ::log::debug!("Dropping ditto kit");
        // Now with everything shut down try to get a mut reference
        // The most likely remaining references are the LiveQueries
        if let Some(_witness_of_unicity) = Arc::get_mut(&mut self.ditto) {
            // Our own strongs counts are equal to 1;
            // so we implicitly drop the `BoxedDitto` which will call
            // `ditto_drop(…)`
        } else {
            // We do NOT want to drop the referent (ffi::Ditto)
            // as LiveQueres are still running. The user is responsible for
            // cleaning these LQs up.
            let count = Arc::strong_count(&self.ditto);
            ::log::debug!("Attempting to drop DittoKit with {} live references", count);
        }
    }
}

impl DittoKit {
    pub fn start_sync(&mut self) {
        // `unwrap` here to preserve old behavior
        self.try_start_sync().unwrap()
    }

    pub fn try_start_sync(&mut self) -> Result<(), DittoError> {
        if self.activated.not() {
            return Err(ErrorKind::NotActivated.into());
        }
        if !self.sync_active {
            self.sync_active = true;
            let all_disabled = TransportConfig::new();
            self.apply_transport_config(&self.transport_config.clone(), &all_disabled);
        }
        Ok(())
    }

    pub fn stop_sync(&mut self) {
        if self.sync_active {
            self.sync_active = false;
            let all_disabled = TransportConfig::new();
            self.apply_transport_config(&all_disabled, &self.transport_config.clone());
        }
    }

    pub fn set_transport_config(&mut self, config: TransportConfig) {
        if self.sync_active {
            self.apply_transport_config(&config, &self.transport_config.clone());
        }
        self.transport_config = config;
    }

    fn apply_transport_config(&mut self, config: &TransportConfig, old_config: &TransportConfig) {
        // Diff the two configs transport-by-transport
        if config.listen.tcp != old_config.listen.tcp {
            self.stop_tcp_listen();
            if config.listen.tcp.enabled {
                self.start_tcp_listen(&config.listen.tcp);
            }
        }

        if config.listen.http != old_config.listen.http {
            self.stop_http_listen();
            if config.listen.http.enabled {
                let _ = self.start_http_listen(&config.listen.http);
            }
        }

        let tcp_connects_to_stop = old_config
            .connect
            .tcp_servers
            .difference(&config.connect.tcp_servers);
        for addr in tcp_connects_to_stop {
            self.stop_tcp_connect(addr);
        }

        let tcp_connects_to_start = config
            .connect
            .tcp_servers
            .difference(&old_config.connect.tcp_servers);
        for addr in tcp_connects_to_start {
            self.start_tcp_connect(addr.clone());
        }

        let ws_connects_to_stop = old_config
            .connect
            .websocket_urls
            .difference(&config.connect.websocket_urls);
        for url in ws_connects_to_stop {
            self.stop_ws_connect(url);
        }

        let ws_connects_to_start = config
            .connect
            .websocket_urls
            .difference(&old_config.connect.websocket_urls);
        for url in ws_connects_to_start {
            self.start_ws_connect(url.clone());
        }

        if config.peer_to_peer.bluetooth_le != old_config.peer_to_peer.bluetooth_le {
            self.stop_bluetooth();
            if config.peer_to_peer.bluetooth_le.enabled {
                self.start_bluetooth();
            }
        }
    }

    /// Start a TCP Server which can listen for connections from other Peers
    /// Generally also requires a update to the relevant DNS zone file to be
    /// discoverable by other peers
    fn start_tcp_listen(&self, config: &crate::transport::TcpListenConfig) {
        let bind_ip = format!("{}:{}", config.interface_ip, config.port);
        let c_addr = char_p::new(bind_ip);
        // Convert to a public Error type
        let _result =
            unsafe { ffi_sdk::ditto_start_tcp_server(&self.ditto, Some(c_addr.as_ref())) };
    }

    fn stop_tcp_listen(&self) {
        unsafe { ffi_sdk::ditto_stop_tcp_server(&self.ditto) };
    }

    /// Starts an HTTP server that other devices will be able to connect to.
    /// * `bind_ip` - IP Address:Port on which to listen for connections
    /// * `enable_websocket` - If true, enable websocket transport
    /// * `static_path` - Optional root of web content directory
    /// * `tls_cert_path` - Optional x509 certificate for web server (also
    ///   requires key)
    /// * `tls_key_path` - Optional TLS private key, required if cert is
    ///   provided
    fn start_http_listen(
        &self,
        config: &crate::transport::HttpListenConfig,
    ) -> Result<(), DittoError> {
        let enable_ws = if config.websocket_sync {
            ffi_sdk::WebSocketMode::Enabled
        } else {
            ffi_sdk::WebSocketMode::Disabled
        };

        let bind_ip = format!("{}:{}", config.interface_ip, config.port);
        let c_addr = char_p::new(bind_ip);
        let c_static_path = config
            .static_content_path
            .as_ref()
            .map(|x| char_p::new(x.to_string_lossy().to_string()));
        let c_tls_cert_path = config
            .tls_certificate_path
            .as_ref()
            .map(|x| char_p::new(x.to_string_lossy().to_string()));
        let c_tls_key_path = config
            .tls_key_path
            .as_ref()
            .map(|x| char_p::new(x.to_string_lossy().to_string()));

        let status = unsafe {
            ffi_sdk::ditto_start_http_server(
                &self.ditto,
                Some(c_addr.as_ref()),
                c_static_path.as_ref().map(|x| x.as_ref()),
                enable_ws,
                c_tls_cert_path.as_ref().map(|x| x.as_ref()), // TLS cert path
                c_tls_key_path.as_ref().map(|x| x.as_ref()),  // TLS key path
            )
        };
        if status != 0 {
            Err(DittoError::from_ffi(ErrorKind::InvalidInput))
        } else {
            Ok(())
        }
    }

    fn stop_http_listen(&self) {
        unsafe { ffi_sdk::ditto_stop_http_server(&self.ditto) };
    }

    fn start_tcp_connect(&mut self, address: String) {
        let addr = char_p::new(address.clone());
        // this handle stores a tx entangled with an rx across the FFI boundary which
        // will drop if all tx elements drop
        let tcp_client_handle =
            unsafe { ffi_sdk::ditto_add_static_tcp_client(&self.ditto, addr.as_ref()) };
        ::log::info!("Static TCP client transport {:?} started", &address);
        self.tcp_clients
            .insert(address, Box::new(tcp_client_handle));
    }

    fn stop_tcp_connect(&mut self, address: &str) {
        let _ = self.tcp_clients.remove(address);
    }

    fn start_ws_connect(&mut self, url: String) {
        let c_url = char_p::new(url.clone());
        let ws_client_handle =
            unsafe { ffi_sdk::ditto_add_websocket_client(&self.ditto, c_url.as_ref()) };
        ::log::info!("Websocket client transport {:?} started", &url);
        self.ws_clients.insert(url, Box::new(ws_client_handle));
    }

    fn stop_ws_connect(&mut self, url: &str) {
        let _ = self.ws_clients.remove(url);
    }

    fn start_bluetooth(&mut self) {
        // BlueZ cfg guard
        #[cfg(all(target_os = "linux", not(target_env = "musl")))]
        {
            let ble_client_handle =
                unsafe { ffi_sdk::ditto_add_internal_ble_client_transport(&self.ditto) };
            ::log::info!("BLE client transport started");
            self.ble_client_transport = Some(Box::new(ble_client_handle));
            let ble_server_handle =
                unsafe { ffi_sdk::ditto_add_internal_ble_server_transport(&self.ditto) };
            ::log::info!("BLE server transport started");
            self.ble_server_transport = Some(Box::new(ble_server_handle));
        }
        ::log::info!("handling BLE transport")
    }

    fn stop_bluetooth(&mut self) {
        let _to_drop = self.ble_client_transport.take();
        let _to_drop = self.ble_server_transport.take();
    }
}

impl DittoKit {
    pub fn with_sdk_version<R>(ret: impl FnOnce(&'_ str) -> R) -> R {
        ret(unsafe { ffi_sdk::ditto_get_sdk_version().to_str() })
    }
}

impl DittoKit {
    pub fn set_logging_enabled(enabled: bool) {
        unsafe { ffi_sdk::ditto_logger_enabled(enabled) }
    }

    pub fn get_logging_enabled() -> bool {
        unsafe { ffi_sdk::ditto_logger_enabled_get() }
    }

    pub fn get_emoji_log_level_headings_enabled() -> bool {
        unsafe { ffi_sdk::ditto_logger_emoji_headings_enabled_get() }
    }

    pub fn set_emoji_log_level_headings_enabled(enabled: bool) {
        unsafe {
            ffi_sdk::ditto_logger_emoji_headings_enabled(enabled);
        }
    }

    pub fn get_minimum_log_level() -> ffi_sdk::CLogLevel {
        unsafe { ffi_sdk::ditto_logger_minimum_log_level_get() }
    }

    pub fn set_minimum_log_level(log_level: ffi_sdk::CLogLevel) {
        unsafe {
            ffi_sdk::ditto_logger_minimum_log_level(log_level);
        }
    }
}

impl DittoKit {
    /// Activate a Ditto instance by setting a license token. You cannot
    /// sync with Ditto before you have activated it.
    pub fn set_license_token(&mut self, license_token: &str) -> Result<(), DittoError> {
        use ffi_sdk::LicenseVerificationResult;
        let c_license: char_p::Box = char_p::new(license_token);

        let mut err_msg = None;
        let out_err_msg = err_msg.manually_drop_mut().as_out();
        let res = unsafe { ffi_sdk::verify_license(c_license.as_ref(), Some(out_err_msg)) };

        if res == LicenseVerificationResult::LicenseOk {
            self.activated = true;
            return Ok(());
        }

        self.activated = false;
        let err_msg = err_msg.unwrap();

        ::log::error!("{}", err_msg);

        match res {
            LicenseVerificationResult::LicenseExpired => {
                Err(DittoError::license(LicenseError::LicenseTokenExpired {
                    message: err_msg.as_ref().to_string(),
                }))
            }
            LicenseVerificationResult::VerificationFailed => Err(DittoError::license(
                LicenseError::LicenseTokenVerificationFailed {
                    message: err_msg.as_ref().to_string(),
                },
            )),
            LicenseVerificationResult::UnsupportedFutureVersion => Err(DittoError::license(
                LicenseError::LicenseTokenUnsupportedFutureVersion {
                    message: err_msg.as_ref().to_string(),
                },
            )),
            _ => panic!("Unexpected license verification result {:?}", res),
        }
    }

    /// Activate a Ditto instance by setting a license token. You cannot
    /// sync with Ditto before you have activated it.
    pub fn set_access_license(&mut self, license_str: &str) {
        self.set_license_token(license_str).unwrap()
    }

    /// Returns a reference to the underlying local data store
    pub fn store(&self) -> &Store {
        &self.store
    }

    /// Returns the site ID that the instance of DittoKit is using as part of
    /// its identity.
    pub fn site_id(&self) -> u64 {
        self.site_id
    }

    /// Returns a custom identifier for the current device.
    ///
    /// When using observePeers(), each remote peer is represented by a short
    /// UTF-8 “device name”. By default this will be a truncated version of
    /// the device’s hostname. It does not need to be unique among peers.
    /// Configure the device name before calling start(). If it is too long
    /// it will be truncated.
    pub fn device_name(&self) -> &str {
        todo!();
    }

    /// Set a custom identifier for the current device
    pub fn set_device_name(&mut self, _name: String) {
        todo!();
    }

    /// Request bulk status information about the transports. This is mostly
    /// intended for statistical or debugging purposes.
    pub fn transport_diagnostics(&self) -> TransportDiagnostics {
        todo!();
    }

    /// Request information about Ditto peers in range of this device.
    ///
    /// This method returns an observer which should be held as long as updates
    /// are required. A newly registered observer will have a peers update
    /// delivered to it immediately. Then it will be invoked repeatedly when
    /// Ditto devices come and go, or the active connections to them change.
    pub fn observe_peers<H>(&self, _handler: H) -> PeerObserver
    where
        H: Fn(&[RemotePeer]),
    {
        todo!();
    }
}

impl DittoKit {
    /// Removes all sync metadata for any remote peers which aren't currently
    /// connected. This method shouldn't usually be called. Manually running
    /// garbage collection often will result in slower sync times. Ditto
    /// automatically runs a garbage a collection process in the background
    /// at optimal times.
    ///
    /// Manually running garbage collection is typically only useful during
    /// testing if large amounts of data are being generated. Alternatively,
    /// if an entire data set is to be evicted and it's clear that
    /// maintaining this metadata isn't necessary, then garbage collection
    /// could be run after evicting the old data.
    pub fn run_garbage_collection(&self) {
        unsafe {
            ffi_sdk::ditto_run_garbage_collection(&self.ditto);
        }
    }
}

pub struct TransportDiagnostics;

pub struct RemotePeer;

pub struct PeerObserver;

pub type SiteId = u64;
