use std::{
    fs::File,
    io::Read,
    path::{Path, PathBuf},
};

pub use ::safer_ffi::prelude::*;
pub use ::tap::*;
use chrono::{DateTime, TimeZone, Utc};
pub use dittolive_ditto::prelude::*;
use serde::{Deserialize, Serialize};

#[allow(dead_code)]
pub fn test_license_from_file<P: AsRef<Path>>(path: P) -> std::io::Result<String> {
    let mut f = File::open(path)?;
    let mut key = String::new();
    let _bytes = f.read_to_string(&mut key)?;
    let result = key.trim().to_string();
    if result.is_empty() {
        Err(std::io::Error::new(
            std::io::ErrorKind::InvalidData,
            "License Key is empty",
        ))
    } else {
        Ok(result)
    }
}

pub fn temp_persistence_dir() -> ::tempdir::TempDir {
    let tmpdir = &std::env::temp_dir().tap_mut(|it| it.push("dittokit"));
    ::std::fs::create_dir_all(tmpdir)
        .unwrap_or_else(|err| panic!("Failed to create `{}`: {}", tmpdir.display(), err,));
    ::tempdir::TempDir::new_in(tmpdir, "test").expect("Failed to create temporary directory")
}

#[allow(dead_code)]
pub fn with_temp_persistence_dir<R>(ret: impl FnOnce(&'_ Path) -> R) -> R {
    ret(temp_persistence_dir().path())
}

pub fn get_ditto_kit(directory: impl Into<PathBuf>) -> DittoKit {
    use ::rand::Rng;
    use ::uuid::Uuid;

    let app_name = format!("test.ditto.{}", Uuid::new_v4());
    let site_id: SiteId = ::rand::thread_rng().gen_range(2..SiteId::MAX);
    let working_dir = directory.into();
    let identity = Identity::new_development(&app_name, site_id, Some(&working_dir)).unwrap();
    let mut ditto = DittoKit::new(identity, Some(working_dir));
    DittoKit::set_minimum_log_level(CLogLevel::Info);

    let license_str = std::env::var("DITTO_LICENSE").expect("No License Env Var provided");
    ditto.set_access_license(license_str.as_str());
    ditto
}

// A Re-usable test document type
#[derive(Serialize, Deserialize, Debug, PartialEq, Eq, Clone)]
pub enum TestColor {
    Red,
    Blue,
    Crimson,
    Orange,
    Yellow,
    Pink,
    White,
    Black,
    Brown,
}

#[derive(Serialize, Deserialize, Debug, PartialEq, Eq, Clone)]
pub struct TestType {
    pub make: String,
    pub color: TestColor,
    pub age: u8,
    pub mileage: i64,
    pub purchased_on: DateTime<Utc>,
    pub tags: Vec<String>,
    pub data: Vec<u8>, // most common repr for bin data in std lib
}

impl TestType {
    #[allow(dead_code)] // This actually is used
    pub fn random() -> TestType {
        use ::rand::rngs::StdRng;
        use rand::{Fill, SeedableRng};

        let mut bytes: Vec<u8> = Vec::with_capacity(2048);
        let mut rng = StdRng::seed_from_u64(123456789);
        bytes.try_fill(&mut rng).unwrap();

        TestType {
            make: String::from("Honda"),
            color: TestColor::Red,
            age: 12,
            mileage: 52000,
            purchased_on: Utc::now(),
            tags: vec![
                String::from("a"),
                String::from("-1"),
                String::from("test_tag"),
            ],
            data: bytes,
        }
    }
}

impl Default for TestType {
    fn default() -> TestType {
        TestType {
            make: String::default(),
            color: TestColor::Red,
            age: 0,
            mileage: 0,
            purchased_on: Utc.ymd(1970, 1, 1).and_hms(0, 0, 0),
            tags: Vec::new(),
            data: Vec::new(),
        }
    }
}
