use_prelude!();

use std::sync::Arc;

/// A Handle for a Ditto Subscription to documents in a collection
/// matching a Query
/// When the Subscription is dropped, it will automatically trigger a clean up
/// of itself.
pub struct Subscription {
    pub(super) ditto: Arc<ffi_sdk::BoxedDitto>,
    pub(super) collection_name: char_p::Box,
    pub(super) query: char_p::Box,
}

impl Subscription {
    pub fn new(ditto: Arc<ffi_sdk::BoxedDitto>, collection_name: &str, query: &str) -> Self {
        let collection_name = char_p::new(collection_name);
        let query = char_p::new(query);
        Subscription {
            ditto,
            collection_name,
            query,
        }
    }
}

impl Drop for Subscription {
    fn drop(&mut self) {
        ::log::debug!(
            "Dropping subscription for collection {} query {}",
            &self.collection_name,
            &self.query
        );
        unsafe {
            ffi_sdk::ditto_remove_subscription(
                &self.ditto,
                self.collection_name.as_ref(),
                self.query.as_ref(),
            );
        }
    }
}
