use std::mem::MaybeUninit;

use ffi_sdk::BoxedAuthClient;

use crate::error::{DittoError, ErrorKind};
use_prelude!();

pub struct Identity {
    pub(crate) auth_client: BoxedAuthClient,
}

impl Identity {
    pub fn new_development(
        app_id: &str,
        site_id: u64,
        persistence_dir: Option<&Path>,
    ) -> Result<Self, DittoError> {
        let working_dir = DittoKit::default_persistence_dir(persistence_dir);
        let working_dir = char_p::new(working_dir);
        let app_id = char_p::new(app_id);
        let mut auth_client = MaybeUninit::new(None);
        match unsafe {
            ffi_sdk::ditto_auth_client_make_for_development(
                Some(working_dir.as_ref()),
                app_id.as_ref(),
                site_id,
                (&mut auth_client).into(),
            )
        } {
            0 => {
                let auth_client = unsafe { auth_client.assume_init() };
                Ok(Self {
                    auth_client: auth_client.unwrap(),
                })
            }
            _ => Err(DittoError::from(ErrorKind::InvalidInput)),
        }
    }

    pub fn new_shared_key(
        app_id: &str,
        site_id: u64,
        key_der_b64: &str,
        persistence_dir: Option<&Path>,
    ) -> Result<Self, DittoError> {
        let working_dir = DittoKit::default_persistence_dir(persistence_dir);
        let working_dir = char_p::new(working_dir);
        let app_id = char_p::new(app_id);
        let key_der = char_p::new(key_der_b64);
        let mut auth_client = MaybeUninit::new(None);
        match unsafe {
            ffi_sdk::ditto_auth_client_make_with_shared_key(
                Some(working_dir.as_ref()),
                app_id.as_ref(),
                key_der.as_ref(),
                site_id,
                (&mut auth_client).into(),
            )
        } {
            0 => {
                let auth_client = unsafe { auth_client.assume_init() };
                Ok(Self {
                    auth_client: auth_client.unwrap(),
                })
            }
            _ => Err(DittoError::from(ErrorKind::InvalidInput)),
        }
    }

    pub fn new_production(certificate_config_b64: &str) -> Result<Self, DittoError> {
        let mut auth_client = MaybeUninit::new(None);
        let certificate_config = char_p::new(certificate_config_b64);
        match unsafe {
            ffi_sdk::ditto_auth_client_make_with_static_x509(
                certificate_config.as_ref(),
                (&mut auth_client).into(),
            )
        } {
            0 => {
                let auth_client = unsafe { auth_client.assume_init() };
                Ok(Self {
                    auth_client: auth_client.unwrap(),
                })
            }
            _ => Err(DittoError::from(ErrorKind::InvalidInput)),
        }
    }
}
