//! Ditto Error Types
//! Modeled after std::io::error::Error;

use std::{error::Error as ErrorTrait, fmt};

use crate::ffi_sdk;

pub struct DittoError {
    repr: Repr,
}

pub enum Repr {
    Simple(ErrorKind),
    Ffi(FfiError),
    Rust(RustError),
}

#[derive(Clone, Copy, Debug, Eq, Hash, Ord, PartialEq, PartialOrd)]
pub enum ErrorKind {
    Duplicate,
    Internal,
    InvalidInput,
    NonExtant,
    /// Error originated in the FFI layer
    Ffi, // use this when you can't re-map into a Rust Error
}

/// Errors Originating in C code
/// Shadows the FFI's CError type which doesn't have a C Repr
pub struct FfiError {
    pub code: i32,
    pub msg: String,
}

/// Errors originating in Rust Code
#[derive(Debug)]
pub struct RustError {
    pub kind: ErrorKind,
    pub error: Box<dyn ErrorTrait + Send + Sync>,
}

impl ErrorKind {
    pub fn as_str(&self) -> &'static str {
        match *self {
            ErrorKind::Duplicate => "This ID already exists",
            ErrorKind::Internal => "DittoSyncKit encountered an internal error",
            ErrorKind::InvalidInput => "Invalid client input provided",
            ErrorKind::NonExtant => "The target entity can not be found",
            ErrorKind::Ffi => "Unmapped DittoSyncKit Error",
        }
    }
}

impl From<ErrorKind> for DittoError {
    fn from(kind: ErrorKind) -> DittoError {
        DittoError {
            repr: Repr::Simple(kind),
        }
    }
}

impl fmt::Debug for DittoError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        fmt::Debug::fmt(&self.repr, f)
    }
}

impl ErrorTrait for DittoError {}

impl DittoError {
    pub fn new<E>(kind: ErrorKind, rust_err: E) -> DittoError
    where
        E: Into<Box<dyn ErrorTrait + Send + Sync>>,
    {
        DittoError {
            repr: Repr::Rust(RustError {
                kind,
                error: rust_err.into(),
            }),
        }
    }

    /// Manually specify the Error Kind, but fetch the message from the FFI
    /// The result is returned as a RUST style error
    /// with the cause of <String as ErrorTrait>
    pub fn from_ffi(kind: ErrorKind) -> DittoError {
        match unsafe { ffi_sdk::ditto_error_message() } {
            Some(c_msg) => {
                let msg = c_msg.into_string();
                DittoError::new(kind, msg)
            }
            None => DittoError {
                repr: Repr::Ffi(FfiError {
                    msg: "No Message".to_owned(),
                    code: -1, // undefined
                }),
            },
        }
    }

    pub fn kind(&self) -> ErrorKind {
        match &self.repr {
            Repr::Simple(kind) => *kind,
            Repr::Rust(e) => e.kind,
            Repr::Ffi(_c) => ErrorKind::Ffi, /* could implement uniform code to kind mapping in
                                              * future */
        }
    }
}

impl fmt::Debug for Repr {
    fn fmt(&self, fmt: &mut fmt::Formatter<'_>) -> fmt::Result {
        match *self {
            Repr::Simple(kind) => fmt.debug_tuple("Kind").field(&kind).finish(),
            Repr::Rust(ref e) => fmt::Debug::fmt(&e, fmt),
            Repr::Ffi(ref c) => fmt
                .debug_struct("FFIError")
                .field("code", &c.code)
                .field("message", &c.msg)
                .field("kind", &ErrorKind::Ffi)
                .finish(),
        }
    }
}

impl fmt::Display for DittoError {
    fn fmt(&self, fmt: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self.repr {
            Repr::Simple(kind) => write!(fmt, "{}", kind.as_str()),
            Repr::Rust(ref e) => e.error.fmt(fmt), // forward to cause error
            Repr::Ffi(ref c) => write!(fmt, "{} (code {})", &c.msg, &c.code),
        }
    }
}

impl From<i32> for FfiError {
    fn from(code: i32) -> FfiError {
        match unsafe { ffi_sdk::ditto_error_message() } {
            Some(c_msg) => {
                let msg = c_msg.into_string();
                FfiError { code, msg }
            }
            None => FfiError {
                msg: "No Message".to_owned(),
                code,
            },
        }
    }
}

impl From<i32> for DittoError {
    fn from(code: i32) -> DittoError {
        let ffi_err = FfiError::from(code);
        DittoError {
            repr: Repr::Ffi(ffi_err),
        }
    }
}
