//! Example Ditto Cars App "Listener"
//!
//! Listens for incoming connections from the Ditto Car's App
//! (live.ditto.carsapp) via TCP/IP.
//!
//! Example Invocation
//! ``` shell
//! RUST_LOG=debug  DITTO_SITE_ID=2 \
//! DITTO_DB_PATH=./receiver cargo run -p dittolive-ditto --example carsapp
//! ```
//!
//! Note you may need to clear the local database files (`rm -rf receiver` in
//! the example above) between runs in order to keep the replication state
//! intact

extern crate dittolive_ditto;
extern crate log;
extern crate safer_ffi;
extern crate serde;
extern crate serde_cbor;
extern crate tempdir;

use std::{env, path::PathBuf};

use dittolive_ditto::prelude::*;
use rand::Rng;
fn main() {
    env_logger::init();
    // Fetch the app Name
    let app_name = String::from("live.ditto.carsapp");
    // define actor/site ID
    let site_id: SiteId = env::var("DITTO_SITE_ID")
        .map(|x| x.parse::<SiteId>().unwrap())
        .unwrap_or_else(|_x| {
            // recall the site ID's 0 and 1 have special meaning
            ::rand::thread_rng().gen_range(2..SiteId::MAX)
        });
    let license_token: String = env::var("DITTO_LICENSE").unwrap();
    let data_dir = env::var_os("DITTO_DB_PATH")
        .map(PathBuf::from)
        .or_else(|| -> Option<PathBuf> { Some(std::env::temp_dir()) })
        .map(|mut path: PathBuf| {
            path.push("dittokit");
            let _ = std::fs::create_dir_all(&path);
            path
        });
    let bind_ip: String =
        env::var("DITTO_BINDIP").unwrap_or_else(|_x| String::from("0.0.0.0:4040"));
    let parts: Vec<&str> = bind_ip.split(':').collect();
    let ip = parts[0].to_string();
    let port: u16 = parts[1].parse().unwrap();

    let identity = Identity::new_development(&app_name, site_id, data_dir.as_deref()).unwrap();

    let mut ditto = DittoKit::new(identity, data_dir);
    ::log::debug!("Setting Ditto License.");
    ditto.set_access_license(&license_token);
    // ditto.start_tcp_server("127.0.0.1:4040");
    let mut transport_config = TransportConfig::new();
    transport_config.listen.tcp.enabled = false;
    transport_config.listen.tcp.interface_ip = ip;
    transport_config.listen.tcp.port = port;
    transport_config.peer_to_peer.bluetooth_le.enabled = true;
    ditto.set_transport_config(transport_config);
    ditto.start_sync();
    ::log::debug!("Ditto Cars App started!");

    let store = ditto.store();
    let collection = store.collection("test");

    let (tx, rx) = std::sync::mpsc::sync_channel(120);
    // This handler is called every time docs from local or remote sources are
    // committed to the local store which match the associated query.
    // `documents` is a vec of ALL documents matching the query after application of
    // the transaction.
    // `event` can be used to dissect out which of these are insertions.
    let event_handler = move |documents: Vec<ffi_sdk::BoxedDocument>, event| {
        ::log::trace!(
            "Latency Receiver got {:?} with {} updated documents",
            &event,
            documents.len()
        );
        match event {
            LiveQueryEvent::Initial => {
                // On an initial sync, we can calculate the latency for arrival of the first
                // document
                ::log::info!("Initial Data loaded");
            }
            LiveQueryEvent::Update { insertions, .. } => {
                // We only want to send the newest event
                for idx in insertions.iter() {
                    if let Some(car) = documents.get(*idx) {
                        let car_cbor = car.to_cbor().unwrap();
                        let _ = tx.send(car_cbor);
                    }
                }
            }
        }
    };
    // downgrade our logging output before running the query
    DittoKit::set_minimum_log_level(CLogLevel::Debug);
    // Find and report on all cars
    let _lq = collection.unwrap().find_all().observe(event_handler);
    for car_cbor in rx.iter() {
        let car_str = serde_json::to_string(&car_cbor).unwrap();
        println!("{}", &car_str);
    }
}
